//
// TaskShelfTableModel.java
//
//	A class that controls the JTable that is the TaskShelf
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui;

import com.sgi.sysadm.category.*;
import com.sgi.sysadm.ui.richText.*;
import com.sgi.sysadm.ui.event.*;
import com.sgi.sysadm.util.*;
import javax.swing.table.*;
import javax.swing.*;
import java.awt.*;
import java.awt.event.*;
import java.net.*;


/**
 * This class is used by a JTable to show a TaskShelf.
 */
/*package*/  class TaskShelfTableModel extends AbstractTableModel {
    

    /** 
     * A resource <i>TaskShelfTableModel.columns</i> that specifies
     * how many columns of tasks to display in the task shelf.
     */
    private static final String COLUMNS = "TaskShelfTableModel.columns";
    
    private static final String RICH_TEXT_NAME = "TaskShelfText";
    
    private TaskLoader[] _taskLoaderList = new TaskLoader[0];
    private RichTextComponent[] _labels = new RichTextComponent[0];
    private JLabel[] _icons = new JLabel[0];
    private Mouser[] _mouseListeners = new Mouser[0];
    private ResourceStack _rs;
    private int _columns;  // The number of columns of tasks to display

    /** 
     * Constructor
     * 
     * @param rs The ResourceStack that the class can use to look up
     *           icons.
     */
    public TaskShelfTableModel(ResourceStack rs) {
	_rs = rs;
	_columns = rs.getInt(COLUMNS);
    }
    
    /** 
     * Sets the TaskList to display a list of tasks.
     *
     * @param taskLoaderList The list of tasks to display
     *
     * @param tlrl A TaskLaunchRequestListener that will be notified
     *             if a user clicks on a task.
     */
    public void setTaskList(TaskLoader[] taskLoaderList,
			    final TaskLaunchRequestListener tlrl) {
	_taskLoaderList = taskLoaderList;
	for (int ii = 0; ii < _labels.length; ii++) {
	    _icons[ii].removeMouseListener(_mouseListeners[ii]);
	}
	_labels = new RichTextComponent[taskLoaderList.length];
	_icons =  new JLabel[taskLoaderList.length];
	_mouseListeners = new Mouser[taskLoaderList.length];
	for (int ii = 0; ii < _labels.length; ii++) {
	    ResourceStack rs = _taskLoaderList[ii].getResourceStack();
	    String rt = "<A HREF="  + ii + ">" + 
		 _taskLoaderList[ii].getLongTaskName() + "</A>";
	    RichTextComponent label = 
		new RichTextComponent(RICH_TEXT_NAME, _rs, rt);
	    label.setMargins(new Insets(0,0,0,0));
	    label.setOpaque(true);
	    label.addLinkListener(new LinkListener() {
		public void linkActivated(LinkEvent event) {
		    tlrl.taskLaunchRequested(new TaskLaunchRequestEvent(
			TaskShelfTableModel.this,
			_taskLoaderList[Integer.parseInt(event.getTarget())]));
		}
	    });
	    _labels[ii] = label;

	    _icons[ii] = new JLabel(rs.getIcon(Task.TASK_SHELF_ICON));
	    _icons[ii].addMouseListener(
		_mouseListeners[ii] = new Mouser(ii, tlrl));
	}
	fireTableDataChanged();
    }
    
    private class Mouser extends MouseAdapter {
	int _index;
	int _currentLinkState;
	
	TaskLaunchRequestListener _tlrl;
	public Mouser(int index, TaskLaunchRequestListener tlrl) {
	    _index = index;
	    _tlrl = tlrl;
	}
	public void mouseClicked(MouseEvent e) {
	    _labels[_index].setLinkState(
		0, RichTextComponent.LINK_STATE_VISITED);
	    _currentLinkState = RichTextComponent.LINK_STATE_VISITED;
	    fireTableCellUpdated(_index / _columns,
				 (_index % _columns)*2 + 1);
	    _tlrl.taskLaunchRequested(new TaskLaunchRequestEvent(
		TaskShelfTableModel.this,_taskLoaderList[_index]));
	}
	
	public void mousePressed(MouseEvent e) {
	    _labels[_index].setLinkState(
		0, RichTextComponent.LINK_STATE_ACTIVE);
	}
	public void mouseEntered(MouseEvent e) {
	    _currentLinkState = _labels[_index].getLinkState(0);
	}
	

	public void mouseExited(MouseEvent e) {
	    _labels[_index].setLinkState(
		0, _currentLinkState);
	    fireTableCellUpdated(_index / _columns,
				 (_index % _columns)*2 + 1);
	}
    }
    
    /**
     * A work around for the fact that inner classes can't call
     * protected methods of the outter classes' base class
     */
    public void fireTableCellUpdated(int row, int column) {
	super.fireTableCellUpdated(row, column);
    }

    /** 
     * Return the current number of rows that are being displayed
     * @return Current number of rows that are being displayed
     */
    public int getRowCount() {
	if (_taskLoaderList.length == 0) {
	    return 0;
	}
	return (_taskLoaderList.length -1 ) / _columns + 1;
    }

    /** 
     * Returns the current number of columns being displayed 
     * @return Current number of columns being displayed 
     */
    public int getColumnCount() {
	return _columns*2; // There is an icon and a label for each task
    }
    
    /** 
     * Returns the Object at particular row and column
     * @param row The row
     * @param column the column
     * @return The Object at particular row and column
     */
    public Object getValueAt(int row, int column) {
	TaskLoader tl  = getTaskAt(row, column);
	if (tl == null) {
	    // There's no task where they clicked.
	    return null;
	}
	if (column % 2 == 0) { // if it's an even column (ie. Task Icon)
	    return _icons[row];
	}
	else { // it's an odd column (ie. Task Name)
	    return _labels[row];
	}
    }

    /** 
     * Gives the Table information on how to display a column
     * @param column The column in question
     */
    public Class getColumnClass(int column) {
	if (column % 2 == 0) { // if it's an even column
	    return JLabel.class;
	} else {  // it's an odd column
	    return RichTextComponent.class;
	}
    }
    
    /** 
     * Returns which task is at a particular row.
     * @param row The row of interest
     * @return The TaskLoader that represents the Task in a cell.
     */
    public TaskLoader getTaskForCell(int row, int column) {
	return getTaskAt(row,column);
    }

    /**
     * Computes the offset in the _taskLoaderList of task given the
     * row and column.
     * @param row
     * @param column
     * @return The offset into the _taskLoaderList
     */
    private TaskLoader getTaskAt(int row, int column) {
	try{
	    return _taskLoaderList[row * _columns + (column / 2)];
	}
	catch (ArrayIndexOutOfBoundsException e) {
	    return null;
	}
    }  
}
