//
// TwoColumnPanel.java
//
//      A Panel for laying out labels and components in two columns.
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui;

import javax.swing.*;
import java.util.*;
import java.awt.*;
import com.sgi.sysadm.util.*;

/**
 * A Panel for laying out labels and components in two columns, the
 * labels in the left column, and the components in the right column.
 * The layout for this panel is identical to the layout one would get
 * by using a LabelComponentPanel and laying out all the components in
 * the horizontal direction.
 *
 * @see LabelComponentPanel
 */

public class TwoColumnPanel extends RPanel {

    /**
     * A resource <i>&lt;name&gt;.splitSpacing</i> is an
     * integer that gives the width (in points) of the space to be put
     * between the labels and their values.  &lt;name&gt; is the name
     * passed to the constructor.  If name is not set, or if
     * <i>&lt;name&gt;.splitSpacing</i> is not found,
     * <i>TwoColumnPanel.splitSpacing</i> will be used.
     */
    public static final String SPLIT_SPACING =
	".splitSpacing";

    /**
     * A resource <i>&lt;name&gt;.verticalSpacing</i> is an
     * integer that gives the vertical space (in points) to put
     * between the label-value pairs.  &lt;name&gt; is the name
     * passed to the constructor.  If name is not set, or if
     * <i>&lt;name&gt;.verticalSpacing</i> is not found,
     * <i>TwoColumnPanel.verticalSpacing</i> will be used.
     */
    public static final String VERTICAL_SPACING =
	".verticalSpacing";

    private static final String CLASS_NAME = "TwoColumnPanel";

    private LabelComponentConstraints _c =
	new  LabelComponentConstraints();
    private LabelComponentLayoutManager _manager =
	new LabelComponentLayoutManager();

    private int getPixels(ResourceStack rs, String name, String postfix) {
	String[] lookup;
	if (name != null) {
	    lookup = new String[] {name + postfix, CLASS_NAME + postfix};
	} else {
	    lookup = new String[] {CLASS_NAME + postfix};
	}
	return rs.getPixels(lookup);
    }

    /**
     * A constructor that takes a ResourceStack.
     * Since no name is provided, all resource lookups will be
     * prepended with "TwoColumnPanel"
     *
     * @param rs The ResourceStack to use for lookups
     */
    public TwoColumnPanel(ResourceStack rs) {
	this(rs,null);
    }

    /**
     * A constructor that takes a ResourceStack and a name.
     * Since a name is provided, all resource lookups will be
     * prepended with the name that is passed in.  If that lookup
     * fails, the class will try again, this time prepending
     * "TwoColumnPanel".
     *
     * @param rs The ResourceStack to use for lookups
     * @param name The name of this TwoColumnPanel
     */
    public TwoColumnPanel(ResourceStack rs, String name) {

	_c.hgap =  getPixels(rs, name, SPLIT_SPACING);
	_c.bottomGap =  getPixels(rs, name, VERTICAL_SPACING);
	_c.layoutType = LabelComponentConstraints.HORIZONTAL;
	_c.rightAttachment = LabelComponentConstraints.DEFAULT;
	setLayout(_manager);
    }

    /**
     * Adds a component without a label to the panel
     * @param component The component to add
     */
    public void addComponent(Component component) {
	addComponent(component,(Component)null);
    }

    /**
     * Adds a component that spans both the left and right columns.
     * No label will be shown for the component.
     * @param component The component to add
     */
    public void addSpanningComponent(Component component) {
	_c.layoutType = LabelComponentConstraints.VERTICAL;
	add(component, _c);
	_c.layoutType = LabelComponentConstraints.HORIZONTAL;
    }

    /**
     * Adds a component and a label to the panel
     *
     * @param component The component to add
     * @param label The label that describes this component.  A JLabel
     *		    will be created with the label, and that JLabel
     *		    will be added to the panel as the label for
     *		    <tt>component</tt>
     */
    public void addComponent(Component component, String label) {
	addComponent(component,new JLabel(label));
    }

    /**
     * Adds a component and a label to the panel.
     *
     * @param component The component to add
     * @param label The label that describes this component
     */
    public void addComponent(Component component, Component label) {
	if (label != null) {
	    add(label,LabelComponentConstraints.LABEL);
	    _c.label = label;
	    add(component, _c);
	    _c.label = null;
	}
	else {
	    add(component, _c);
	}
    }

    /**
     * Removes a component and any label associated with it.
     *
     * @param component The component to remove
     */
    public void removeComponent(Component component) {
	Component label = _manager.getConstraints(component).label;
	remove(component);
	if (label != null) {
	    remove(label);
	}
    }
}
