//
// UIContext.java
//
//	Provides dialog posting services and busy handling.
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//
package com.sgi.sysadm.ui;

import com.sgi.sysadm.util.*;
import com.sgi.sysadm.ui.richText.*;
import javax.swing.*;
import javax.swing.event.*;
import java.awt.*;
import java.awt.event.*;
import java.awt.peer.*;
import java.lang.reflect.*;
import java.text.MessageFormat;
import java.util.*;

/**
 * UIContext provides dialog posting and busy handling services.
 * <P>
 * Call UIContext.setDialogParent() to set the Component over which
 * posted dialogs should be centered and for which the busy cursor will
 * be set.  If the dialog parent has not been set or is set to null, 
 * posted dialogs will be centered on the screen and the busy() methods
 * will have no effect on the cursor.
 * <P>
 * The methods in this class are Swing thread safe.  In other words, if
 * they are called from a thread other than the event dispatch thread,
 * SwingUtilities.invokeLater() will be used to perform the operations on
 * the event dispatch thread.
 */
public class UIContext {
    private JComponent _dialogParent;

    /**
     * _rs is the ResourceStack for this UIContext and is available
     * directly to subclasses as a convenience.  (Well, it used to be, except
     * that javadoc couldn't handle the leading underbar.  Now subclasses
     * can use getResourceStack() like everyone else.)
     */
    private ResourceStack _rs;

    private String _dialogTitle;
    private Cursor _cursor;
    private int _busyLevel;
    private int _blockLevel;
    private int _dialogCount;
    private Window _topWindow;
    private boolean _globalMessage = false;
    private HostContext _hostContext;
    private boolean _inputBlocked = false;
    private Component _glassPane;
    private boolean _triedGlassPane = false;
    private Frame _frameParent;
    private UIContextDialog _busyDialog;
    private JButton _cancelBusyButton;
    private ActionListener _cancelListener;
    private UIContextDialog _messageDialog;
    private JButton _confirmMessageButton;
    private ActionListener _confirmListener;
    private UIContextDialog _questionDialog;
    private ActionListener _yesListener;
    private ActionListener _noListener;
    private String _class = ResourceStack.getClassName(UIContext.class);
    private Component _saveFocus;
    private int _maxLineLen;

    // _pendingBusyLevel is for debugging.  The asserts that use
    // _busyLevel happen within a SwingUtilities.invokeLater(), which
    // makes their stack traces useless.  _pendingBusyLevel is used to
    // give useful stack traces when notBusy() is called too many times.
    private int _pendingBusyLevel;
    
    private void incPendingBusyLevel() {
	_pendingBusyLevel++;
    }
    private void decPendingBusyLevel() {
	Log.assert(_pendingBusyLevel > 0,
		   "Too many calls to UIContext.notBusy()");
	_pendingBusyLevel--;
    }
    

    /**
     * The property <I>UIContext.Dialog.okLabel</I> is a string that is
     * displayed on the confirmation button of dialogs (e.g. "OK").  It is used
     * for dialogs posted by UIContext.postError(), UIContext.postWarning(),
     * and UIContext.postInfo().
     * <P>
     * The default value for this property is defined in
     * com.sgi.sysadm.ui.SysadmUIP.properties and may be overridden in
     * any properties file on the ResourceStack.
     */
    public static final String OK_LABEL = "UIContext.Dialog.okLabel";
    
    /**
     * The property <I>UIContext.Dialog.topInset</I> is an integer that
     * defines the height, in <A HREF="glossary.html#Point">points</A>,
     * of the inset between the top of a dialog and the
     * contents of a dialog.
     * <P>
     * The default value for this property is defined in
     * com.sgi.sysadm.ui.SysadmUIP.properties and may be overridden in
     * any properties file on the ResourceStack.
     */
    public static final String TOP_INSET = "UIContext.Dialog.topInset";

    /**
     * The property <I>UIContext.Dialog.leftInset</I> is an integer that
     * defines the width, in <A HREF="glossary.html#Point">points</A>,
     * of the inset between the left edge of a dialog and the contents
     * of a dialog.
     * <P>
     * The default value for this property is defined in
     * com.sgi.sysadm.ui.SysadmUIP.properties and may be overridden in
     * any properties file on the ResourceStack.
     */
    public static final String LEFT_INSET = "UIContext.Dialog.leftInset";

    /**
     * The property <I>UIContext.Dialog.bottomInset</I> is an integer that
     * defines the height, in <A HREF="glossary.html#Point">points</A>,
     * of the inset between the bottom of a dialog and the contents of
     * a dialog.
     * <P>
     * The default value for this property is defined in
     * com.sgi.sysadm.ui.SysadmUIP.properties and may be overridden in
     * any properties file on the ResourceStack.
     */
    public static final String BOTTOM_INSET = "UIContext.Dialog.bottomInset";

    /**
     * The property <I>UIContext.Dialog.rightInset</I> is an integer that
     * defines the width, in <A HREF="glossary.html#Point">points</A>,
     * of the inset between the right edge of 
     * a dialog and the contents of a dialog.
     * <P>
     * The default value for this property is defined in
     * com.sgi.sysadm.ui.SysadmUIP.properties and may be overridden in
     * any properties file on the ResourceStack.
     */
    public static final String RIGHT_INSET = "UIContext.Dialog.rightInset";
    
    /**
     * The property <I>UIContext.Dialog.hgap</I> is an integer that
     * defines the width, in <A HREF="glossary.html#Point">points</A>,
     * of the space between the icon and 
     * message in a dialog.
     * <P>
     * The default value for this property is defined in
     * com.sgi.sysadm.ui.SysadmUIP.properties and may be overridden in
     * any properties file on the ResourceStack.
     */
    public static final String HGAP = "UIContext.Dialog.hgap";
    
    /**
     * The property <I>UIContext.Dialog.vgap</I> is an integer that
     * defines the height, in <A HREF="glossary.html#Point">points</A>,
     * of the space between the message and 
     * buttons in a dialog.
     * <P>
     * The default value for this property is defined in
     * com.sgi.sysadm.ui.SysadmUIP.properties and may be overridden in
     * any properties file on the ResourceStack.
     */
    public static final String VGAP = "UIContext.Dialog.vgap";
    
    /**
     * The property <I>UIContext.Dialog.busyIcon</I> is the
     * <A HREF="glossary.html#CLASSPATHRelative">CLASSPATH relative</A>
     * name of the <A HREF="glossary.html#IconImageFile">icon image
     * file</A> of the icon to use in the busy dialog.
     * <P>
     * The default value for this property is defined in
     * com.sgi.sysadm.ui.SysadmUIP.properties and may be overridden in
     * any properties file on the ResourceStack.
     */
    public static final String BUSY_ICON = "UIContext.Dialog.busyIcon";
    
    /**
     * The property <I>UIContext.Dialog.errorIcon</I> is the
     * <A HREF="glossary.html#CLASSPATHRelative">CLASSPATH
     * Relative</A> name of the 
     * <A HREF="glossary.html#IconImageFile">icon image file</A> of the
     * icon to use in the error dialog posted via UIContext.postError().
     * <P>
     * The default value for this property is defined in
     * com.sgi.sysadm.ui.SysadmUIP.properties and may be overridden in
     * any properties file on the ResourceStack.
     */
    public static final String ERROR_ICON = "UIContext.Dialog.errorIcon";
    
    /**
     * The property <I>UIContext.Dialog.warningIcon</I> is the     
     * <A HREF="glossary.html#CLASSPATHRelative">CLASSPATH
     * Relative</A> name of the 
     * <A HREF="glossary.html#IconImageFile">icon image file</A>
     * of the icon to use in the
     * warning dialog posted using UIContext.postWarning().
     * <P>
     * The default value for this property is defined in
     * com.sgi.sysadm.ui.SysadmUIP.properties and may be overridden in
     * any properties file on the ResourceStack.
     */
    public static final String WARNING_ICON = "UIContext.Dialog.warningIcon";
    
    /**
     * The property <I>UIContext.Dialog.infoIcon</I> is the
     * <A HREF="glossary.html#CLASSPATHRelative">CLASSPATH
     * Relative</A> name of the 
     * <A HREF="glossary.html#IconImageFile">icon image file</A>
     * of the icon to use in the
     * information dialog posted using UIContext.postInfo().
     * <P>
     * The default value for this property is defined in
     * com.sgi.sysadm.ui.SysadmUIP.properties and may be overridden in
     * any properties file on the ResourceStack.
     */
    public static final String INFO_ICON = "UIContext.Dialog.infoIcon";
    
    /**
     * The property <I>UIContext.Dialog.questionIcon</I> is the
     * <A HREF="glossary.html#CLASSPATHRelative">CLASSPATH
     * Relative</A> name of the 
     * <A HREF="glossary.html#IconImageFile">icon image file</A>
     * of the icon to use in the
     * question dialog posted using UIContext.postQuestion().
     * <P>
     * The default value for this property is defined in
     * com.sgi.sysadm.ui.SysadmUIP.properties and may be overridden in
     * any properties file on the ResourceStack.
     */
    public static final String QUESTION_ICON = "UIContext.Dialog.questionIcon";
    
    /**
     * The property <I>UIContext.Dialog.font</I> is the name of the font
     * to use for dialogs posted via UIContext.
     * <P>
     * The default value for this property is defined in
     * com.sgi.sysadm.ui.SysadmUIP.properties and may be overridden in
     * any properties file on the ResourceStack.
     */
    public static final String FONT = "UIContext.Dialog.font";
    
    /**
     * The property <I>UIContext.Dialog.cancelLabel</I> is a string
     * displayed on the cancel button of the busy dialog.
     * <P>
     * The default value for this property is defined in
     * com.sgi.sysadm.ui.SysadmUIP.properties and may be overridden in
     * any properties file on the ResourceStack.
     */
    public static final String CANCEL_LABEL = "UIContext.Dialog.cancelLabel";

    /**
     * The property <I>UIContext.Dialog.yesLabel</I> is a string
     * displayed on the yes button of the question dialog.
     * <P>
     * The default value for this property is defined in
     * com.sgi.sysadm.ui.SysadmUIP.properties and may be overridden in
     * any properties file on the ResourceStack.
     */
    public static final String YES_LABEL = "UIContext.Dialog.yesLabel";
    
    /**
     * The property <I>UIContext.Dialog.noLabel</I> is a string
     * displayed on the no button of the question dialog.
     * <P>
     * The default value for this property is defined in
     * com.sgi.sysadm.ui.SysadmUIP.properties and may be overridden in
     * any properties file on the ResourceStack.
     */
    public static final String NO_LABEL = "UIContext.Dialog.noLabel";

    /** 
     * The property <I>UIContext.Dialog.maxLineLen</I> is the length,
     * in characters, of the longest line in any error, warning, or
     * info dialog.  Any line which exceeds this length will be broken
     * into shorter lines by the addition of newline characters.
     * <P>
     * The default value for this property is defined in
     * com.sgi.sysadm.ui.SysadmUIP.properties and may be overridden in
     * any properties file on the ResourceStack.
     */
    // This is for bug 633403.  It could be smarter and be in points,
    // for a gentler rag right, but this is close enough.
    public static final String MAX_LINE_LEN = "UIContext.Dialog.maxLineLen";

    private static final String EMPTY_STRING = "";

    
    private class UIContextDialog extends RDialog {
	private Frame _dialogFrameParent;
	private boolean _dialogFrameParentWasShowing;
	private JLabel _icon;
	private FontMetrics _metrics;
	private JTextArea _text;
	private JPanel _buttonPanel = new JPanel();
	private Vector _buttons = new Vector();
	
	public UIContextDialog(Frame frameParent) {
	    super(frameParent);
//          setResizable(false);

	    _dialogFrameParent = frameParent;
	    _dialogFrameParentWasShowing =
		(frameParent == null ? false : frameParent.isShowing());

	    _maxLineLen = _rs.getInt(MAX_LINE_LEN);

	    JPanel panel = new JPanel();
	    panel.setBorder(BorderFactory.createEmptyBorder(
		_rs.getPixels(TOP_INSET), _rs.getPixels(LEFT_INSET),
		_rs.getPixels(BOTTOM_INSET), _rs.getPixels(RIGHT_INSET)));
	    panel.setLayout(new BorderLayout(0, _rs.getPixels(VGAP))); 
	    
	    JPanel messagePanel = new JPanel();
	    messagePanel.setLayout(new BorderLayout(_rs.getPixels(HGAP), 0)); 
	    panel.add(messagePanel, BorderLayout.NORTH);
	    
	    _text = new JTextArea();
	    Font font = _rs.getFont(FONT);
	    _metrics = Toolkit.getDefaultToolkit().getFontMetrics(font);
	    _text.setFont(font);
	    _text.setEditable(false);
	    _text.setOpaque(false);
	    _text.setBorder(null);
	    messagePanel.add(_text, BorderLayout.CENTER);
	    
	    _icon = new JLabel();
	    messagePanel.add(_icon, BorderLayout.WEST);
	    
	    panel.add(_buttonPanel, BorderLayout.SOUTH);
	    setContentPane(panel);
	}
	
	public UIContextDialog(Frame frameParent, JPanel contents) {
	    super(frameParent);
	    _dialogFrameParent = frameParent;
	    _dialogFrameParentWasShowing =
		(frameParent == null ? false :
		 frameParent.isShowing()); 
	     setContentPane(contents);
	}
	
	public void setText(String text) {
	    _text.setText(text);
	    Dimension textSize = getTextSize(text);
	    _text.setPreferredSize(textSize);
	    
	    // A call to invalidate is required by NT.
	    _text.invalidate();
	}
	
	public void setIcon(String iconName) {
	    _icon.setIcon(_rs.getIcon(iconName));
	}
	
	public void setButtons(JButton[] buttons) {
	    // Remove the existing elements, if any.
	    Enumeration enum = _buttons.elements();
	    while (enum.hasMoreElements()) {
		JButton button = (JButton)enum.nextElement();
		_buttonPanel.remove(button);
	    }
	    _buttons.removeAllElements();
	    
	    // Now add the new buttons.
	    int ii; 
	    for (ii = 0; ii < buttons.length; ii++) {
		_buttons.addElement(buttons[ii]);
		_buttonPanel.add(buttons[ii]);
	    }
	}
	
	public Frame getFrameParent() {
	    return _dialogFrameParent;
	}
	
	public boolean wasFrameParentShowing() {
	    return _dialogFrameParentWasShowing;
	}
	
	private Dimension getTextSize(String text) {
	    Dimension size = new Dimension(0, 0);
	    
	    if (text == null) {
		return size;
	    }
	    
	    int textLength = text.length();
	    int height = _metrics.getHeight();
	    int rowStart = 0;
	    size.height = 0;
	    for (int ii = 0; ii < textLength; ii++) {
		if (text.charAt(ii) == '\n') {
		    if (ii - rowStart > 0) {
			size.width = Math.max(_metrics.stringWidth(
						text.substring(rowStart, ii)),
					      size.width);
		    }
		    size.height += height;
		    rowStart = ii+1;
		}
	    }
	    if (textLength - rowStart > 0) {
		size.width = Math.max(_metrics.stringWidth(
					text.substring(rowStart, textLength)),
				      size.width);
	    }
	    size.height += height;
	    
	    return size;
	}
    }
    
    /**
     * Default constructor.
     */
    public UIContext() {
	_rs = new ResourceStack();
	_rs.pushBundle(ResourceStack.getPackageName(UIContext.class) +
		       "SysadmUI" + ResourceStack.BUNDLE_SUFFIX);
    }
    
    /**
     * Construct a UIContext with the specified <TT>dialogParent</TT>.
     * Convenience for combining the constructor with a call to
     * UIContext.setDialogParent().
     *
     * @param dialogParent JComponent for posting dialogs.
     */
    public UIContext(JComponent dialogParent) {
	this();
	setDialogParent(dialogParent);
    }
    
    /**
     * Protected constructor for creating a UIContext with an existing
     * ResourceStack.
     * 
     * @param rs A ResourceStack to use for this UIContext.
     */
    protected UIContext(ResourceStack rs) {
	_rs = rs;
    }
    
    /**
     * Protected constructor for creating a UIContext with an existing
     * ResourceStack and a <TT>dialogParent</TT>.
     *
     * @param rs A ResourceStack to use for this UIContext.
     * @param dialogParent JComponent for posting dialogs.
     */
    protected UIContext(ResourceStack rs, JComponent dialogParent) {
	_rs = rs;
	setDialogParent(dialogParent);
    }
    
    
    /**
     * Workaround for Component.setCursor that actually changes the
     * cursor without requiring the user to move the mouse.  Works on
     * both IRIX and NT.
     * 
     * @param comp Component to set cursor of.
     * @param cursor new Cursor.
     */
    public static void setCursor(Component comp, Cursor cursor) {
//
// Commented out code in this function is a workaround that works on
// Windows.  Unfortunately, it leads to really awful flashing which is
// arguably worse than the stuck cursor problem.
//
// 	final int UNKNOWN = 0;
// 	final int WINDOWS = 1;
// 	final int MOTIF = 2;
// 	int awtType = UNKNOWN;
	Component topLevel = comp;
	while (true) {
// 	    if (awtType == UNKNOWN && topLevel instanceof Window) {
// 		String peerClass = topLevel.getPeer().getClass().getName();
// 		if (peerClass.startsWith("sun.awt.windows")) {
// 		    awtType = WINDOWS;
// 		} else if (peerClass.startsWith("sun.awt.motif")) {
// 		    awtType = MOTIF;
// 		}
// 	    }

	    Component parent = topLevel.getParent();
	    if (parent == null) {
		break;
	    }
	    topLevel = parent;
	}
	setAllCursors(topLevel, cursor);

// 	if (awtType == WINDOWS) {
// 	    Dimension size = topLevel.getSize();
// 	    topLevel.setSize(size.width + 1, size.height + 1);
// 	    topLevel.setSize(size);
// 	}
    }

    /**
     * Find a Component in <tt>parent</tt>'s Component hierarchy that
     * is showing, enabled, and focus traversable.
     * 
     * @param parent Parent whose hierarchy is to be searched.
     * 
     * @return traversable, showing, enabled Component, or null if no such
     *         Component exists in the hierarchy.
     */
    public static Component findFocusTraversable(Container parent) {
	Component kids[] = parent.getComponents();
	for (int ii = 0; ii < kids.length; ii++) {
	    if (kids[ii].isFocusTraversable() && kids[ii].isShowing()
		&& kids[ii].isEnabled()) {
		return kids[ii];
	    }
	    try {
		Component comp = findFocusTraversable((Container)kids[ii]);
		if (comp != null) {
		    return comp;
		}
	    } catch (ClassCastException ex) {
	    }
	}
	return null;
    }

    /**
     * Set the dialog parent associated with this context.
     *
     * @param dialogParent JComponent over which subsequent dialogs
     *                     should be posted and over which the cursor
     *                     should be changed when using the busy() methods.
     */
    public void setDialogParent(JComponent dialogParent) {
	setDialogParent(dialogParent, null);
    }

    /**  
     * Set the dialog parent associated with this context and notify
     * <TT>listener</TT> when the dialog parent has been set in the
     * Swing UI thread.
     *
     * @param dialogParent JComponent over which subsequent dialogs
     *                     should be posted and over which the cursor
     *                     should be changed when using the busy() methods.
     * @param listener A ResultListener that is called when the
     *                 dialogParent has been set.
     */
    public void setDialogParent(final JComponent dialogParent,
				final ResultListener listener) {
	SwingUtilities.invokeLater(new Runnable() {
	    public void run() {
		Log.assert(!_inputBlocked,
			   "Can't change dialogParent while blocked!");
		try {
		    _dialogParent = dialogParent;
		    _glassPane = null;
		    _triedGlassPane = false;

		    if (_dialogParent == null) {
			_frameParent = null;
		    } else if (_dialogParent.isShowing()) {
			setupFrameParent();
		    } else {
			_dialogParent.addAncestorListener(new AncestorListener() {
			    public void ancestorAdded(AncestorEvent event) {
				_dialogParent.removeAncestorListener(
						    AncestorListener.this);
				setupFrameParent();
			    }
			    
			    public void ancestorMoved(AncestorEvent event) {
			    }
			    
			    public void ancestorRemoved(AncestorEvent event) {
			    }
			});
		    }
		} catch (Exception exception) {
		    // Prevent Swing from consuming exceptions silently.
		    exception.printStackTrace();
		}
		if (listener != null) {
		    listener.succeeded(new ResultEvent(this));
		}		
	    }
	});
    }
    
    /**
     * Get the JComponent which will be used for posting dialogs.
     *
     * @return The JComponent that will be used for posting dialogs.
     */
    public JComponent getDialogParent() {
	return _dialogParent;
    }
    
    /**
     * Set the HostContext associated with this UIContext.
     * 
     * @param hostContext The HostContext associated with this UIContext.
     */
    public void setHostContext(HostContext hostContext) {
	_hostContext = hostContext;
    }

    /**
     * Get the HostContext associated with this UIContext.
     * 
     * @return HostContext associated with this UIContext.
     */
    public HostContext getHostContext() {
	return _hostContext;
    }

    /**
     * Get the ResourceStack associated with this context.
     *
     * @return The ResourceStack associated with this context.
     */
    public ResourceStack getResourceStack() {
	return _rs;
    }

    /**
     * Sets the resource stack used by this context.
     * 
     * @param stack The ResourceStack to use
     */
    public void setResourceStack(ResourceStack stack) {
	_rs = stack;
    }
    
    
    /**
     * Set the title of subsequent dialogs posted via this context.
     *
     * @param dialogTitle Localized dialog title.
     */
    public void setDialogTitle(String dialogTitle) {
	_dialogTitle = dialogTitle;
    }
    
    /**
     * Post a modal error dialog and display a localized error message.
     * This method does not block.  If you need to find out when the user
     * dismisses the dialog, use the version of postError() that includes
     * an ActionListener parameter.
     * <P>
     * This method may be called from any thread and the event dispatch
     * thread will be used to post the modal dialog.
     *
     * @param errorMessage Localized error message to display.
     */
    public void postError(String errorMessage) {
	postError(errorMessage, null);
    }

    /**
     * Post a localized error message in a modal dialog that
     * prevents input to all clients registered with the current HostContext.
     * This method does not block.  If you need to find out when the user
     * confirms the dialog, use the version of postError() that includes
     * an ActionListener parameter.
     * <P>
     * This method may be called from any thread and the event dispatch
     * thread will be used to post the modal dialog.
     *
     * @param errorMessage Localized error message to display.
     */
    public void postGlobalError(String errorMessage) {
	Log.assert(_globalMessage == false, "Global message already posted");
	if (_hostContext != null) {
	    _globalMessage = true;
	}
	postError(errorMessage, null);
    }
    
    /**
     * Post a modal error dialog and display a localized error message.
     * This method does not block.  When the user confirms the dialog, the
     * dialog will be hidden and <TT>listener</TT> will be notified.
     * <P>
     * This method may be called from any thread and the event dispatch
     * thread will be used to post the modal dialog.
     *
     * @param errorMessage Localized error message to display.
     * @param listener The listener to be notified when the user confirms
     *                 the dialog.
     */
    public void postError(String errorMessage, ActionListener listener) {
	postMessageDialog(errorMessage, ERROR_ICON, listener);
    }

    /**
     * Post a localized error message in a modal dialog that
     * prevents input to all clients registered with the current HostContext.
     * This method does not block.  When the user confirms the dialog, the
     * dialog will be hidden and <TT>listener</TT> will be notified.
     * <P>
     * This method may be called from any thread and the event dispatch
     * thread will be used to post the modal dialog.
     *
     * @param errorMessage Localized error message to display.
     * @param listener The listener to be notified when the user confirms
     *                 the dialog.
     */
    public void postGlobalError(String errorMessage,
				ActionListener listener) {
	Log.assert(_globalMessage == false, "Global message already posted");
	if (_hostContext != null) {
	    _globalMessage = true;
	}
	postError(errorMessage, listener);
    }
    
    /**
     * Post a modal warning dialog and display a localized warning message.
     * This method does not block.  If you need to find out when the user
     * confirms the dialog, use the version of postWarning() that includes
     * an ActionListener parameter.
     * <P>
     * This method may be called from any thread and the event dispatch
     * thread will be used to post the modal dialog.
     *
     * @param warningMessage Localized warning message to display.
     */
    public void postWarning(String warningMessage) {
	postWarning(warningMessage, null);
    }

    /**
     * Post a localized warning message in a modal dialog that
     * prevents input to all clients registered with the current HostContext.
     * This method does not block.  If you need to find out when the user
     * confirms the dialog, use the version of postWarning() that includes
     * an ActionListener parameter.
     * <P>
     * This method may be called from any thread and the event dispatch
     * thread will be used to post the modal dialog.
     *
     * @param warningMessage Localized warning message to display.
     */
    public void postGlobalWarning(String warningMessage) {
	Log.assert(_globalMessage == false, "Global message already posted");
	if (_hostContext != null) {
	    _globalMessage = true;
	}
	postWarning(warningMessage, null);
    }
    
    /**
     * Post a modal warning dialog and display a localized warning message.
     * This method does not block.  When the user confirms the dialog, the
     * dialog will be hidden and <TT>listener</TT> will be notified.
     * <P>
     * This method may be called from any thread and the event dispatch
     * thread will be used to post the modal dialog.
     *
     * @param warningMessage Localized warning message to display.
     * @param listener The listener to be notified when the user confirms
     *                 the dialog.
     */
    public void postWarning(String warningMessage, ActionListener listener) {
	postMessageDialog(warningMessage, WARNING_ICON, listener);
    }

    /**
     * Post a localized warning message in a modal dialog that
     * prevents input to all clients registered with the current HostContext.
     * This method does not block.  When the user confirms the dialog, the
     * dialog will be hidden and <TT>listener</TT> will be notified.
     * <P>
     * This method may be called from any thread and the event dispatch
     * thread will be used to post the modal dialog.
     *
     * @param warningMessage Localized warning message to display.
     * @param listener The listener to be notified when the user confirms
     *                 the dialog.
     */
    public void postGlobalWarning(String warningMessage,
				  ActionListener listener) {
	Log.assert(_globalMessage == false, "Global message already posted");
	if (_hostContext != null) {
	    _globalMessage = true;
	}
	postWarning(warningMessage, listener);
    }

    /**
     * Post a modal information dialog and display a localized information
     * message. This method does not block.  If you need to find out when the
     * user confirms the dialog, use the version of postInfo() that includes
     * an ActionListener parameter.
     * <P>
     * This method may be called from any thread and the event dispatch
     * thread will be used to post the modal dialog.
     *
     * @param informationMessage Localized information message to display.
     */
    public void postInfo(String informationMessage) {
	postInfo(informationMessage, null);
    }

    /**
     * Post a localized information message in a modal dialog that
     * prevents input to all clients registered with the current HostContext.
     * This method does not block.  If you need to find out when the
     * user confirms the dialog, use the version of postGlobalInfo()
     * that includes an ActionListener parameter.
     * <P>
     * This method may be called from any thread and the event dispatch
     * thread will be used to post the modal dialog.
     *
     * @param informationMessage Localized information message to display.
     */
    public void postGlobalInfo(String informationMessage) {
	Log.assert(_globalMessage == false, "Global message already posted");
	if (_hostContext != null) {
	    _globalMessage = true;
	}
	postInfo(informationMessage, null);
    }
    
    /**
     * Post a modal information dialog and display a localized information
     * message. This method does not block.  When the user confirms the dialog,
     * the dialog will be hidden and <TT>listener</TT> will be notified.
     * <P>
     * This method may be called from any thread and the event dispatch
     * thread will be used to post the modal dialog.
     *
     * @param informationMessage Localized information message to display.
     * @param listener The listener to be notified when the user confirms
     *                 the dialog.
     */
    public void postInfo(String informationMessage, ActionListener listener) {
	postMessageDialog(informationMessage, INFO_ICON, listener);
    }
    
    /**
     * Post a localized information message in a modal dialog that
     * prevents input to all clients registered with the current HostContext.
     * This method does not block.  When the user confirms the dialog,
     * the dialog will be hidden and <TT>listener</TT> will be notified.
     * <P>
     * This method may be called from any thread and the event dispatch
     * thread will be used to post the modal dialog.
     *
     * @param informationMessage Localized information message to display.
     * @param listener The listener to be notified when the user confirms
     *                 the dialog.
     */
    public void postGlobalInfo(String informationMessage,
			       ActionListener listener) {
	Log.assert(_globalMessage == false, "Global message already posted");
	if (_hostContext != null) {
	    _globalMessage = true;
	}
	postInfo(informationMessage, listener);
    }

    /**
     * Post a localized question in a modal question dialog.  This
     * method does not block.  If the user presses the yes button, the
     * dialog will be hidden and <TT>yesListener</TT> will be
     * notified.  If the user presses the no button, the dialog will
     * be hidden and the <TT>noListener</TT> will be notified.
     * <P>
     * This method may be called from any thread and the event dispatch
     * thread will be used to post the modal dialog.
     * 
     * @param questionMessage Localized question to display.
     * @param yesListener Notified when user press yes button.
     * @param noListener Notified when user press no button.
     */
    public void postQuestion(String questionMessage,
			     ActionListener yesListener,
			     ActionListener noListener) {
	postQuestion(questionMessage, yesListener, noListener, null,
		     null, QUESTION_ICON);
    }

    /**
     * Post a localized question in a modal question dialog with
     * localized button labels.  This method does not block.  If the user
     * presses  the "yes" button, the dialog will be hidden and
     * <TT>yesListener</TT> will be notified.  If the user presses the
     * "no" button, the dialog will be hidden and the
     * <TT>noListener</TT> will be notified.  After the dialog is
     * hidden, subsequent postings of the question dialog will use
     * YES_LABEL and NO_LABEL resource strings.
     * <P>
     * This method may be called from any thread and the event dispatch
     * thread will be used to post the modal dialog.
     * 
     * @param questionMessage Localized question to display.
     * @param yesListener Notified when user press yes button.
     * @param noListener Notified when user press no button.
     * @param yesLabel Localized label to display on the "yes"
     *                 button.  YES_LABEL will be used if this
     *                 parameter is null.
     * @param noLabel Localized label to display on the "no" button.
     *                NO_LABEL will be used if this parameter is null.
     */
    public void postQuestion(final String questionMessage,
			     final ActionListener yesListener,
			     final ActionListener noListener,
			     final String yesLabel, final String noLabel) {
	postQuestion(questionMessage, yesListener, noListener,
		     yesLabel, noLabel, QUESTION_ICON);
     }
    

    /**
     * Post a localized question in a modal question dialog with
     * localized button labels.  This method does not block.  If the user
     * presses  the "yes" button, the dialog will be hidden and
     * <TT>yesListener</TT> will be notified.  If the user presses the
     * "no" button, the dialog will be hidden and the
     * <TT>noListener</TT> will be notified.  After the dialog is
     * hidden, subsequent postings of the question dialog will use
     * YES_LABEL and NO_LABEL resource strings.
     * <P>
     * This method may be called from any thread and the event dispatch
     * thread will be used to post the modal dialog.
     * 
     * @param questionMessage Localized question to display.
     * @param yesListener Notified when user press yes button.
     * @param noListener Notified when user press no button.
     * @param yesLabel Localized label to display on the "yes"
     *                 button.  YES_LABEL will be used if this
     *                 parameter is null.
     * @param noLabel Localized label to display on the "no" button.
     *                NO_LABEL will be used if this parameter is null.
     * @param icon The Icon to show.  Should be one of 
     *             <ul>
     *             <li><tt>BUSY_ICON</tt>
     *             <li><tt>ERROR_ICON</tt>
     *             <li><tt>WARNING_ICON</tt>
     *             <li><tt>INFO_ICON</tt>
     *             <li><tt>QUESTION_ICON</tt>
     *             </ul>
     *
     * @see BUSY_ICON
     * @see ERROR_ICON
     * @see WARNING_ICON
     * @see INFO_ICON
     * @see QUESTION_ICON
     */
    public void postQuestion(final String questionMessage,
			     final ActionListener yesListener,
			     final ActionListener noListener,
			     final String yesLabel, final String noLabel,
			     final String icon) {
	incPendingBusyLevel();
	SwingUtilities.invokeLater(new Runnable() {
	    public void run() {
		try {
		    _yesListener = yesListener;
		    _noListener = noListener;
		    blockInputNow(true);
		    setupQuestionDialog(yesLabel, noLabel);
		    if (_globalMessage) {
			_hostContext.blockAllClients(_questionDialog);
		    }
		    _questionDialog.setTitle(_dialogTitle);
		    // 633403: Format message to fit neatly in dialog.
		    _questionDialog.setText(cleanUp(questionMessage));
		    _questionDialog.setIcon(icon);
		    _questionDialog.pack();
		    if (_dialogParent != null) {
			if (_frameParent != null) {
			    // This is necessary to get the dialog to
			    // come up the right size under Windows.
			    _frameParent.toFront();
			}
		    }
                    _questionDialog.setLocationRelativeTo(_dialogParent);
		    _questionDialog.setVisible(true);
		    _dialogCount++;
		} catch (Exception exception) {
		    // Prevent Swing from consuming exceptions silently.
		    exception.printStackTrace();
		}
	    }
	});
    }

    /**
     * Post a localized question in a modal question dialog that
     * prevents input to all clients registered with the current HostContext.
     * This method does not block.  If the user
     * presses the yes button, the dialog will be hidden and
     * <TT>yesListener</TT> will be notified.  If the user presses the
     * no button, the dialog will be hidden and the
     * <TT>noListener</TT> will be notified.
     * <P>
     * This method may be called from any thread and the event
     * dispatch thread will be used to post the modal dialog.
     *
     * @param questionMessage Localized question to display.
     * @param yesListener Notified when user press yes button.
     * @param noListener Notified when user press no button.
     */
    public void postGlobalQuestion(String questionMessage,
				   ActionListener yesListener,
				   ActionListener noListener) {
	Log.assert(_globalMessage == false, "Global message already posted");
	if (_hostContext != null) {
	    _globalMessage = true;
	}
	postQuestion(questionMessage, yesListener, noListener);
    }
    
    /**
     * Post a localized question in a modal question dialog that
     * prevents input to all clients registered with the current HostContext.
     * This method does not block.  If the user
     * presses the yes button, the dialog will be hidden and
     * <TT>yesListener</TT> will be notified.  If the user presses the
     * no button, the dialog will be hidden and the
     * <TT>noListener</TT> will be notified.
     * <P>
     * This method may be called from any thread and the event
     * dispatch thread will be used to post the modal dialog.
     *
     * @param questionMessage Localized question to display.
     * @param yesListener Notified when user press yes button.
     * @param noListener Notified when user press no button.
     * @param yesLabel Localized label to display on the "yes"
     *                 button.  YES_LABEL will be used if this
     *                 parameter is null.
     * @param noLabel Localized label to display on the "no" button.
     *                NO_LABEL will be used if this parameter is null.
     */
    public void postGlobalQuestion(String questionMessage,
				   ActionListener yesListener,
				   ActionListener noListener,
				   final String yesLabel,
				   final String noLabel) {
	Log.assert(_globalMessage == false, "Global message already posted");
	if (_hostContext != null) {
	    _globalMessage = true;
	}
	postQuestion(questionMessage, yesListener, noListener,
		     yesLabel, noLabel);
    }
    
    /**
     * Check to see if we have a set of privileges.  If we don't,
     * prompt user for the root  passwords and use that to get
     * privileges.  Called by Task.checkPrivs().
     * 
     * @param privs Set of privileges to get.
     * @param listener Gets notified of the result.
     */
    public void getPrivs(final String privs[], final ResultListener listener) {
	Log.assert(_hostContext != null,
		   "setHostContext() must be called before getPriv()");
	final PrivBroker broker = _hostContext.getPrivBroker();
	broker.checkPrivs(privs, new ResultListener() {
	    public void succeeded(ResultEvent event) {
		listener.succeeded(event);
	    }
	    public void failed(ResultEvent event) {
		if (event.getResult() == PrivBroker.REASON_NO_SUCH_PRIV) {
		    listener.failed(event);
		} else {
		    postPrivDialog(broker, privs, listener);
		}
	    }
	});
    }

    /**
     * Post a dialog prompting for the root password.  When the user
     * presses OK, we send it to the server to be checked.  If it
     * checks out OK, then we check privileges again to be sure.
     * 
     * @param PrivBroker Privilege broker for privilege checking.
     * @param privs Privileges we're trying to get.
     * @param ResultListener Gets notified of the result.
     */
    private void postPrivDialog(final PrivBroker broker,
				final String privs[],
				final ResultListener listener) {
	RPanel panel = new RPanel();
	panel.setLayout(new BorderLayout());
	final UIContextDialog privDialog =
	    new UIContextDialog(_frameParent, panel);
	final UIContext uic = new UIContext();
	final String busyDialogTitle = _rs.getString(
	    "UIContext.privDialog.dialogTitle");
	uic.setDialogParent(panel);
	uic.setDialogTitle(busyDialogTitle);
	TaskPage page = new TaskPage(_rs);
	page.setTitle(_rs.getString("UIContext.privDialog.intro.text"));
	final RPasswordField passwd = new RPasswordField(
	    _rs.getInt("UIContext.privDialog.passwd.textColumns"));
	page.addTaskComponent(passwd, "UIContext.privDialog.passwd");
	panel.add(page, BorderLayout.CENTER);
	JPanel buttonPanel = new JPanel();
	panel.add(buttonPanel, BorderLayout.SOUTH);
	final JButton okButton = new JButton(_rs.getString(OK_LABEL));
	addKeyCodeShortCut(okButton, "UIContext.privDialog.okButtonKeyCode");
	buttonPanel.add(okButton);
	okButton.addActionListener(new ActionListener() {
	    public void actionPerformed(ActionEvent event) {
		uic.busy(_rs.getString("UIContext.privDialog.busyMessage"));
		broker.setPassword(passwd.getText());
		broker.checkPassword(new ResultListener() {
		    public void succeeded(ResultEvent ev) {
			broker.checkPrivs(privs, new ResultListener() {
			    public void succeeded(ResultEvent ev) {
				uic.notBusy();
				listener.succeeded(ev);
				privDialog.dispatchEvent(
				    new WindowEvent(
					privDialog,
					WindowEvent.WINDOW_CLOSING));
			    }
			    public void failed(ResultEvent ev) {
				uic.notBusy();
				listener.failed(ev);
				privDialog.dispatchEvent(
				    new WindowEvent(
					privDialog,
					WindowEvent.WINDOW_CLOSING));
			    }
			});
		    }
		    public void failed(ResultEvent ev) {
			uic.setDialogTitle(busyDialogTitle);
			uic.notBusy();
			uic.postError(_rs.getString(
			    "UIContext.privDialog.checkPasswordFailed"));
		    }
		});
	    }
	});
	passwd.addActionListener(new ActionListener() {
	    public void actionPerformed(ActionEvent ev) {
		okButton.doClick();
	    }
	});
	JButton cancelButton = new JButton(_rs.getString(CANCEL_LABEL));
	buttonPanel.add(cancelButton);
	cancelButton.addActionListener(new ActionListener() {
	    public void actionPerformed(ActionEvent event) {
		listener.failed(new ResultEvent(UIContext.this));
		privDialog.dispose();
	    }
	});
	addKeyCodeShortCut(cancelButton,
			   "UIContext.privDialog.cancelButtonKeyCode");
	postDialog(privDialog, busyDialogTitle, null);
    }

    /**
     * Add an accelerator to a <TT>button</TT> using the specified resource.
     * 
     * @param button Button to add accelerator to.
     * @param keyCodeResource Specifies a keyCode resource.
     */
    public void addKeyCodeShortCut(JButton button, String keyCodeResource) {
	try {
	    addKeyCodeShortCut(button, _rs.getInt(keyCodeResource));
	} catch (MissingResourceException ex) {
	}
    }

    /**
     * Add an accelerator to a button using the specified <TT>keyCode</TT>.
     * 
     * @param button Button to add accelerator to. 
     * @param keyCode Specifies keyCode to add.
     */
    public static void addKeyCodeShortCut(final JButton button, int keyCode) {
	button.registerKeyboardAction(new ActionListener() {
	    public void actionPerformed(ActionEvent event) {
		button.doClick();
	    }
	}, null, KeyStroke.getKeyStroke(keyCode, 0),
	JComponent.WHEN_IN_FOCUSED_WINDOW);
    }

    //
    // Busy state handling.
    //
    
    /**
     * Set the cursor to a busy cursor and do not allow user input.  This
     * method may be called from any thread and the event dispatch thread
     * will be used to update the busy cursor and block input.
     */
    public void busy() {
	busy((ResultListener)null);
    }

    /**
     * Set the cursor to a busy cursor and do not allow user input;
     * notify the caller when the cursor has changed to busy.  This
     * method may be called from any thread and the event dispatch thread
     * will be used to update the busy cursor and block input.
     *
     * @param listener Listener notified when the cursor has changed
     *                 to busy.
     */
    public void busy(final ResultListener listener) {
	incPendingBusyLevel();
	SwingUtilities.invokeLater(new Runnable() {
	    public void run() {
		try {
		    busyNow();
		} catch (Exception exception) {
		    // Prevent Swing from consuming exceptions
		    // silently.
		    exception.printStackTrace();
		}
		
		if (listener != null) {
		    listener.succeeded(new ResultEvent(UIContext.this));
		}
	    }
	});
    }

    /**
     * Post a modal busy dialog with a localized busy message and do
     * not allow user input to the dialog parent.  This
     * method may be called from any thread and the event dispatch thread
     * will be used to post the busy dialog.  It is an error to
     * attempt to post multiple busy dialogs.
     *
     * @param busyMsg Localized busy message to display.
     */
    public void busy(String busyMsg) {
	busy(busyMsg, null);
    }
    
    /**
     * Post a modal busy dialog with a localized busy message
     * and a cancel button; do not allow user input to the dialog
     * parent.  This method may be called from any thread
     * and the event dispatch thread will be used to post the busy dialog.
     * <P>
     * It is an error to attempt to post multiple busy dialogs.
     *
     * @param busyMsg Localized busy message to display.
     * @param cancelListener Display a cancel button on the busy dialog and
     *			     notify the specified listener when the cancel
     *			     button is pressed.
     */
    public void busy(final String busyMsg,
		     final ActionListener cancelListener) {
	incPendingBusyLevel();
	SwingUtilities.invokeLater(new Runnable() {
	    public void run() {
		try {
		    Log.assert((_busyDialog == null || 
				    !_busyDialog.isVisible()) &&
			            _cancelListener == null, 
			       "duplicate call to busy");
		
		    _cancelListener = cancelListener;
		    setupBusyDialog();
		    busyNow();
		    // 633403: Format message to fit neatly in dialog.
		    _busyDialog.setText(cleanUp(busyMsg));
		    if (_cancelListener != null) {
			_cancelBusyButton.setVisible(true);
			_cancelBusyButton.addActionListener(_cancelListener);
		    } else {
			_cancelBusyButton.setVisible(false);
		    }
		    _busyDialog.pack();
		    if (_dialogParent != null) {
			if (_frameParent != null) {
			    // This is necessary to get the dialog to
			    // come up the right size under Windows.
			    _frameParent.toFront();
			}
		    }
                    _busyDialog.setLocationRelativeTo(_dialogParent);
		    _busyDialog.setVisible(true);
		} catch (Exception exception) {
		    // Prevent Swing from consuming exceptions silently.
		    exception.printStackTrace();
		}
	    }
	});
    }
    
    /**
     * Clear the busy cursor or busy dialog.  This method may be called from
     * any thread and the event dispatch thread will be used to update the UI.
     */
    public void notBusy() {
	notBusy((ResultListener)null);
    }

    /**
     * Clear the busy cursor or busy dialog.  This method may be called from
     * any thread and the event dispatch thread will be used to update
     * the UI.
     *
     * @param listener Listener to notify when the busy cursor or
     *                 dialog has been cleared.
     */
    public void notBusy(final ResultListener listener) {
	decPendingBusyLevel();
	SwingUtilities.invokeLater(new Runnable() {
	    public void run() {
		try {
		    notBusyNow();
		    if (_busyDialog != null) {
			_busyDialog.setVisible(false);
		    }
		    if (_cancelListener != null) {
			_cancelBusyButton.removeActionListener(_cancelListener);
			_cancelListener = null;
		    }
		} catch (Exception exception) {
		    // Prevent Swing from consuming exceptions silently.
		    exception.printStackTrace();
		}
		
		if (listener != null) {
		    listener.succeeded(new ResultEvent(UIContext.this));
		}
	    }
	});
    }
    
    /**
     * Prevent (or allow) input to the dialogParent by raising
     * (or lowering) the glassPane.  If there is no glass pane, do nothing.
     * <P>
     * This method may be called from any thread and the event dispatch
     * thread will be used to block (or unblock) input.
     * 
     * @param block true if input should be blocked, false if it should
     *              be unblocked.
     */
    public void blockInput(boolean block) {
	blockInput(block, (ResultListener)null);
    }

    /**
     * Prevent (or allow) input to the dialogParent by raising
     * the glassPane and bringing <TT>topWindow</TT> to the top if the
     * User clicks on the blocked dialogParent.  If there is no glass
     * pane, do nothing.
     * <P>
     * This method may be called from any thread and the event dispatch
     * thread will be used to block input.
     * 
     * @param block true if input should be blocked, false if it should
     *              be unblocked.
     * @param topWindow Window to bring to the front if the user
     *                  clicks on the blocked context.
     */
    void blockInput(boolean block, Window topWindow) {
	blockInput(block, topWindow, (ResultListener)null);
    }

    /**
     * Prevent (or allow) input to the dialogParent by raising
     * (or lowering) the glassPane.  If there is no glass pane, do nothing.
     * <P>
     * This method may be called from any thread and the event dispatch
     * thread will be used to block input.
     * 
     * @param block true if input should be blocked, false if it should
     *              be unblocked.
     * @param topWindow Window to bring to the front if the user
     *                  clicks on the blocked context.
     * @param listener Listener to notify when input has been blocked
     *                 or unblocked.
     */
    void blockInput(boolean block, Window topWindow,
		    ResultListener listener) {
	if (block) {
	    _topWindow = topWindow;
	} else {
	    Log.assert(_topWindow == topWindow, "unrecognized top window");
	    _topWindow = null;
	}
	blockInput(block, listener);
    }

    /**
     * Prevent (or allow) input to the dialogParent by raising
     * (or lowering) the glassPane, notifying <TT>listener</TT> when
     * input has been blocked (or unblocked).  If there is no glass
     * pane, do nothing.
     * <P>
     * This method may be called from any thread and the event dispatch
     * thread will be used to block input.
     * 
     * @param block true if input should be blocked, false if it should
     *              be unblocked.
     * @param listener Listener to notify when input has been blocked
     *                 or unblocked.
     */
    public void blockInput(final boolean block,
			   final ResultListener listener) {
	if (block) {
	    incPendingBusyLevel();
	} else {
	    decPendingBusyLevel();
	}
	SwingUtilities.invokeLater(new Runnable() {
	    public void run() {
		blockInputNow(block);
		if (listener != null) {
		    listener.succeeded(new
				       ResultEvent(UIContext.this));
		}
	    }
	});
    }
   
    //
    // UIContext private methods
    //
    
    /**
     * Set up the busy dialog for the first time.
     */
    private void setupBusyDialog() {
	if (_busyDialog != null) {
	    if (!hasFrameParentChanged(_busyDialog.getFrameParent(),
				   _busyDialog.wasFrameParentShowing())) {
		return;
	    } else {
		_busyDialog.dispose();
		_busyDialog = null;
	    }
	}

	_busyDialog = new UIContextDialog(_frameParent);
	_busyDialog.setIcon(BUSY_ICON);
	_cancelBusyButton = new JButton(_rs.getString(CANCEL_LABEL));
	_busyDialog.setButtons(new JButton[] { _cancelBusyButton });
	_busyDialog.setTitle(_dialogTitle);
    }
    
    /**
     * Set up the message dialog for the first time.
     */
    private void setupMessageDialog() {
	if (_messageDialog != null) {
	    if (!hasFrameParentChanged(_messageDialog.getFrameParent(),
				_messageDialog.wasFrameParentShowing())) {
		return;
	    } else {
		_messageDialog.dispose();
		_messageDialog = null;
	    }
	}

	_messageDialog = new UIContextDialog(_frameParent);
	_confirmMessageButton = new JButton(_rs.getString(OK_LABEL));
	_confirmMessageButton.addActionListener(new ActionListener() {
	    public void actionPerformed(ActionEvent event) {
		if (!_messageDialog.isVisible()) {
		    // If the message Dialog is already invisible,
		    // then we must have got a double click on the
		    // confirm button.  Just ignore this second click.
		    return;
		}
		
		_messageDialog.setVisible(false);
		if (_globalMessage) {
		    _hostContext.unblockAllClients();
		    _globalMessage = false;
		}
		if (_confirmListener != null) {
		    _confirmListener.actionPerformed(event);
		    _confirmListener = null;
		}
		decPendingBusyLevel();
		blockInputNow(false);
		_dialogCount--;
	    }
	});
	_messageDialog.addWindowListener(new WindowAdapter() {
	    public void windowClosing(WindowEvent event) {
		if (!_messageDialog.isVisible()) {
		    // If the message Dialog is already invisible,
		    // then we must have got a triple click on the
		    // close icon.  Just ignore this third click.
		    return;
		}
		if (_globalMessage) {
		    _hostContext.unblockAllClients();
		    _globalMessage = false;
		}
		if (_confirmListener != null) {
		    _confirmListener.actionPerformed(
			new ActionEvent(_confirmMessageButton,
				        ActionEvent.ACTION_PERFORMED,
					_rs.getString(OK_LABEL)));
		    _confirmListener = null;
		}
		decPendingBusyLevel();
		blockInputNow(false);
		_dialogCount--;
	    }
	});
	_messageDialog.setButtons(new JButton[] { _confirmMessageButton });
    }
    
    /**
     * Set up the Question Dialog with specific labels on the "yes"
     * and "no" buttons.  The labels will revert to their default
     * after the dialog has been dismissed.
     */
    private void setupQuestionDialog(final String yesLabel,
				final String noLabel) {
	if (_questionDialog != null) {		
	    if (yesLabel == null && noLabel == null &&
		!hasFrameParentChanged(_questionDialog.getFrameParent(),
				_questionDialog.wasFrameParentShowing())) {
		return;
	    } else {
		_questionDialog.dispose();
	    }
	}


	_questionDialog = new UIContextDialog(_frameParent);
	JButton yesButton =
	    new JButton(yesLabel != null ? yesLabel
			                 : _rs.getString(YES_LABEL));
	yesButton.addActionListener(new ActionListener() {
	    public void actionPerformed(ActionEvent event) {
		if (!_questionDialog.isVisible()) {
		    // If the question Dialog is already invisible,
		    // then we must have got a double click on the
		    // yes button.  Just ignore this second click.
		    return;
		}
		if (yesLabel == null && noLabel == null) {
		    _questionDialog.setVisible(false);
		} else {
		    // Get rid of the dialog so that the buttons revert to
		    // their original labels when another question dialog
		    // is posted.
		    _questionDialog.dispose();
		    _questionDialog = null;
		}
		decPendingBusyLevel();
		blockInputNow(false);
		_dialogCount--;
		if (_globalMessage) {
		    _hostContext.unblockAllClients();
		    _globalMessage = false;
		}
		if (_yesListener != null) {
		    _yesListener.actionPerformed(event);
		    _yesListener = null;
		}
	    }
	});
	final JButton noButton =
	    new JButton(noLabel != null ? noLabel
	                                : _rs.getString(NO_LABEL));
	noButton.addActionListener(new ActionListener() {
	    public void actionPerformed(ActionEvent event) {
		if (!_questionDialog.isVisible()) {
		    // If the question Dialog is already invisible,
		    // then we must have got a double click on the
		    // no button.  Just ignore this second click.
		    return;
		}
		decPendingBusyLevel();
		blockInputNow(false);
		_dialogCount--;
		if (yesLabel == null && noLabel == null) {
		    _questionDialog.setVisible(false);
		} else {
		    // Get rid of the dialog so that the buttons revert to
		    // their original labels when another question dialog
		    // is posted.
		    _questionDialog.dispose();
		    _questionDialog = null;
		}
		if (_globalMessage) {
		    _hostContext.unblockAllClients();
		    _globalMessage = false;
		}
		if (_noListener != null) {
		    _noListener.actionPerformed(event);
		    _noListener = null;
		}
	    }
	});

	_questionDialog.addWindowListener(new WindowAdapter() {
	    public void windowClosing(WindowEvent event) {
		if (!_questionDialog.isVisible()) {
		    // If the question Dialog is already invisible,
		    // then we must have got a triple click on the
		    // close icon.  Just ignore this third click.
		    return;
		}
		decPendingBusyLevel();
		blockInputNow(false);
		_dialogCount--;
		if (_globalMessage) {
		    _hostContext.unblockAllClients();
		    _globalMessage = false;
		}
		if (_noListener != null) {
		    _noListener.actionPerformed(
			new ActionEvent(noButton,
					ActionEvent.ACTION_PERFORMED,
					_rs.getString(NO_LABEL)));
		    _noListener = null;
		}
		if (yesLabel != null || noLabel != null) {
		    // Get rid of the dialog so that the buttons revert to
		    // their original labels when another question dialog
		    // is posted.
		    _questionDialog.dispose();
		    _questionDialog = null;
		}
	    }
	});
	_questionDialog.setButtons(new JButton[] { yesButton, noButton });
    }

    /**
     * 633403: Used in cleanUp to first strip out all single newlines
     * in the message.  Multiple adjacent newlines (ex.,
     * "foo\n\n\nbar") are allowed to stand as is, because the text
     * writer really wants a newline and we mustn't strip it out.
     */
    private String stripSingleNewlines(String str) {
	int posNewline = str.indexOf('\n');
	if (-1 == posNewline) { // No newlines found.
	    return str;
	}
	
	int posLastNewline = str.lastIndexOf('\n');
	if (posNewline == posLastNewline) {
	    return str.replace('\n', ' '); // Only one newline found.
	}

	String strToReturn = EMPTY_STRING;
	String nextWord = EMPTY_STRING;
	StringTokenizer st = new StringTokenizer(str, " \t");
	while (st.hasMoreTokens()) {
	    nextWord = st.nextToken();
	    posNewline = nextWord.indexOf('\n');
            // XXX Java bug: charAt doesn't work, so use regionMatches.
	    if (nextWord.regionMatches(true, posNewline, "\n\n", 0, 2)) {
		if (strToReturn.equals(EMPTY_STRING)) {
		    strToReturn = nextWord + " ";
		} else {
		    strToReturn += nextWord + " ";
		}
	    } else {
		strToReturn += nextWord.replace('\n', ' ') + " "; 
	    }
	}
	return strToReturn;
    }
    
    /**
     * 633403: Reformat the message (if needed) into nice short
     * readable lines of approximately equal length.
     */
    private String cleanUp(String message) {

	message = stripSingleNewlines(message);
	int msgLen = message.length();
	if (msgLen <= _maxLineLen) {
	    return message; // Message is already short enough.
	}

	StringTokenizer msg = new StringTokenizer(message, " \t");
	String cleanMsg = EMPTY_STRING;

	boolean needNewline = false;
	int currLineLen = 0;
	while (msg.hasMoreTokens()) {
	    String nextWord = EMPTY_STRING;
	    while (msg.hasMoreTokens() && !needNewline) {
		nextWord = msg.nextToken();
		int posNewline = nextWord.indexOf('\n');
		// Two cases of newlines embedded in word:
		if (-1 != posNewline) {
		    // Case A.  Part up to embedded newline can be
		    // added without exceeding _maxLineLen.  Respect
		    // the embedded newlines by using lastIndexOf.
		    if (currLineLen + 1 + posNewline <= _maxLineLen)
		    {
			if (cleanMsg.equals(EMPTY_STRING)) {
			    cleanMsg = nextWord;
			} else {
			    cleanMsg += " " + nextWord;
			}
			currLineLen = nextWord.length() - 
			    nextWord.lastIndexOf('\n');
		    }
		    // Case B.  Appending part up to newline would
		    // make current line longer than _maxLineLen.  We
		    // append newline, left part, newlines (do 1 extra
		    // to preserve writer's intention), then right
		    // part; reset currLineLen to length of right.
		    else {
			StringTokenizer str = 
			    new StringTokenizer(nextWord, "\n");
			int posLastNewline =
			    nextWord.lastIndexOf('\n');
			String strNextToNewline = str.nextToken();
			cleanMsg += "\n" + strNextToNewline;
			currLineLen = strNextToNewline.length();
			for (int ii=0; ii <= posLastNewline-posNewline;
			     ii++) {
			    cleanMsg += "\n";
			    currLineLen = 0;
			}
			while (str.hasMoreTokens()) {
			    strNextToNewline = str.nextToken();
                            cleanMsg += strNextToNewline;
			    currLineLen += strNextToNewline.length();
			}
		    }
		} 
		// No embedded newlines exist in the word.
		else {
		    if (currLineLen + 1 + nextWord.length() <=
			_maxLineLen) {
			if (cleanMsg.equals(EMPTY_STRING)) {
			    cleanMsg = nextWord;
			    currLineLen = nextWord.length();
			} else {
			    cleanMsg += " " + nextWord;
			    currLineLen += 1 + nextWord.length();
			}
		    } else {
			needNewline = true;
		    }
		}
	    }
	    if (needNewline) {
		cleanMsg += "\n" + nextWord;
		currLineLen = nextWord.length();
		needNewline = false;
	    }
	}

	if (!cleanMsg.equals(EMPTY_STRING)) {
	    return cleanMsg;
	} else {
	    return message;
	}
    }
    
    /**
     * Post a modal message dialog.
     *
     * @param msg Message to be displayed in the dialog.
     * @param messageType The type of message to display, one of
     *                    JOptionDialog.ERROR_MESSAGE,
     *		          JOptionDialog.WARNING_MESSAGE, or 
     *                    JOptionDialog.INFORMATION_MESSAGE.
     */
    private void postMessageDialog(final String message, final String iconName,
				   final ActionListener listener) {
	Log.assert(message != null,
		   "null message passed to postMessageDialog");
	incPendingBusyLevel();
	SwingUtilities.invokeLater(new Runnable() {
	    public void run() {
		_confirmListener = listener;
		try {
		    blockInputNow(true);
		    setupMessageDialog();
		    if (_globalMessage) {
			_hostContext.blockAllClients(_messageDialog);
		    }
		    _messageDialog.setTitle(_dialogTitle);
		    // 633403: Format message to fit neatly in dialog.
		    _messageDialog.setText(cleanUp(message));
		    _messageDialog.setIcon(iconName);
		    _messageDialog.pack();
		    if (_dialogParent != null) {
			if (_frameParent != null) {
			    // This is necessary to get the dialog to
			    // come up the right size under Windows.
			    _frameParent.toFront();
			}
		    }
                    _messageDialog.setLocationRelativeTo(_dialogParent);
		    _messageDialog.setVisible(true);
		    _dialogCount++;
		} catch (Exception exception) {
		    // Prevent Swing from consuming exceptions silently.
		    exception.printStackTrace();
		}
	    }
	});
    }
    
    /**
     * Post a custom dialog.  This method creates a JDialog and makes
     * <tt>panel</tt> its contents.  When the dialog has been posted,
     * <tt>listener.succeeded</tt> will be called.  The JDialog that
     * was posted may be obtained by calling
     * <tt>ResultEvent.getResult</tt> on the ResultEvent that is
     * passed to <tt>listener.succeeded</tt>.
     * 
     * @param panel Contents of the custom dialog.
     * @param title Title of the custom dialog.
     * @param listener Gets notified when the dialog has been posted.
     */
    public void postDialog(final JPanel panel, final String title,
			   final ResultListener listener) {
	final UIContextDialog dialog = 
	    new UIContextDialog(_frameParent, panel);
	postDialog(dialog, title, listener);
    }
		
    /**
     * Post a dialog.
     * 
     * @param dialog Dialog to post. 
     * @param title Title of dialog.
     * @param listener Gets notified when the dialog has been posted.
     */
    private void postDialog(final UIContextDialog dialog,
			    final String title,
			    final ResultListener listener) {
	incPendingBusyLevel();
	SwingUtilities.invokeLater(new Runnable() {
	    public void run() {
		try {
		    blockInputNow(true);
		    dialog.setTitle(title);
		    dialog.pack();
		    if (_dialogParent != null) {
			if (_frameParent != null) {
			    // This is necessary to get the dialog to
			    // come up the right size under Windows.
			    _frameParent.toFront();
			}
		    }
                    dialog.setLocationRelativeTo(_dialogParent);
		    dialog.addWindowListener(new WindowAdapter() {
			public void windowClosing(WindowEvent event) {
			    handleWindowClose();
			}
			public void windowClosed(WindowEvent event) {
			    handleWindowClose();
			}
			private void handleWindowClose() {
			    dialog.removeWindowListener(WindowListener.this);
			    dialog.dispose();
			    decPendingBusyLevel();
			    blockInputNow(false);
			    _dialogCount--;
			}
		    });
		    dialog.show();
		    _dialogCount++;
		    if (listener != null) {
			listener.succeeded(new ResultEvent(UIContext.this,
							   dialog));
		    }
		} catch (Exception exception) {
		    // Prevent Swing from consuming exceptions silently.
		    exception.printStackTrace();
		}
	    }
	});
    }

    /**
     * Block input, and set the busy cursor.
     */
    private void busyNow() {
	if (_dialogParent == null) {
	    return;
	}
	blockInputNow(true);
	if (_busyLevel++ == 0) {
	    _cursor = _dialogParent.getCursor();
	    Cursor busyCursor = new Cursor(Cursor.WAIT_CURSOR);
	    setCursor(_glassPane != null ?
		      _glassPane : _dialogParent, busyCursor);
	}
    }

    /**
     * Restore the cursor back to normal, and unblock input.
     */
    private void notBusyNow() {
	if (_dialogParent == null) {
	    return;
	}
	if (--_busyLevel == 0) {
	    setCursor(_glassPane != null ? _glassPane : _dialogParent,
		      _cursor);
	}
	blockInputNow(false);
    }

    /**
     * Prevent input from getting to the dialogParent by raising
     * the glassPane.  If there is no glass pane, do nothing other
     * than displaying a busy cursor.  This method must be called from
     * the event dispatch thread.
     * 
     * @param block true if input should be blocked, false if it should
     *              be unblocked.
     */
    private void blockInputNow(boolean block) {
	if (_dialogParent == null) {
	    return;
	}

	if (block) {
	    _blockLevel++;
	    if (_blockLevel > 1) {
		Log.assert(_inputBlocked, "busyLevel out of sync.");
		// Input is already blocked, so simply return.
		return;
	    }
	} else {
	    Log.assert(_blockLevel > 0, "unblocked input too many times");
	    _blockLevel--;
	    if (_blockLevel > 0) {
		// Input should remain blocked, so simply return
		return;
	    }
	}

	_inputBlocked = block;

	if (!_triedGlassPane) {
	    setGlassPane();
	}

	if (block) {
	    if (_glassPane != null) {
		_glassPane.setVisible(true);
		Container topLevelAncestor =
		    _dialogParent.getTopLevelAncestor();
		if (topLevelAncestor == null ||
		    !(topLevelAncestor instanceof Window))
		{
		    _saveFocus = null;
		} else {
		    _saveFocus =
			((Window)topLevelAncestor).getFocusOwner();
		}
		_glassPane.requestFocus();
	    }
	} else {
	    if (_glassPane != null) {
		_glassPane.setVisible(false);
		if (_saveFocus != null && _saveFocus.isShowing()
		    && _saveFocus.isEnabled()) {
		    _saveFocus.requestFocus();
		}
		_saveFocus = null;
	    }
	}
    }
    
    /**
     * Get the glassPane for this _dialogParent so we can block
     * input during modal dialogs and busy blocks.
     */
    private void setGlassPane() {
	_triedGlassPane = true;
	if (_dialogParent instanceof RootPaneContainer) {
	    _glassPane =
		((RootPaneContainer)_dialogParent).getGlassPane();
	} else if (_dialogParent instanceof JComponent) {
	    if (((JComponent)_dialogParent).getRootPane() == null) {
		// Try again when _dialogParent is visible.
		_triedGlassPane = false;
		return;
	    }
	    _glassPane =
		((JComponent)_dialogParent).getRootPane().getGlassPane();
	} else {
	    Log.debug("UIContext.java", "_dialogParent is neither a " +
				        "RootPaneContainer nor JComponent!");
	    return;
	}
	
	// Beep if the user attempts to click on the blocked area.
	_glassPane.addMouseListener(new MouseAdapter() {
	    public void mouseClicked(MouseEvent event) {
		if (_dialogCount > 0) {
		    Toolkit.getDefaultToolkit().beep();
		}
		if (_topWindow != null) {
		    _topWindow.toFront();
		    Toolkit.getDefaultToolkit().beep();
		}
	    }
	});
    }
    

    /**
     * Get the frame parent of the dialog parent, if any.
     */
    private void setupFrameParent() {
	Component frameParent = (Component)_dialogParent;
	while (frameParent != null && !(frameParent instanceof Frame)) {
	    // If we're posting one dialog on top of the other, this
	    // causes us to use the same frame parent for both, which
	    // has the effect that the topmost dialog can't go behind
	    // the frame.  It can still go behind the dialog below it,
	    // but there's nothing we can do about it.
	    if (frameParent instanceof RDialog) {
		frameParent = ((RDialog)frameParent).getOwner();
		break;
	    }
	    frameParent = frameParent.getParent();
	}

	_frameParent = (Frame)frameParent;
    }
    
    /**
     * Determine if there is a new parent frame or if the visibility of the
     * parent frame has changed since the last time a dialog was posted.
     */
    private boolean hasFrameParentChanged(Frame prevFrameParent,
                                          boolean prevFrameIsShowing) {
	if (_frameParent != prevFrameParent || (_frameParent != null &&
		_frameParent.isShowing() != prevFrameIsShowing)) {
	    return true;
	}
	
	return false;
    } 

    /**
     * Set <tt>comp</tt>'s and all of its children's cursors.  This is
     * necessary to workaround java bug in which cursor does not
     * change until user moves the mouse.
     * 
     * @param comp Component to set cursor on.
     * @param cursor Cursor to set.
     */
    private static void setAllCursors(Component comp, Cursor cursor) {
	comp.setCursor(cursor);
	try {
	    Component kids[] = ((Container)comp).getComponents();
	    for (int ii = 0; ii < kids.length; ii++) {
		setAllCursors(kids[ii], cursor);
	    }
	} catch (ClassCastException ex) {
	}
    }
}
