//
// GuideNavigationEvent.java
//
//	An event sent to TaskControlListeners when the user 
//      navigates among pages of a Guide.
//	
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui.event;

import com.sgi.sysadm.util.*;
import com.sgi.sysadm.ui.*;

/**
 * GuideNavigationEvent is sent to TaskControlListeners to indicate that
 * the user wants to navigate among the pages of a Guide.
 */
public class GuideNavigationEvent extends java.util.EventObject {
    
    /**
     * This pageCode indicates that the user has requested that the first
     * page of the guide be displayed.
     */
    public static final int FIRST_PAGE = 0;
    
    /**
     * This pageCode indicates that the user has requested that the last
     * page of the guide be displayed.
     */
    public static final int LAST_PAGE = 1;
    
    /**
     * This pageCode indicates that the user has requested that the next
     * page of the guide be displayed.
     */
    public static final int NEXT_PAGE = 2;
    
    /**
     * This pageCode indicates that the user has requested that the previous
     * page of the guide be displayed.
     */
    public static final int PREV_PAGE = 3;
    
    /**
     * This pageCode indicates that the user has requested that a specific
     * page of the guide be displayed.  The pageName can be retrieved via
     * GuideNavigationEvent.getPageName().
     */
    public static final int SPECIFIC_PAGE = 4;
    
    private int _pageCode;
    private String _pageName;
    
    /**
     * Constructor.
     *
     * @param controlPanel The control panel generating the navigation event.
     * @param pageCode The code describing the navigation desired.
     */
    public GuideNavigationEvent(TaskControlPanel controlPanel, int pageCode) {
	super(controlPanel);
	Log.assert(pageCode == FIRST_PAGE || pageCode == LAST_PAGE ||
		   pageCode == NEXT_PAGE || pageCode == PREV_PAGE,
		   "pageCode must be one of FIRST_PAGE, LAST_PAGE, " +
		   " or NEXT_PAGE.");
	_pageCode = pageCode;
	_pageName = null;
    }
    
    /**
     * Constructor.
     *
     * @param controlPanel The control panel generating the navigation event.
     * @param pageName The name of the specific page to which the user wants
     *                 to navigate.  pageCode will be set to SPECIFIC_PAGE.
     */
    public GuideNavigationEvent(TaskControlPanel controlPanel, String pageName) {
	super(controlPanel);
	_pageCode = SPECIFIC_PAGE;
	_pageName = pageName;
    }
    
    /**
     * Get the control panel that generated this event.
     *
     * @return The control panel that generated this event.
     */
    public TaskControlPanel getControlPanel() {
	return (TaskControlPanel)getSource();
    }
    
    /**
     * Get the code describing the navigation desired.
     *
     * @return The code describing the navigation desired.
     */
    public int getPageCode() {
	return _pageCode;
    }
    
    /**
     * Get the name of the page to which the user wants to navigate.
     *
     * @return The name of the page to which the user wants to navigate
     *         if this is a SPECIFIC_PAGE event, null otherwise.
     */
    public String getPageName() {
	if (_pageCode != GuideNavigationEvent.SPECIFIC_PAGE) {
	    return null;
	} else {
	    return _pageName;
	}
    }
}
