//
// TaskLaunchRequestEvent
//   
//	An event describing a request to launch a task
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui.event;

import com.sgi.sysadm.ui.*;
import com.sgi.sysadm.ui.taskData.*;
import com.sgi.sysadm.category.*;
import java.util.*;

/**
 * An object that specifies an Task.  From the data contained in
 * this object, a TaskLaunchRequestListener can
 * create an Task of the type that the <tt>source</tt> is
 * requesting be launched.
 * <p>
 * See the TaskLaunchRequestListener documentation about how this
 * event is used.
 *
 * @see TaskLaunchRequestListener
 * @see Task
 */
public class TaskLaunchRequestEvent extends EventObject {
    private TaskLoader _taskLoader;
    private Vector _operands;
    private TaskData _taskData;
    private int _resultHandling = TaskFrame.POST_RESULT_PANEL;

    /**
     * Constructor to use when there are operands or TaskData to pass
     * to the new Task.
     *
     * @param source The source of the event.
     * @param taskLoader The TaskLoader representing the task to be launched.
     * @param operands A Vector of Object containing the set of operands to
     *                 pass to the Task.  <TT>operands</TT> may be null or
     *                 empty if there are no operands to pass.
     * @param taskData TaskData containing the TaskData attributes to set on
     *                 the new Task. <TT>taskData</TT> may be null or empty if
     *                 there are no attributes to set.
     */
    public TaskLaunchRequestEvent(Object source, TaskLoader taskLoader,
				  Vector operands, TaskData taskData) {
	super(source);

	_taskLoader = taskLoader;

	if (operands != null && operands.size() > 0) {
	    _operands = operands;
	}

	if (taskData != null && taskData.size() > 0) {
	    _taskData = taskData;
	}
    }

    /**
     * Constructor to use when there is an Item to pass to the new
     * Task as an operand.
     *
     * @param source The source of this event.
     * @param taskLoader The TaskLoader representing the task to be launched.
     * @param item An Item to be passed to the Task as an operand.
     */
    public TaskLaunchRequestEvent(Object source, 
				  TaskLoader taskLoader, 
				  Item item) {
	super(source);

	_taskLoader = taskLoader;

	if (item != null) {
	    _operands = new Vector();
	    _operands.addElement(item.getSelector());
	}	
    }
     
    /** 
     * Constructor to use when there are no operands or TaskData to
     * pass to the new Task.
     *
     * @param source The source of this event.
     * @param taskLoader The TaskLoader representing the task to be launched.
     */
    public TaskLaunchRequestEvent(Object source, TaskLoader taskLoader) {
	this(source, taskLoader, null, null);
    }

    /** 
     * Get the TaskLoader representing the task to be launched.
     *
     * @return A TaskLoader
     */ 
    public TaskLoader getTaskLoader() {
	return _taskLoader;
    }

    /**
     * Get the operands to pass to the Task.
     *
     * @return A Vector of objects to pass to the Task.  Will be <tt>null</tt> if there
     *         are no operands.
     */
    public Vector getOperands() {
	return _operands;
    }

    /**
     * Set the operands to pass to the Task.
     *
     * @param operands A Vector of Objects to pass to the Task as operands.
     *                 May be null or empty if there are no operands to pass.
     */
    public void setOperands(Vector operands) {
	if (operands == null || operands.size() == 0) {
	    _operands = null;
	} else {
	    _operands = operands;
	}
    }

    /**
     * Get the TaskData attributes to set on the Task.
     *
     * @return TaskData containing the attributes to set on the Task.
     * Will be <tt>null</tt> if there are no attributes to set.
     */
    public TaskData getTaskData() {
	return _taskData;
    }

    /**
     * Set the TaskData attributes to set on the Task.
     *
     * @param taskData TaskData containing the attributes to set on the Task.
     *                 May be null or empty if there are no attributes to set.
     */
    public void setTaskData(TaskData taskData) {
	if (taskData == null || taskData.size() == 0) {
	    _taskData = null;
	} else {
	    _taskData = taskData;
	}
   }
}
