//
// TaskResult.java
//
//	An event that provides information about the successful or failed
//	completion of a Task. 
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//
package com.sgi.sysadm.ui.event;

import com.sgi.sysadm.util.*;
import com.sgi.sysadm.ui.*;
/**
 * An event that provides information about whether a task succeeded or
 * was cancelled by the User.  If a task fails, the Task interface
 * remains displayed so that the User may make changes to the data
 * entered and try again.  Ultimately the User will cancel the task if
 * it cannot be performed for some reason.
 */
public class TaskResult extends java.util.EventObject {
    private int _resultType;
    private String _outputText = "";
    private String _errorText = "";

    /**
     * Results of type TaskResult.SUCCEEDED are generated when the task
     * privileged command(s) complete(s) successfully.  Even if the
     * task is successful, there may
     * be warnings or other text sent either to stdout or stderr.  Use
     * TaskResult.getOutputText() and TaskResult.getErrorText() to get the
     * output and error data from the privileged command(s).
     */
    public static final int SUCCEEDED = 0;
    
    /**
     * Results of type TaskResult.CANCELLED are generated when a task
     * is cancelled by the user.
     */
    public static final int CANCELLED = 1;
    
    /**
     * Constructor.
     *
     * @param task The Task that generated this result.
     * @param type The type of result generated.  One of:
     *             TaskResult.SUCCEEDED or TaskResult.CANCELLED.
     */
    public TaskResult(Task task, int resultType) {
	super(task);
	
	Log.assert(resultType == SUCCEEDED || resultType == CANCELLED,
		   "Illegal TaskResult type.  It must be one of: " +
		   "TaskResult.SUCCEEDED or TaskResult.CANCELLED.");
	
	_resultType = resultType;
    }
    
    /**
     * Get the Task that generated this result.
     *
     * @return The Task that generated this result.
     */
    public Task getTask() {
	return (Task)getSource();
    }
    
    /**
     * Get the type of this result.
     *
     * @return The type of this result.  One of: TaskResult.SUCCEEDED or
     *         TaskResult.CANCELLED.
     */
    public int getResultType() {
	return _resultType;
    }
    
    /**
     * Set the output text associated with a SUCCEEDED result.
     *
     * @param outputText The output data generated by the privileged command(s).
     */
    public void setOutputText(String outputText) {
	Log.assert(_resultType == SUCCEEDED,
		   "Result type must be ResultEvent.SUCCEEDED");
	_outputText = outputText;
    }
    
    /**
     * Get the output text associated with a SUCCEEDED result.
     *
     * @return The output data generated by the privileged command(s).
     */
    public String getOutputText() {
	Log.assert(_resultType == SUCCEEDED,
		   "Result type must be ResultEvent.SUCCEEDED");
	return _outputText;
    }
    
    /**
     * Set the error text associated with a SUCCEEDED result.
     *
     * @param errorText The error data generated by the privileged command(s).
     */
    public void setErrorText(String errorText) {
	Log.assert(_resultType == SUCCEEDED,
		   "Result type must be ResultEvent.SUCCEEDED");
	_errorText = errorText;
    }
    
    /**
     * Get the error text associated with a SUCCEEDED result.
     *
     * @return The error data generated by the privileged command(s).
     */
    public String getErrorText() {
	Log.assert(_resultType == SUCCEEDED,
		   "Result type must be ResultEvent.SUCCEEDED");
	return _errorText;
    }
}
