//
// Document.java
//
//	A rich text document.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui.richText;

import java.awt.*;
import java.util.*;

/**
 * Document objects are for the root of the object hierarchy used to
 * represent the text areas "value" resource.
 */
class Document extends TextBlock {

    Vector _blocks = new Vector();
    int _componentWidth = -1;

    /**
     * Document constructor.  Create a TokenFactory, and parse tokens
     * into text blocks.
     * 
     * @param state Graphics state.
     */
    Document(GraphicsState state) {
	if (state._text == null) {
	    return;
	}

	TokenFactory factory = new TokenFactory(state._text);

	boolean lineBreakNeeded = false;
	while (true) {
	    Token token = factory.getNextToken();
	    TextBlock block = null;
	    if (token.getType() == Token.TAG_TOKEN) {
		switch (token.getTagValue()) {
		case Token.UNORDERED_LIST:
		    block = new UnorderedList(state, factory, 0);
		    break;
		case Token.ORDERED_LIST:
		    block = new OrderedList(state, factory, 0);
		    break;
		case Token.FONT:
		case Token.LINK:
		case Token.ITALIC:
		case Token.BOLD:
		    factory.unget();
		    // Intentional fall-thru; a paragraph might begin with
		    // bold, italic, or a link.
		case Token.PARAGRAPH:
		    block = new Paragraph(state, factory, 0);
		    break;
		case Token.BREAK:
		    // This seems counter-intuitive, but it makes
		    // sense.  If a paragraph ends with a <br> tag,
		    // the Paragraph code will unget the token.  We
		    // want to start the next Paragraph without an
		    // extra line in between, so we set
		    // lineBreakNeeded to false.
		    lineBreakNeeded = false;
		    break;
		case Token.END_DOCUMENT:
		    return;
		}
	    } else if (token.getType() == Token.WORD_TOKEN) {
		factory.unget();
		block = new Paragraph(state, factory, 0);
	    }
	    if (block != null) {
		if (lineBreakNeeded) {
		    _blocks.addElement(new LineBreak());
		}
		lineBreakNeeded = true;
		_blocks.addElement(block);
	    }
	}
    }

    /**
     * Calculate our geometry, which involves calculating geometry
     * for all objects in the document.
     * 
     * @param state Graphics parameters.
     * @param y Where we're supposed to start.
     * @param componentWidth Width of the rich text component.
     */
    void setGeometry(GraphicsState state, int y, int componentWidth) {
	_componentWidth = componentWidth;
	_x = state._margin.left;
	_y = y + state._margin.top;

	y += state._margin.top;
	state._maxX = 0;

	final int size = _blocks.size();
	for (int ii = 0; ii < size; ii++) {
	    TextBlock block = (TextBlock)_blocks.elementAt(ii);
	    block.setGeometry(state, y, componentWidth);
	    y += block.getHeight();
	}

	_height = y + state._margin.bottom;
	_width = state._maxX + state._margin.right;
    }

    /**
     * Draw a document.
     * 
     * @param state Graphics parameters.
     */
    void draw(GraphicsState state) {
	final int size = _blocks.size();
	for (int ii = 0; ii < size; ii++) {
	    TextBlock block = (TextBlock)_blocks.elementAt(ii);
	    block.draw(state);
	}
    }

    /**
     * Get the componentWidth passed to the last call to setGeometry.
     * This is used by RichTextComponent to decide whether it needs to
     * call setGeometry again.
     * 
     * @return componentWidth that was passed to the last call to
     *         setGeometry.
     */
    int getComponentWidth() {
	return _componentWidth;
    }
}
