//
// GraphicsState.java
//
//	Parameters shared by elements in a RichText scene graph that
//	govern how a Document is drawn.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui.richText;

import java.awt.*;
import java.util.*;
import com.sgi.sysadm.util.*;
import com.sgi.sysadm.ui.*;

/**
 * GraphicsState maintains parameters shared by the elements in a
 * RichText scene graph such as fonts and colors.  Parameters are
 * normally set by specifying them in a .properties file.  These
 * settings are then passed to GraphicsState via a ResourceStack.
 *
 * More discussion of these resources can be found in the RichText
 * documentation.
 */
class GraphicsState {

    Graphics _graphics = null;
    Font _font = null;
    FontMetrics _fontMetrics = null;
    boolean _inLink = false;
    String _text;
    int _maxX;
    String _listNumberFormat;
    
    int _width;
    int _indentWidth;
    int _marginLeft;
    int _marginRight;
    int _marginTop;
    int _marginBottom;
    int _lineSpacing;

    Insets _margin = null;

    Color _textColor;
    Color _linkColor;
    Color _activeLinkColor;
    Color _visitedLinkColor;

    Font _plainFont;
    FontMetrics _plainFontMetrics;

    Font _boldFont;
    Font _italicFont;
    Font _boldItalicFont;

    int _bulletLeftOffset;
    int _bulletTopOffset;

    boolean _underlineLinks;
    boolean _boldLinks;
    boolean _autoWrap;
    float _lineBreakScale;

    // We maintain a list of links separate from the rest of the
    // Document for more efficient hit testing when the user moves or
    // clicks the mouse.
    Vector _links = new Vector();

    private Stack _colorStack = new Stack();
    
    private static ResourceStack _defaults;
    
    static {
	_defaults = new ResourceStack();
	_defaults.pushBundle("com.sgi.sysadm.ui.richText.RichText" +
			    ResourceStack.BUNDLE_SUFFIX);
    }

    /**
     * Called by RichText to prepare us for geometry calculation or
     * drawing.
     * 
     * @param graphics Graphics for geometry calculationd and /drawing.
     */
    void setGraphics(Graphics graphics) {
	if (_graphics == null) {
	    // The first time we're called, we do some initial setup.
	    _font = _plainFont;
	    _fontMetrics = graphics.getFontMetrics(_font);
	    _plainFontMetrics = _fontMetrics;
	}
	_graphics = graphics;
	graphics.setColor(_textColor);
    }

    /**
     * Add a link to the list of links in this document.
     * 
     * @param link Link to add.
     */
    void addLink(Link link) {
	_links.addElement(link);
    }

    /**
     * Return the Link under a position if there is one.
     * 
     * @param x horizontal coordinate.
     * @param y vertical coordinate.
     * 
     * @return a Link, or null if there is no Link at (x, y).
     */
    Link getLink(int x, int y) {
	final int size = _links.size();
	for (int ii = 0; ii < size; ii++) {
	    Link link = (Link)_links.elementAt(ii);
	    if (link.intersects(x, y)) {
		return link;
	    }
	}
	return null;
    }

    /**
     * Remove all links.  Useful when we switch documents.
     */
    void removeAllLinks() {
       _links.removeAllElements();
    }

    /**
     * Called when our RichText component gets a Border.  We calculate
     * our own Insets which combine our component's insets with our
     * own margins.
     * 
     * @param insets RichText component insets.
     */
    void setInsets(Insets insets) {
	_margin = new Insets(insets.top + _marginTop,
			     insets.left + _marginLeft,
			     insets.bottom + _marginBottom,
			     insets.right + _marginRight);
    }

    /**
     * Set our values from a ResourceStack.
     * 
     * @param name prepend to resource names we lookup.
     * @param rs ResourceStack to get resources from.
     */
    void setValues(String name, ResourceStack rs) {

	_text = getString(rs, name, RichTextComponent.TEXT);
	_width = getInt(rs, name, RichTextComponent.WIDTH);
	_listNumberFormat = getString(rs, name,
				      RichTextComponent.LIST_NUMBER_FORMAT);

	_marginLeft = getInt(rs, name, RichTextComponent.MARGIN_LEFT);
	_marginRight = getInt(rs, name,RichTextComponent. MARGIN_RIGHT);
	_marginTop = getInt(rs, name, RichTextComponent.MARGIN_TOP);
	_marginBottom =  getInt(rs, name, RichTextComponent.MARGIN_BOTTOM);
	_indentWidth = getInt(rs, name, RichTextComponent.INDENT_WIDTH);
 	_lineSpacing = getInt(rs, name, RichTextComponent.LINE_SPACING);
 	_textColor = getColor(rs, name, RichTextComponent.TEXT_COLOR);
 	_linkColor = getColor(rs, name, RichTextComponent.LINK_COLOR);
 	_activeLinkColor = getColor(rs, name,
				    RichTextComponent.ACTIVE_LINK_COLOR);
 	_visitedLinkColor = getColor(rs, name,
				     RichTextComponent.VISITED_LINK_COLOR);
  	_plainFont = getFont(rs, name, RichTextComponent.FONT);
	String fontName = _plainFont.getName();
	int size = _plainFont.getSize();
	_boldFont = SysUtil.loadFont(fontName, Font.BOLD, size);
	_italicFont = SysUtil.loadFont(fontName, Font.ITALIC, size);
	_boldItalicFont = SysUtil.loadFont(fontName, Font.BOLD | Font.ITALIC, size);

 	_bulletLeftOffset = getInt(rs, name,
				   RichTextComponent.BULLET_LEFT_OFFSET);
 	_bulletTopOffset = getInt(rs, name,
				  RichTextComponent.BULLET_TOP_OFFSET);
	_underlineLinks = getBool(rs, name,
				  RichTextComponent.UNDERLINE_LINKS);
	_boldLinks = getBool(rs, name, RichTextComponent.BOLD_LINKS);
	_lineBreakScale = getFloat(rs, name,
				   RichTextComponent.LINE_BREAK_SCALE);
	_autoWrap = getBool(rs, name, RichTextComponent.AUTO_WRAP);
	convertSizes();
    }

    /**
     * Push the current color onto our color stack.  Make
     * <tt>color</tt> the new current color.
     * 
     * @param color Color to push onto the stack.
     */
    void pushColor(Color color) {
	_colorStack.push(_textColor);
	_textColor = color;
	_graphics.setColor(_textColor);
    }

    /**
     * Pop a color off of our color stack.  Make the popped color the
     * current color.
     */
    void popColor() {
	if (_colorStack.isEmpty()) {
	    Log.warning(RichTextComponent.CLASS_NAME, "Empty color stack");
	    return;
	}
	_textColor = (Color)_colorStack.pop();
	_graphics.setColor(_textColor);
    }

    /**
     * Get an int value from <tt>rs</tt>.  First look up
     * <tt>name</tt>.<tt>resource</tt>.  If that fails, try
     * <tt>CLASS_NAME</tt>.<tt>resource</tt>.  The second lookup
     * should never fail because we have RichTextP at the bottom of
     * our ResourceStack.
     * 
     * @param rs ResourceStack to look resources up in.
     * @param name Prepend this to <tt>resource</tt>.
     * @param resource Name of resource to look up.
     * 
     * @return int resource.
     */
    private static int getInt(ResourceStack rs, String name, String resource) {
	String defaultLookup = 
	    RichTextComponent.CLASS_NAME_DOT + resource;
	if (rs != null) {
	    return rs.getInt(name != null ? 
			     new String[] { name + "." + resource,
					    defaultLookup} :
			     new String[] {defaultLookup},
			     _defaults.getInt(defaultLookup));
	}
	return _defaults.getInt(defaultLookup);
    }

    /**
     * Get a boolean value from <tt>rs</tt>.  First look up
     * <tt>name</tt>.<tt>resource</tt>.  If that fails, try
     * <tt>CLASS_NAME</tt>.<tt>resource</tt>.  The second lookup
     * should never fail because we have RichTextP at the bottom of
     * our ResourceStack.
     * 
     * @param rs ResourceStack to look resources up in.
     * @param name Prepend this to <tt>resource</tt>.
     * @param resource Name of resource to look up.
     * 
     * @return boolean resource.
     */
    private static boolean getBool(ResourceStack rs, String name, String resource) {
	String defaultLookup = 
	    RichTextComponent.CLASS_NAME_DOT + resource;
	if (rs != null) {
	    return rs.getBoolean(name != null ? 
				 new String[] {name + "." + resource,
					       defaultLookup} :
				 new String[] {defaultLookup},
				 _defaults.getBoolean(defaultLookup));  
	}
	return _defaults.getBoolean(defaultLookup);
    }

    /**
     * Get a String value from <tt>rs</tt>.  First look up
     * <tt>name</tt>.<tt>resource</tt>.  If that fails, try
     * <tt>CLASS_NAME</tt>.<tt>resource</tt>.  The second lookup
     * should never fail because we have RichTextP at the bottom of
     * our ResourceStack.
     * 
     * @param rs ResourceStack to look resources up in.
     * @param name Prepend this to <tt>resource</tt>.
     * @param resource Name of resource to look up.
     * 
     * @return String resource.
     */
    private static String getString(ResourceStack rs, String name, String resource) {
	String returnVal;
	String defaultLookup = 
	    RichTextComponent.CLASS_NAME_DOT + resource;
	if (rs != null) {
	    returnVal = rs.getString(name != null ? 
				     new String[] {name + "." + resource,
						   defaultLookup} :
				     new String[] {defaultLookup}, 
				     null);
	    if (returnVal != null) {
		return returnVal;
	    }
	}
	// If we get here, then we didn't find the resource, and need
	// to look in the defaults.
	return _defaults.getString(defaultLookup);
    }

    /**
     * Get a float value from <tt>rs</tt>.  First look up
     * <tt>name</tt>.<tt>resource</tt>.  If that fails, try
     * <tt>CLASS_NAME</tt>.<tt>resource</tt>.  The second lookup
     * should never fail because we have RichTextP at the bottom of
     * our ResourceStack.
     * 
     * @param rs ResourceStack to look resources up in.
     * @param name Prepend this to <tt>resource</tt>.
     * @param resource Name of resource to look up.
     * 
     * @return float resource.
     */
    private static float getFloat(ResourceStack rs, String name, String resource) {
	String defaultLookup = 
	    RichTextComponent.CLASS_NAME_DOT + resource;
	if (rs != null) {
	    return rs.getFloat(name != null ? 
			       new String[] {name + "." + resource,
					     defaultLookup} :
			       new String[] {defaultLookup},
			       _defaults.getFloat(defaultLookup));
	} else {
	    return _defaults.getFloat(defaultLookup);   
	}
    }

    /**
     * Get a Font value from <tt>rs</tt>.  First look up
     * <tt>name</tt>.<tt>resource</tt>.  If that fails, try
     * <tt>CLASS_NAME</tt>.<tt>resource</tt>.  The second lookup
     * should never fail because we have RichTextP at the bottom of
     * our ResourceStack.
     * 
     * @param rs ResourceStack to look resources up in.
     * @param name Prepend this to <tt>resource</tt>.
     * @param resource Name of resource to look up.
     * 
     * @return Font resource.
     */
    private static Font getFont(ResourceStack rs, String name, String resource) {
	return SysUtil.loadFont(getString(rs, name, resource));
    }

    /**
     * Get a Color value from <tt>rs</tt>.  First look up
     * <tt>name</tt>.<tt>resource</tt>.  If that fails, try
     * <tt>CLASS_NAME</tt>.<tt>resource</tt>.  The second lookup
     * should never fail because we have RichTextP at the bottom of
     * our ResourceStack.
     * 
     * @param rs ResourceStack to look resources up in.
     * @param name Prepend this to <tt>resource</tt>.
     * @param resource Name of resource to look up.
     * 
     * @return Color resource.
     */
    private static Color getColor(ResourceStack rs, String name, String resource) {
	return Color.decode(getString(rs, name, resource));
    }

    /**
     * Convert all of our size parameters from points to pixels.
     */
    private void convertSizes() {
	_width = SysUtil.pixelsFromPoints(_width);
	_marginLeft = SysUtil.pixelsFromPoints(_marginLeft);
	_marginRight = SysUtil.pixelsFromPoints(_marginRight);
	_marginTop = SysUtil.pixelsFromPoints(_marginTop);
	_marginBottom = SysUtil.pixelsFromPoints(_marginBottom);
	_indentWidth = SysUtil.pixelsFromPoints(_indentWidth);
	_lineSpacing = SysUtil.pixelsFromPoints(_lineSpacing);
	_bulletLeftOffset
	    = SysUtil.pixelsFromPoints(_bulletLeftOffset);
	_bulletTopOffset = SysUtil.pixelsFromPoints(_bulletTopOffset);
    }
}
