//
// Link.java
//
//	The start of a link.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui.richText;

import java.awt.*;
import java.util.*;
import com.sgi.sysadm.util.*;

/**
 * A Link object represents the start of a link.  The draw method does
 * stuff that affects the rendering of subsequent words.
 * GraphicsState maintains a list of words that comprise the link, so
 * that we can do intersection tests in response to mouse motion and
 * button presses.
 */
class Link extends Word {

    // States that links can be in.
    static final int INITIAL = RichTextComponent.LINK_STATE_INITIAL;
    static final int ACTIVE = RichTextComponent.LINK_STATE_ACTIVE;
    static final int VISITED = RichTextComponent.LINK_STATE_VISITED;

    private int _state = INITIAL;
    private String _target;
    private Vector _words= new Vector();

    /**
     * Construct a link.
     * 
     * @param target The target of this link.
     */
    Link(String target) {
	_target = target;
    }

    /**
     * Get the target of this link.
     * 
     * @return The target of this link.
     */
    String getTarget() {
	return _target;
    }

    /**
     * Set the state (INITIAL, ACTIVE, or VISITED) of this Link.
     * 
     * @param state state of this link.
     */
    void setState(int state) {
	Log.assert(state == INITIAL || state == ACTIVE || state == VISITED,
		   "Illegal link state setting.");
	_state = state;
    }

    /**
     * Return the state (INITIAL, ACTIVE, or VISITED) of this link.
     * 
     * @return the state of this link.
     */
    int getState() {
	return _state;
    }

    /**
     * Make <tt>word</tt> part of this Link.
     * 
     * @param word Word to add.
     */
    void addWord(Word word) {
	_words.addElement(word);
    }

    /**
     * Called when the RichText scene graph is rendered.  Change the
     * foreground color, and set the "_inLink" flag so that subsequent
     * SingleWord objects will look like they're part of a link.
     * 
     * @param state graphics parameters.
     */
    void draw(GraphicsState state) {
	Color color = null;
	switch (_state) {
	case INITIAL:
	    color = state._linkColor;
	    break;
	case ACTIVE:
	    color = state._activeLinkColor;
	    break;
	case VISITED:
	    color = state._visitedLinkColor;
	    break;
	}

	Log.assert(color != null,
		   "RichText Link with illegal state value");

	state._graphics.setColor(color);
	state._inLink = true;
    }

    /**
     * See if (<tt>x</tt>, <tt>y</tt>) intersects this link.
     * 
     * @param x horizontal coordinate.
     * @param y vertical coordinate.
     * 
     * @return true if (<tt>x</tt>, <tt>y</tt>) intersects, false
     *         otherwise.
     */
    boolean intersects(int x, int y) {
	final int size = _words.size();
	for (int ii = 0; ii < size; ii++) {
	    Word word = (Word)_words.elementAt(ii);
	    if (word.intersects(x, y)) {
		return true;
	    }
	}

	return false;
    }
}
