//
// List.java
//
//	List of items within a RichText component.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui.richText;

import java.util.*;
import java.awt.*;

/**
 * List objects represent lists of paragraphs.  Each paragraph is
 * annotated with a bullet.
 */
abstract class List extends TextBlock {

    private Vector _blocks = new Vector();

    /**
     * Construct a List.  Parse paragraphs and other lists into our
     * Vector.
     * 
     * @param state Graphics parameters.
     * @param factory Gives us tokens.
     * @param indent indent level (used in draw).
     */
    List(GraphicsState state, TokenFactory factory, int indent) {
	super(indent);
	while (true) {
	    Token token = factory.getNextToken();
	    if (token.getType() == Token.TAG_TOKEN) {
		switch (token.getTagValue()) {
		case Token.LIST_ITEM:
		    {
			Paragraph paragraph =
			    new Paragraph(state, factory, _indent + 1);
			_blocks.addElement(paragraph);
		    }
		    break;
		case Token.UNORDERED_LIST:
		    {
			List list = new UnorderedList(state, factory,
						      _indent + 1);
			_blocks.addElement(list);
		    }
		    break;
		case Token.ORDERED_LIST:
		    {
			List list = new OrderedList(state, factory,
						    _indent + 1);
			_blocks.addElement(list);
		    }
		    break;
		case Token.END_ORDERED_LIST:
		case Token.END_UNORDERED_LIST:
		    return;
		default:
		    factory.unget();
		    return;
		}
	    }
	}
    }

    /**
     * Calculate our geometry.  The only thing we need to figure out
     * is our height, which is the aggregate height of our list items.
     * 
     * @param state Graphics parameters.
     * @param y Where we're supposed to start.
     * @param componentWidth Width of the rich text component.
     */
    void setGeometry(GraphicsState state, int y, int componentWidth) {
	_y = y;
	_height = 0;
	_x = state._margin.left + _indent * state._indentWidth;
	_width = componentWidth - _x - state._margin.right;

	Enumeration enum = _blocks.elements();
	while (enum.hasMoreElements()) {
	    TextBlock block = (TextBlock)enum.nextElement();
	    block.setGeometry(state, y, componentWidth);
	    _height += block.getHeight();
	    y += block.getHeight();
	}
    }

    /**
     * Draw a list of paragraphs and embedded lists.
     * 
     * @param state Graphics parameters.
     */
    void draw(GraphicsState state) {
	int y = _y;
	int itemNum = 1;
	
	Enumeration enum = _blocks.elements();
	while (enum.hasMoreElements()) {
	    TextBlock block = (TextBlock)enum.nextElement();
	    // We don't draw bullets in front of embedded lists; the
	    // embedded lists will take care of that themselves.
	    if (!(block instanceof List)) {
		drawBullet(state, y, itemNum++);
	    }
	    block.draw(state);
	    y += block.getHeight();
	}
    }

    /**
     * Subclasses override this to draw bullets next to items in the
     * list.
     * 
     * @param state Graphics parameters.
     * @param top Top of bullet.
     * @param itemNum Number of this item.
     */
    abstract void drawBullet(GraphicsState state, int top, int itemNum);
}
