//
// Paragraph.java
//
//	Paragraph within RichText.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui.richText;

import java.awt.*;
import java.util.*;
import com.sgi.sysadm.util.Log;

/**
 * Paragraph objects consist of words.  The setGeometry()
 * method does the word wrapping.
 */
class Paragraph extends TextBlock {

    CompoundWord _compoundWord = null;
    Link _link = null;
    Vector _words = new Vector();

    /**
     * Construct a Paragrap.  Parses a stream of Token objects into a
     * collection of Word objects.  Interspersed with word tokens will
     * be tokens for changing the font.  This gets complicated,
     * because font changes can happen within a "word" (in the written
     * language sense).  Since our SingleWord objects can each use
     * only one font, we'll need to construct a CompoundWord object
     * when we find a font change within a "word".
     * 
     * @param state graphics state.
     * @param factory Gives us tokens.
     * @param indent Indent level, used for paragraphs which are list
     *               items.  This gets used in draw().
     */
    Paragraph(GraphicsState state, TokenFactory factory, int indent) {
	super(indent);

	Font font = state._plainFont;
	_words.addElement(new FontChange(font));

	boolean done = false;
	boolean bold = false;
	boolean italic = false;
	while (!done) {
	    Token token = factory.getNextToken();
	    switch (token.getType()) {
	    case Token.WORD_TOKEN:
		SingleWord word = new SingleWord(token.getWordValue());
		addWord(word, token.getOKToBreakAfter());
		break;
	    case Token.SPACE_TOKEN:
		// This catches spaces after bold or italic tags.
		if (_compoundWord != null) {
		    _compoundWord.addWord(new SingleWord(" "));
		    _words.addElement(_compoundWord);
		    _compoundWord = null;
		} else if (_words.size() > 1) {
		    // Only add a space if we already have a word.  This
		    // prevents spaces at the beginnings of paragraphs.
		    // Since we always add a font change at the beginning,
		    // the check in the if statement above is for "1"
		    // rather than "0".
		    _words.addElement(new SingleWord(" "));
		}
		break;
	    case Token.TAG_TOKEN:
		// It's an HTML tag.  Look for font changes or the end of
		// this paragraph.
		switch (token.getTagValue()) {
		case Token.BOLD:
		    bold = true;
		    if (italic) {
			font = state._boldItalicFont;
		    } else {
			font = state._boldFont;
		    }
		    addWord(new FontChange(font), false);
		    break;
		case Token.END_BOLD:
		    bold = false;
		    if (italic) {
			font = state._italicFont;
		    } else {
			font = state._plainFont;
		    }
		    addWord(new FontChange(font), false);
		    break;
		case Token.ITALIC:
		    italic = true;
		    if (bold) {
			font = state._boldItalicFont;
		    } else {
			font = state._italicFont;
		    }
		    addWord(new FontChange(font), false);
		    break;
		case Token.END_ITALIC:
		    italic = false;
		    if (bold) {
			font = state._boldFont;
		    } else {
			font = state._plainFont;
		    }
		    addWord(new FontChange(font), false);
		    break;
		case Token.END_LIST_ITEM:
		case Token.END_PARAGRAPH:
		    done = true;
		    break;
		case Token.LINK:
		    _link = new Link(token.getTagParam("href"));
		    if (state._boldLinks && !bold) {
			if (italic) {
			    addWord(new FontChange(state._boldItalicFont),
				    false);
			} else {
			    addWord(new FontChange(state._boldFont), false);
			}
		    }
		    addWord(_link, false);
		    state.addLink(_link);
		    break;
		case Token.END_LINK:
		    _link = null;
		    if (state._boldLinks && !bold) {
			if (italic) {
			    addWord(new FontChange(state._italicFont), false);
			} else {
			    addWord(new FontChange(state._plainFont), false);
			}
		    }
		    addWord(new EndLink(), false);
		    break;
		case Token.FONT:
		    try {
			addWord(new FontColor(
			    Color.decode(token.getTagParam("color"))), false);
		    } catch (NumberFormatException ex) {
			System.out.println(ex.getMessage());
			ex.printStackTrace();
		    }
		    break;
		case Token.END_FONT:
		    addWord(new EndFontColor(), false);
		    break;
		default:
		    factory.unget();
		    done = true;
		    break;
		}
		break;
	    }
	}
	// If we had a compound word going when the paragraph ended, we
	// need to put it in our list.
	if (_compoundWord != null) {
	    _words.addElement(_compoundWord);
	}
    }
	
    /**
     * Set the geometry of all of the words we contain, calculating
     * our geometry as we go.  This is where we do word wrapping.
     * 
     * @param state Graphics parameters.
     * @param y Where we're supposed to start
     * @param componentWidth width of rich text component.
     */
    void setGeometry(GraphicsState state, int y, int componentWidth) {
	_y = y;
	_x = state._margin.left + _indent * state._indentWidth;
	_width = componentWidth - _x - state._margin.right;
	int textHeight = state._fontMetrics.getHeight();
	_height = textHeight;

	int x = _x;
	int maxX = componentWidth - state._margin.right;

	Enumeration enum = _words.elements();
	while (enum.hasMoreElements()) {
	    Word word = (Word)enum.nextElement();
	    word.setGeometry(state, x, y, textHeight);
	    int wordW = word.getWidth();
	    if (state._autoWrap &&
		x > state._margin.left && x + wordW > maxX) {
		x = _x;
		_height += textHeight + state._lineSpacing;
		y += textHeight + state._lineSpacing;
		word.setGeometry(state, x, y, textHeight);
	    }
	    x += wordW;
	    if (x > state._maxX) {
		state._maxX = x;
	    }
	}

	_height += state._lineSpacing;
    }

    /**
     * Draw a paragraph.  Draw the words, doing word wrapping as
     * appropriate.
     * 
     * @param state Drawing parameters.
     */
    void draw(GraphicsState state) {
	Enumeration enum = _words.elements();
	while (enum.hasMoreElements()) {
	    Word word = (Word)enum.nextElement();
	    word.draw(state);
	}
    }

    /**
     * Add a word to a paragraph.  This is complicated by the fact
     * that at any given time we can have a compound word going and/or
     * we can be processing a link.
     * 
     * @param word word to add.
     * @param okToBreakAfter Whether a line break is allowed after "word".
     */
    void addWord(Word word, boolean okToBreakAfter) {
	if (_compoundWord != null) {
	    // We've got a compound word going, so we'll add this
	    // new one to it.
	    _compoundWord.addWord(word);
	    if (okToBreakAfter) {
		// This new word is the last word in the compound
		// word, so we store the compound word in our list
		// and reset it.
		_words.addElement(_compoundWord);
		_compoundWord = null;
	    }		    
	} else if (okToBreakAfter) {
	    // We don't have a compound word going, and this word
	    // doesn't need to be in one, so we just store the
	    // word in our list.
	    _words.addElement(word);
	} else {
	    // We don't have a compound word going yet, and we
	    // can't do a line break at the end of the word we
	    // just got, so we'll start a compound word.
	    _compoundWord = new CompoundWord();
	    _compoundWord.addWord(word);
	}
	// Check the logic above.
	Log.assert(_compoundWord == null || !okToBreakAfter,
		   "Logic botch in Paragraph.addWord");

	if (_link != null && word != _link) {
	    _link.addWord(word);
	}
    }
}
