//
// RichTextArea.java
//
//	RichTextComponent subclass that adds handling of links as
//	glossary entries.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui.richText;

import java.util.*;
import javax.swing.*;
import com.sgi.sysadm.util.*;
import com.sgi.sysadm.ui.*;

/**
 * RichTextArea is a subclass of RichTextComponent that adds the
 * handling of link activations by popping up a glossary window.  The
 * target of each link that is clicked on is looked up in a
 * ResourceStack.  The value retrieved is displayed in a separate
 * glossary window.  If the resource lookup fails, the target of the
 * link is displayed as text in the glossary window.
 */
public class RichTextArea extends RichTextComponent {

    /**
     * <i>RichTextArea.glossary</i> is the name of the RichTextArea
     * that will be created when a link is clicked.  Thus,
     * RichTextComponent resources relative to
     * <i>RichTextArea.glossary</i> will apply to the RichTextArea that
     * displays glossary entries.
     */
    public static final String GLOSSARY_NAME = "RichTextArea.glossary";

    /**
     * <i>RichTextArea.glossary.title</i> is the name of the String
     * resource which specifies the title given to the glossary Frame.
     */
    public static final String GLOSSARY_TITLE = "RichTextArea.glossary.title";

    private static RFrame _glossaryFrame;
    private static RichTextArea _glossary;
    private ResourceStack _rs;

    /**
     * Construct a RichTextArea.
     * 
     * @param name Instance name of this RichTextArea.  Used for looking
     *             up resource settings.
     * @param rs ResourceStack containing resource settings for this
     *           instance of RichTextArea.  Unless
     *           setGlossaryResources is called, <tt>rs</tt> is also
     *           used for looking up glossary entries.
     */
    public RichTextArea(String name, ResourceStack rs) {
	super(name, rs);
	setGlossaryResources(rs);
	addLinkListener(new LinkListener() {
	    public void linkActivated(LinkEvent event) {
		handleLinkActivated(event);
	    }
	});
    }

    /**
     * Set the resource stack to be used for looking up glossary
     * entries.
     * 
     * @param rs The ResourceStack to be used for looking up glossary
     *           entries.
     */
    public void setGlossaryResources(ResourceStack rs) {
	_rs = rs;
    }

    /**
     * Called when the user clicks on a link.  Pop up the glossary
     * window.
     * 
     * @param event LinkEvent.
     */
    private void handleLinkActivated(LinkEvent event) {
	if (_glossaryFrame == null) {
	    _glossaryFrame = new RFrame();
	    _glossary = new RichTextArea(GLOSSARY_NAME, _rs);
	    _glossaryFrame.getContentPane().add(_glossary);
	    _glossaryFrame.setTitle(_rs.getString(GLOSSARY_TITLE, "Glossary"));
	}

	// Pass our resource stack the glossary window so that links
	// within glossary entries will work.
	_glossary.setGlossaryResources(_rs);
	String target = event.getTarget();
	_glossary.setText(_rs.getString(target, target));
 	_glossaryFrame.pack();
	_glossaryFrame.setVisible(true);
    }
}
