//
// RichTextComponent.java
//
//	RichText component.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui.richText;

import com.sgi.sysadm.ui.*;
import javax.swing.*;
import java.awt.*;
import java.awt.event.*;
import java.util.*;
import com.sgi.sysadm.util.*;

/////////////////////////////////////////////////////////////////////////
//
// Java classes that implement the RichText component:
// 
// Parsing:
//   Token
//   TokenFactory
//
// State:
//   GraphicState
//
// Geometry:
//   RectObj
//
// Words:
//   Word
//   SingleWord
//   CompoundWord
//   FontChange
//   Link
//   EndLink
//
// Text blocks:
//   TextBlock
//   Paragraph
//   List
//   LineBreak
//   Document
//
// The contents of the text area are represented by a structure of
// objects of these classes.  Document is the outer-most container; it
// creates a TokenFactory for parsing the component's text, and uses
// the factory to get Tokens.
//
// The result of Document's parsing is a collection of TextBlocks.
// TextBlock is the base class for Paragraph and List (as well as
// Document).
//
// A Paragraph consists of Word objects.  It's the Word objects
// that actually draw themselves.
//
// A List consists of Paragraph objects and embedded List objects.
// List objects let their lists and paragraphs do most of the drawing,
// but the lists do add bullets to the beginning of each list item.
//
// RectObj is used as the base class for any objects which have
// geometry.  RectObj is used instead of Rectangle because none of the
// methods on Rectangle are applicable to RichText internals.
//
// GraphicsState maintains drawing parameters for the other classes.
// 
// More details about the operation of each class and method can be
// found in the comments for those classes and methods.
//
/////////////////////////////////////////////////////////////////////////

/**
 * RichTextComponent is a view-only JComponent for displaying rich
 * text.  The format of the text displayed by RichText is a small
 * subset of HTML.  The following HTML tags are supported:
 * <ul>
 * <li>&lt;p&gt; marks the beginning of a paragraph.</li>
 * <li>&lt;ul&gt; marks the beginning of an unnumbered list.</li>
 * <li>&lt;/ul&gt; marks the end of an unnumbered list.</li>
 * <li>&lt;ol&gt; marks the beginning of an ordered list.</li>
 * <li>&lt;/ol&gt; marks the end of an ordered list.</li>
 * <li>&lt;li&gt;, &lt;/li&gt; delineates a list item.</li>
 * <li>&lt;b&gt;, &lt;/b&gt; delineates bold text.</li>
 * <li>&lt;i&gt;, &lt;/i&gt; delineates italicized text.</li>
 * <li>&lt;a href=<tt>target</tt>&gt;, &lt;/a&gt; delineates a link.</li>
 * <li>&lt;font color=<tt>color</tt>&gt;, &lt;/font&gt; delineates a change
 * in the color used to draw text</li>.
 * </ul>
 * <p>
 * The text to display can be specified by passing it to a
 * constructor, by calling the <tt>setText</tt> method, or via a
 * <tt>ResourceStack</tt>.
 * <p>
 * If constructed by specifying a <tt>ResourceStack</tt>,
 * RichTextComponent looks in the <tt>ResourceStack</tt> for many
 * resources which customize its appearance.  RichTextComponent looks
 * up each resource using two names.  The first is the <tt>name</tt>
 * argument that is passed to the RichTextComponent constructor
 * prepended to the resource name.  The second is the string
 * "RichText" prepended to the resource name.  If the resource cannot
 * be found using either name, RichTextComponent uses a default value.
 * <p>
 * To specify resources for all instances of RichTextComponent that
 * use a given ResourceStack, prepend the string "RichText" to the
 * names of the resource settings in the properties file.  To specify
 * resources for a specific RichTextComponent instance give that
 * instance a name and prepend that name to the resource settings in
 * the properties file.  The settings for the named RichTextComponent
 * instances will override settings for all RichTextComponent
 * instances.
 * <p>
 * Here is an example of a properties file containing resource
 * settings for RichTextComponents:
 * <pre>
 *   # All RichText components are to have a width of 600 points.
 *   RichText.width: 600
 * <p>
 *   # The text to be displayed in the RichText component named
 *   # "homeDirectory"
 *   homeDirectory.text: &lt;B&gt;home directory&lt;/B&gt;&lt;P&gt;\
 *   A directory in which you create and store your work.
 * <p>
 *   # Set the left margin of the userName component to 0.
 *   userName.marginLeft: 0
 * <p>
 *   # The userID RichText component should have a width of 500 points.
 *   userID.width: 500
 * </pre>
 * <p>
 * All resources which specify sizes on the screen (<tt>WIDTH</tt>,
 * <tt>MARGIN_LEFT</tt>, <tt>MARGIN_RIGHT</tt>, <tt>MARGIN_TOP</tt>,
 * <tt>MARGIN_BOTTOM</tt>, <tt>INDENT_WIDTH</tt>,
 * <tt>LINE_SPACING</tt>, <tt>BULLET_LEFT_OFFSET</tt>, and
 * <tt>BULLET_TOP_OFFSET</tt>) are specified in units of
 * points.  A point is 1/72 of an inch.  RichTextComponent divides the
 * pixel height of a test font by the font's point size to determine
 * how many pixels correspond to a point.  RichTextComponent does not
 * use java.awt.Toolkit's notion of screen resolution because
 * experience has shown that different Java implementations do not
 * have predictable relationships between screen resolutions and the
 * correspondence between a font's point and pixel sizes.
 * <p>
 * The full list of the resources which can be specified is described
 * in the Variable Index below.
 */
public class RichTextComponent extends JComponent 
                               implements DynamicSize, DynamicCursor {

    /**
     * The resource <i>RichText.text</i> or
     * <i>&lt;componentName&gt;.text</i> is a String containing the
     * HTML to be displayed.
     */
    public static final String TEXT = "text";

    /**
     * The resource <i>RichText.width</i> or
     * <i>&lt;componentName&gt;.width</i> is an Integer specifying the
     * preferred width of a RichTextComponent in points.  The
     * preferred height is determined dynamically depending on the
     * text being displayed.
     */
    public static final String WIDTH = "width";

    /**
     * The resource <i>RichText.marginLeft</i> or
     * <i>&lt;componentName&gt;.marginLeft</i> is an Integer
     * specifying the left margin in points.
     */
    public static final String MARGIN_LEFT = "marginLeft";

    /**
     * The resource <i>RichtText.marginRight</i> or
     * <i>&lt;componentName&gt;.marginRight</i> is an Integer
     * specifying the right margin in points.
     */
    public static final String MARGIN_RIGHT = "marginRight";

    /**
     * The resource <i>RichText.marginTop</i> or
     * <i>&lt;componentName&gt;.marginTop</i> is an Integer specifying
     * the top margin in points.
     */
    public static final String MARGIN_TOP = "marginTop";

    /**
     * The resource <i>RichText.marginBottom</i> or
     * <i>&lt;componentName&gt;.marginBottom</i> is an Integer
     * specifying the bottom margin in points.
     */
    public static final String MARGIN_BOTTOM = "marginBottom";

    /**
     * The resource <i>RichText.indentWidth</i> or
     * <i>&lt;componentName&gt;.indentWidth</i> is an Integer
     * specifying how many points to ident each level of list item.
     */
    public static final String INDENT_WIDTH = "indentWidth";

    /**
     * The resource <i>RichText.lineSpacing</i> or
     * <i>&lt;componentName&gt;.lineSpacing</i> is an Integer
     * specifying the number of points to insert between each line in
     * a paragraph.
     */
    public static final String LINE_SPACING = "lineSpacing";

    /**
     * The resource <i>RichText.textColor</i> or
     * <i>&lt;componentName&gt;.textColor</i> is an Integer specifying
     * the color to be used for normal text.
     */
    public static final String TEXT_COLOR = "textColor";

    /**
     * The resource <i>RichText.linkColor</i> or
     * <i>&lt;componentName&gt;.linkColor</i> is an Integer specifying
     * the color to be used in the text of a link.
     */
    public static final String LINK_COLOR = "linkColor";

    /**
     * The resource <i>RichText.activeLinkColor</i> or
     * <i>&lt;componentName&gt;.activeLinkColor</i> is an Integer
     * specifying the color to be used in the text of a link when the
     * user is holding the mouse button down over that link.
     */
    public static final String ACTIVE_LINK_COLOR = "activeLinkColor";

    /**
     * The resource <i>RichText.visitedLinkColor</i> or
     * <i>&lt;componentName&gt;.visitedLinkColor</i> is an Integer
     * specifying the color to be used in the text of a link after the
     * user has clicked on it.
     */
    public static final String VISITED_LINK_COLOR = "visitedLinkColor";

    /**
     * The resource <i>RichText.underlineLinks</i> or
     * <i>&lt;componentName&gt;.underlineLinks</i> is a Boolean
     * specifying whether or not links should be underlined.
     */
    public static final String UNDERLINE_LINKS = "underlineLinks";

    /**
     * The resource <i>RichText.boldLinks</i> or
     * <i>&lt;componentName&gt;.boldLinks</i> is a Boolean
     * specifying whether or not links should always be bold.
     */
    public static final String BOLD_LINKS = "boldLinks";

    /**
     * The resource <i>RichText.font</i> or
     * <i>&lt;componentName&gt;.font</i> is a String specifying the
     * Font to be used.  Only the name and point size should be
     * specified; RichTextComponent inserts different styles to create
     * different fonts for supporting the &lt;b&gt; and &lt;i&gt; tags.
     * <p>
     * For example, to specify that a 12 point sans-serif font should
     * be used for all RichTextComponents, specify the following
     * resource:
     * <pre>
     *   RichText.font = SansSerif-12
     * </pre>
     * @see setFont
     */
    public static final String FONT = "font";

    /**
     * The resource <i>RichText.bulletLeftOffset</i> or
     * <i>&lt;componentName&gt;.bulletLeftOffset</i> is an Integer
     * specifying the left offset of a bullet in a list item relative
     * to the paragraph it is in.  The list item itself will be
     * indented INDENT_WIDTH points, so BULLET_LEFT_OFFSET should be
     * smaller than INDENT_WIDTH.
     */
    public static final String BULLET_LEFT_OFFSET = "bulletLeftOffset";

    /**
     * The resource <i>RichText.bulletTopOffset</i> or
     * <i>&lt;componentName&gt;.bulletTopOffset</i> is an Integer
     * specifying padding at the top of a bullet.  RichTextComponent
     * calculates the location of the top of a bullet by centering the
     * bullet with respect to the ascent of the font being used and
     * then adding in <i>bulletTopOffset</i>.  This is necessary
     * because the ascent includes extra space which makes the bullet
     * look too high without the padding.
     */
    public static final String BULLET_TOP_OFFSET = "bulletTopOffset";

    /**
     * The resource <i>RichText.listNumberFormat</i> or 
     * <i>&lt;componentName&gt;.listNumberFormat</i> is a format
     * string used to display the numbers in ordered lists.
     */
    public static final String LIST_NUMBER_FORMAT = "listNumberFormat";

    /**
     * The resource <i>RichText.lineBreakScale</i> or
     * <i>&lt;componentName&gt;.lineBreakScale</i> is the float by
     * which the height of the current font is multiplied to determine
     * the vertical separation between paragraphs.
     */
    public static final String LINE_BREAK_SCALE = "lineBreakScale";

    /**
     * The resource <i>RichText.autoWrap</i> or
     * <i>&lt;componentName&gt;.autoWrap</i> is a Boolean specifying
     * whether RichText should wrap long lines in paragraphs.  This
     * affects the behavior of the
     * <a href=#getPreferredSize>getPreferredSize</a> method.  If
     * AUTO_WRAP is <tt>true</tt>, then <tt>getPreferredSize</tt>
     * will return a width corresponding to
     * the width of the longest line in the text.
     * @see #setAutoWrap
     * @see #getAutoWrap
     */
    public static final String AUTO_WRAP = "autoWrap";

    /**
     * <i>RichText</i> is the name prepended to resource names when
     * looking for resource settings that apply to all instances.
     * Such resources can be overridden by giving a specific instance
     * a name and prepending that name to the resource names.
     */
    public static final String CLASS_NAME = "RichText";

    /* Package */ static final String CLASS_NAME_DOT = "RichText.";

    /**
     * State of a link before it has been clicked.
     */
    public static final int LINK_STATE_INITIAL = 0;

    /**
     * State of a link while the mouse button is down.
     */
    public static final int LINK_STATE_ACTIVE = 1;

    /**
     * State of a link after it has been pressed.
     */
    public static final int LINK_STATE_VISITED = 2;

    // This is the square of the distance in pixels that the mouse can
    // move while held down over a link and still activate the link
    // when released.
    private static final int CLICK_THRESHHOLD_SQUARED = 16;

    private GraphicsState _state = new GraphicsState();
    private Document _doc;
    private boolean _handCursor = false;
    private Cursor _cursor;
    private Vector _linkListeners = new Vector();
    private Link _activeLink;
    private int _savedLinkState;
    private int _clickX;
    private int _clickY;
    private boolean _focusTraversable = false;

    /**
     * Construct a RichTextComponent with a name, a ResourceStack, and
     * a text string.  
     * 
     * @param name if non-null, name to use for resource lookup.
     * @param rs if non-null, stack for getting resources.
     * @param text if non-null, initial text to display
     */
    public RichTextComponent(String name, ResourceStack rs, String text) {
	_cursor = getCursor();

	addMouseListener(new MouseAdapter() {
	    public void mousePressed(MouseEvent event) {
		handleMousePressed(event);
	    }
	    public void mouseReleased(MouseEvent event) {
		handleMouseReleased(event);
	    }
	    public void mouseExited(MouseEvent event) {
		handleMouseExited();
	    }
	});

	addMouseMotionListener(new MouseMotionAdapter() {
	    public void mouseMoved(MouseEvent event) {
		handleMouseMoved(event);
	    }
	    public void mouseDragged(MouseEvent event) {
		handleMouseDragged(event);
	    }
	});
	_state.setValues(name, rs);
	if (text != null) {
	    setText(text);
	}
    }

    /**
     * Construct a RichTextComponent component.  All settings get their default
     * values.  Use setText to set the text that is to be displayed.
     */
    public RichTextComponent() {
	this(null, null, null);
    }

    /**
     * Construct a RichTextComponent with a text string.  All settings
     * get their default values.  <tt>text</tt> is displayed.
     * 
     * @param text The text to display.
     */
    public RichTextComponent(String text) {
	this(null, null, text);
    }

    /**
     * Construct a RichTextComponent with a name and a ResourceStack.
     * Settings, including the text to be displayed, are taken from
     * <tt>rs</tt>.
     * 
     * @param name Name of this instance.  Used for retrieving
     *             resources from <tt>rs</tt>.
     * @param rs Resource stack containing settings for this
     *           RichTextComponent instance.
     */
    public RichTextComponent(String name, ResourceStack rs) {
	this(name, rs, null);
    }

    /**
     * Construct a RichTextComponent with a ResourceStack.  Setting
     * will be taken from the default values found in <tt>rs</tt>.
     * The text to display comes from the resource <a href=#TEXT>TEXT</a>.
     *
     * @param rs Resource stack containing settings for this
     *           RichTextComponent instance.
     */
    public RichTextComponent(ResourceStack rs) {
	this(null, rs, null);
    }

    /**
     * Set the text to be displayed.
     * 
     * @param text The text to be displayed.
     */
    public void setText(String text) {
	_state._text = text;
	_state.removeAllLinks();
	_doc = null;
	// Calling repaint will eventually result in a call to
	// paintComponent.
	invalidate();
	repaint();
    }

    /**
     * Get the text being displayed.
     * 
     * @return The text being displayed.  This can be null.
     */
    public String getText() {
	return _state._text;
    }

    /**
     * Set the font that RichText uses to display text.  Font is
     * specified using <tt>family</tt> and <tt>size</tt> rather than
     * an actual Font because RichTextComponent uses several
     * different styles of fonts with the same family and size.
     * 
     * @param family family of font to use.
     * @param size size of font to use.
     * @see FONT
     */
    public void setFont(String family, int size) {
	_state._plainFont = SysUtil.loadFont(family, Font.PLAIN, size);
	_state._boldFont = SysUtil.loadFont(family, Font.BOLD, size);
	_state._italicFont = SysUtil.loadFont(family, Font.ITALIC, size);
	_state._boldItalicFont = SysUtil.loadFont(family, 
						  Font.ITALIC | Font.BOLD, 
						  size);
	_doc = null;
    }
    
    /**
     * Set RichText's wrapping behavior.  If <tt>autoWrap</tt> is
     * true, then long lines will be wrapped.
     * 
     * @param autoWrap Whether to wrap long lines.
     * @see #AUTO_WRAP
     */
    public void setAutoWrap(boolean autoWrap) {
	_state._autoWrap = autoWrap;
	_doc = null;
    }

    /**
     * Get the value of the auto wrap flag.
     * 
     * @return true if long lines will be wrapped, false otherwise.
     * @see AUTO_WRAP
     */
    public boolean getAutoWrap() {
	return _state._autoWrap;
    }

    /**
     * Set the state of a link in the document.  <tt>linkNumber</tt>
     * specifies the number of the link to set; the links in a
     * document are numbered starting from 0.
     * <p>
     * Note that setting a link's state to LINK_STATE_ACTIVE will
     * will prevent RichTextComponent from notifying LinkListeners
     * when a link is clicked on.
     * 
     * @param linkNumber Number of link to set.
     * @param state State of link to set.  Should be one of the
     *              constants LINK_STATE_INITIAL, LINK_STATE_ACTIVE,
     *              or LINK_STATE_VISITED.
     */
    public void setLinkState(int linkNumber, int state) {
	createDocument();
	Link link = (Link)_state._links.elementAt(linkNumber);
	link.setState(state);
    }

    /**
     * Returns the link state of a specified link.  <tt>linkNumber</tt>
     * specifies the number of the link to get; the links in a
     * document are numbered starting from 0. 
     *
     * @param linkNumber Number of link to get.
     * @return State of link.  Will be one of the
     *         constants LINK_STATE_INITIAL, LINK_STATE_ACTIVE,
     *         or LINK_STATE_VISITED.     
     */
    public int getLinkState(int linkNumber) {
	return ((Link)_state._links.elementAt(linkNumber)).getState();
    }


    /**
     * Returns the link target (URL) of a specified link.  <tt>linkNumber</tt>
     * specifies the number of the link to get; the links in a
     * document are numbered starting from 0. 
     *
     * @param linkNumber Number of link to get.
     * @return Target of link.
     */
    public String getLinkTarget(int linkNumber) {
	return TokenFactory.unescape(
	    ((Link)_state._links.elementAt(linkNumber)).getTarget());
    }
    
    
    /**
     * Determine the number of links in a document.
     * 
     * @return The number of links in our document.
     */
    public int getNumLinks() {
	createDocument();
	return _state._links.size();
    }

    /**
     * Set the margins in pixels.  Margins are separate from and in
     * addition to insets.
     * 
     * @param margins new margins.
     */
    public void setMargins(Insets margins) {
	_state._marginLeft = margins.left;
	_state._marginRight = margins.right;
	_state._marginTop = margins.top;
	_state._marginBottom = margins.bottom;
	if (_doc != null) {
	    _state.setGraphics(getGraphics());
	    _state.setInsets(getInsets());
	    _doc.setGeometry(_state, 0, getSize().width);
	    repaint();
	}
    }

    /**
     * Utility method for removing all of the tags from a
     * RichTextComponent's text string.
     * 
     * @param richTextString String from which tags are to be
     *                       removed.  This could be the value
     *                       returned by getText.
     * 
     * @return <tt>richTextString</tt> with tags removed.
     */
    public static String removeTags(String richTextString) {
	if (richTextString == null) {
	    return null;
	}
	int length = richTextString.length();
	StringBuffer tagLess = new StringBuffer(length);
	boolean inTag = false;
	for (int i = 0; i < length; i++) {
	    char ch = richTextString.charAt(i);
	    if (ch == '<') {
		inTag = true;
	    } else if (ch == '>') {
		inTag = false;
	    } else if (!inTag) {
		tagLess.append(ch);
	    }
	}
	return tagLess.toString();
    }

    /**
     * Replace any characters in <tt>str</tt> that have special
     * meaning to RichTextComponent with escape sequences, so that
     * when the returned string is displayed in a RichTextComponent
     * the special characters will be displayed.
     * 
     * @param str String whose special characters are to be replaced.
     * 
     * @return A quoted version of <tt>str</tt>.
     */
    public static String quoteString(String str) {
	if (str == null) {
	    return null;
	}
	return TokenFactory.escape(str);
    }

    /**
     * Add a listener to be notified when the user clicks on a link.
     * 
     * @param listener The listener to add.
     */
    public void addLinkListener(LinkListener listener) {
	_linkListeners.addElement(listener);
    }

    /**
     * Remove a listener from our link listeners.
     * 
     * @param listener The listener to remove.
     */
    public void removeLinkListener(LinkListener listener) {
	_linkListeners.removeElement(listener);
    }

    /**
     * Called when it's time to paint ourselves.
     * 
     * @param graphics Graphics to be used for painting.
     */
    public void paintComponent(Graphics graphics) {
	if (isOpaque()) {
	    Rectangle rect = graphics.getClipBounds();
	    Color color = getBackground();
	    Log.assert(color != null,
		       "Opaque RichTextComponent with no background color");
	    graphics.setColor(color);
	    graphics.fillRect(rect.x, rect.y, rect.width, rect.height);
	}
	_state.setGraphics(graphics);
	createDocument();
	_state.setInsets(getInsets());
	int width = getSize().width;
	if (width != _doc.getComponentWidth()) {
	    _doc.setGeometry(_state, 0, width);
	}
	_doc.draw(_state);
    }

    /**
     * Called when our size changes.  Adjust text accordingly.
     * 
     * @param x new horizontal position.
     * @param y new vertical position.
     * @param width new width.
     * @param height new height.
     */
    public void setBounds(int x, int y, int width, int height) {
	super.setBounds(x, y, width, height);

	if (_doc != null) {
	    _state.setGraphics(getGraphics());
	    _state.setInsets(getInsets());
	    _doc.setGeometry(_state, 0, width);
	}
    }

    /**
     * Calculate our preferred size.  Our width comes from the
     * <i>width</i> resource setting, and the height is calculated
     * based on the text which is being displayed.
     * <p>
     * If <a href=#getAutoWrap>getAutoWrap</a> returns <tt>true</tt>,
     * the width returned is the width of the widest line + margins
     * that fits within the value of the <i>width</i> resource
     * setting.  This means that the actual size may be less than the
     * <i>width</i> resource setting.
     * <p>
     * If <a href=#getAutoWrap>getAutoWrap</a> returns <tt>false</tt>,
     * the width returned is the width of the widest line + margins.
     * 
     * @return Our preferred size.
     */
    public Dimension getPreferredSize() {
	Graphics graphics = getGraphics();
	if (graphics == null) {
	    return super.getPreferredSize();
	}
	_state.setGraphics(graphics);
	createDocument();
	_state.setInsets(getInsets());
	_doc.setGeometry(_state, 0, _state._width);

	return new Dimension(_doc.getWidth(), _doc.getHeight());
    }

    /**
     * Calculate our preferred height given a particular width.  Can
     * be used by LayoutManagers that have special knowledge of the
     * fact that RichTextComponent's height depends on its width.
     * 
     * @param width A proposed width for this component.
     * 
     * @return the height this component would like to be given a
     *         width of <tt>width</tt>.
     */
    public int getPreferredHeight(int width) {
	int saveWidth = _state._width;
	_state._width = width;
	int height = getPreferredSize().height;
	_state._width = saveWidth;
	return height;
    }

    /**
     * Control whether or not this RichTextComponent is focus
     * traversable.  RichTextComponent does not give any visual
     * indication when it has focus.
     * 
     * @param focusTraversable whether or not this RichTextComponent
     *                         should be focus traversable.
     */
    public void setFocusTraversable(boolean focusTraversable) {
	_focusTraversable = focusTraversable;
    }

    /**
     * Controls whether or not this RichTextComponent can get focus
     * when the user presses the TAB key.  The return value of this
     * method can be controlled by calling <tt>setFocusTraversable</bb>
     * 
     * @return true if this RichTextComponent is focus traversable.
     */
    public boolean isFocusTraversable() {
	return _focusTraversable;
    }

    /**
     * Called when a mouse button is pressed.  If the mouse is over a
     * link, change its color.
     * 
     * @param event mouse event.
     */
    private void handleMousePressed(MouseEvent event) {
	if (!isEnabled()) {
	    return;
	}
    
	_clickX = event.getX();
	_clickY = event.getY();
	Link link = _state.getLink(event.getX(), event.getY());
	if (link != null && link != _activeLink
	    && link.getState() != Link.ACTIVE) {
	    _activeLink = link;
	    _savedLinkState = link.getState();
	    link.setState(Link.ACTIVE);
	    _state.setGraphics(getGraphics());
	    _doc.draw(_state);
	    // This is necessary when running on Windows; otherwise
	    // only part of the label gets changed.
	    Toolkit.getDefaultToolkit().sync();
	}
    }

    /**
     * Called when the mouse is released.  If it's in the same place
     * it was when it was pressed, and it's over a link, activate that
     * link.
     * 
     * @param event mouse event.
     */
    private void handleMouseReleased(MouseEvent event) {
	if (!isEnabled()) {
	    return;
	}
	if (_activeLink == null) {
	    return;
	}

	// The reason I look for mouseReleased and check the click
	// position rather than using mouseClicked is that I want to
	// change the color of the text back to what it was before if
	// the user moved the mouse and to the VISITED color if the
	// user did not move the mouse.  mouseReleased gets called
	// before mouseClicked, so trying to use mouseReleased to
	// restore the color and mouseClicked to change the color to
	// VISITED results in flashing.
	//
	// Another reason is that mouseClicked on both Irix and NT
	// requires that the mouse does not move *at all* between the
	// press and release.  In order to make clicks easier for the
	// user, we give them a little bit of room for error.
	//	--rogerc
	if (inClickThreshhold(event.getX(), event.getY())) {
	    LinkEvent linkEvent = new LinkEvent(
		this, TokenFactory.unescape(_activeLink.getTarget()));
	    Enumeration enum = _linkListeners.elements();
	    while (enum.hasMoreElements()) {
		LinkListener listener = (LinkListener)enum.nextElement();
		listener.linkActivated(linkEvent);
	    }
	    _savedLinkState = Link.VISITED;
	}

	_activeLink.setState(_savedLinkState);
	_activeLink = null;

	_state.setGraphics(getGraphics());
	_doc.draw(_state);
    }

    /**
     * Set the cursor back to normal when the mouse leaves the
     * component
     */
    private void handleMouseExited() {
	setCursor(_cursor);
	_handCursor = false;
    }

    /**
     * Get the cursor to display when the mouse is over the point
     * <tt>x</tt>, <tt>y</tt>.
     * 
     * @param x horizontal coordinate within this RichTextComponent.
     * @param y vertical coordinate within this RichTextComponent.
     * @see com.sgi.sysadm.ui.DynamicCursor
     */
    public Cursor getCursorAt(int x, int y) {
	Link link = _state.getLink(x, y);
	if (link == null) {
	    return null;
	} else {
	    return Cursor.getPredefinedCursor(Cursor.HAND_CURSOR);
	}
    }
    
    /**
     * Change the mouse cursor to a hand when the mouse is over a
     * link.
     * 
     * @param event mouse event.
     */
    private void handleMouseMoved(MouseEvent event) {
	if (!isEnabled()) {
	    if (_handCursor == true) {
		setCursor(_cursor);
	    }
	    return;
	}
	Link link = _state.getLink(event.getX(), event.getY());
	if (link != null && !_handCursor) {
	    setCursor(Cursor.getPredefinedCursor(Cursor.HAND_CURSOR));
	    _handCursor = true;
	} else if (link == null && _handCursor) {
	    setCursor(_cursor);
	    _handCursor = false;
	}
    }

    /**
     * Called when the mouse is dragged.  Deactivate the link if the
     * mouse moves too far.
     * 
     * @param event mouse event.
     */
    private void handleMouseDragged(MouseEvent event) {
	if (!isEnabled()) {
	    return;
	}
	if (_activeLink != null
	    && !inClickThreshhold(event.getX(), event.getY())) {
	    setCursor(_cursor);
	    _handCursor = false;
	    _activeLink.setState(_savedLinkState);
	    _activeLink = null;
	    _state.setGraphics(getGraphics());
	    _doc.draw(_state);
	}
    }

    /**
     * Determine whether an (x, y) position is close enough to the
     * mouse down event we got to constitute a click.
     * 
     * @param x horizontal position.
     * @param y vertical position.
     * 
     * @return true if it's close enough for a click, false otherwise.
     */
    private boolean inClickThreshhold(int x, int y) {
	int xdelta = _clickX - x;
	int ydelta = _clickY - y;
	return xdelta * xdelta + ydelta * ydelta < CLICK_THRESHHOLD_SQUARED;
    }

    /**
     * Create our document if it does not exist yet.
     */
    private void createDocument() {
	if (_doc == null) {
	    _doc = new Document(_state);
	}
    }
}
