//
// Token.java
//
//	RichText Token.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui.richText;
import com.sgi.sysadm.util.*;
import java.util.*;

/**
 * Token holds information about a semantic object that was parsed
 * from the component's "text" resource.  The "text" resource is just
 * a string of characters, whereas a Token is a higher-level entity
 * which can be an HTML tag, a word, or a space.
 * <p>
 * Token has no smarts of its own; TokenFactory does all the grungy
 * parsing.
 */
class Token {

    // Tokens.
    static final int TAG_TOKEN = 0;
    static final int WORD_TOKEN = 1;
    static final int SPACE_TOKEN = 2;
    static final int INVALID_TOKEN = 3;

    // Tags.
    static final int PARAGRAPH = 0;
    static final int END_PARAGRAPH = 1;
    static final int BOLD = 2;
    static final int END_BOLD = 3;
    static final int ITALIC = 4;
    static final int END_ITALIC = 5;
    static final int UNORDERED_LIST = 6;
    static final int END_UNORDERED_LIST = 7;
    static final int ORDERED_LIST = 8;
    static final int END_ORDERED_LIST = 9;
    static final int LIST_ITEM = 10;
    static final int END_LIST_ITEM = 11;
    static final int LINK = 12;
    static final int END_LINK = 13;
    static final int FONT = 14;
    static final int END_FONT = 15;
    static final int BREAK = 16;
    static final int END_DOCUMENT = 17;
    static final int INVALID = 18;

    //
    // A map from HTML tags to Tags.
    //
    static String TAG_MAP[] = {
	"P",			// Paragraph
	"/P",			// EndParagraph
	"B",			// Bold
	"/B",			// EndBold
	"I",			// Italic
	"/I",			// EndItalic
	"UL",			// UnorderedList
	"/UL",			// EndUnorderedList
	"OL",			// OrderedList
	"/OL",			// EndOrderedList
	"LI",			// ListItem
	"/LI",			// EndListItem
	"A",			// Link
	"/A",			// EndLink
	"FONT",			// Font
	"/FONT",		// EndFont
	"BR",			// Break
	null,
    };

    private int _type = INVALID_TOKEN;
    private int _tag = INVALID;
    private String _string;
    private String _target;
    private boolean _okToBreakAfter = false;
    private Hashtable _tagParams = new Hashtable();

    /**
     * Get the type of this Token.
     * 
     * @return The type of this Token.
     */
    int getType() {
	return _type;
    }

    /**
     * Get the tag of this Token.
     * 
     * @return 
     */
    int getTagValue() {
	Log.assert(_type == TAG_TOKEN,
		   "Attempt to get the tag of non-TAG Token");
	return _tag;
    }

    /**
     * Get the word of this token.
     * 
     * @return The word of this token.
     */
    String getWordValue() {
	Log.assert(_type == WORD_TOKEN,
		   "Attempt to get the word from a non-WORD Token");
	return _string;
    }

    /**
     * Set the type of this token.
     * 
     * @param token Type of this token.
     */
    void setType(int token) {
	_type = token;
    }

    /**
     * Set the tag of this token.
     * 
     * @param tag tag of this token.
     */
    void setTagValue(int tag) {
	Log.assert(_type == TAG_TOKEN,
		   "Attempt to set the tag of non-TAG Token");
	_tag = tag;
    }

    /**
     * Set the word of this token.
     * 
     * @param string word of this token.
     */
    void setWordValue(String string) {
	Log.assert(_type == WORD_TOKEN,
		   "Attempt to set the word of a non-WORD Token");
	_string = string;
    }

    /**
     * Set a parameter from an HTML tag.
     * 
     * @param key key of parameter.  key is converted to lower case.
     * @param value value of parameter.
     */
    void setTagParam(String key, String value) {
	Log.assert(_type == TAG_TOKEN,
		   "Attempt to set param of a non-TAG Token");
	_tagParams.put(key.toLowerCase(), value);
    }

    /**
     * Get a parameter from an HTML tag.
     * 
     * @param key key of parameter.
     * 
     * @return value of parameter.
     */
    String getTagParam(String key) {
	Log.assert(_type == TAG_TOKEN,
		   "Attempt to get param of a non-TAG Token");
  	return (String)_tagParams.get(key.toLowerCase());
    }

    /**
     * Reset all HTML tag parameters.
     */
    void clearTagParams() {
	_tagParams.clear();
    }

    /**
     * Set the okToBreakAfter flag for this token.
     * 
     * @param okToBreakAfter Whether it's OK to insert a line break
     *                       after this token.
     */
    void setOKToBreakAfter(boolean okToBreakAfter) {
	_okToBreakAfter = okToBreakAfter;
    }

    /**
     * Get the okToBreakAfter flag for this token.
     * 
     * @return true if it's OK to insert a line break after this
     *         token, false otherwise.
     */
    boolean getOKToBreakAfter() {
	return _okToBreakAfter;
    }
}
