//
// AttributeAbstractButtonBinder.java
//
//	Bind an Attribute to a button in a ButtonGroup.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui.taskData;
import javax.swing.*;
import javax.swing.event.*;
import com.sgi.sysadm.util.*;

/**
 * AttributeAbstractButtonBinder binds an AbstractButton in a ButtonGroup
 * to a piece of taskData.  Whenever the button's state is set to
 * true, the corresponding piece of taskData is set to the value
 * passed to the constructor or bind method.  Whenever the taskData
 * changes to the value passed to the constructor or bind method, the
 * state of the corresponding button is set to true.
 * <p>
 * To use AttributeAbstractButtonBinder, the caller adds the buttons
 * to be bound to a button group, and then creates an
 * AttributeAbstractButtonBinder, and passes the attribute name and that
 * AttributeAbstractButtonBinder to TaskData.addTaskDataBinder.
 * Each button must be bound separately.  For example:
 * <pre>
 *	JRadioButton roButton = new RRadioButton("Read Only");
 *	JRadioButton rwButton = new RRadioButton("Read/Write");
 *	ButtonGroup group = new ButtonGroup();
 *	group.add(roButton);
 *	group.add(rwButton);
 *	_taskData.addTaskDataBinder(AbstractRhinoExampleTask.WRITABLE,
 *	    new AttributeAbstractButtonBinder(roButton,
 *		new Attribute(AbstractRhinoExampleTask.WRITABLE, false)));
 *	_taskData.addTaskDataBinder(AbstractRhinoExampleTask.WRITABLE,
 *	    new AttributeAbstractButtonBinder(rwButton,
 *		new Attribute(AbstractRhinoExampleTask.WRITABLE, true)));
 * </pre>
 */
public class AttributeAbstractButtonBinder extends TaskDataBinder
                                        implements ChangeListener {
    private Attribute _attr;
    private AbstractButton _button;

    /**
     * Construct an AttributeAbstractButtonBinder.
     *
     * @param button Button to bind to <tt>attr</tt>.
     * @param attr Attribute to bind to <tt>button</tt>.
     */
    public AttributeAbstractButtonBinder(AbstractButton button,
					 Attribute attr) {
	_attr = attr;
	_button = button;
	_button.addChangeListener(this);
    }

    /**
     * Bind a Button to a piece of TaskData.
     *
     * @param taskData TaskData to bind.
     * @param name name of Attribute to bind.
     * @param button Button to bind to <tt>attr</tt>.
     * @param attr Attribute to bind to <tt>button</tt>.
     *
     * @deprecated Use <CODE>taskData.addTaskDataBinder(name,
     * new AttributeAbstractButtonBinder(button, attr))</CODE>
     */
    static public void bind(TaskData taskData, String name,
			    AbstractButton button,
			    Attribute attr) {
	Log.assert(name.equals(attr.getKey()),
		   "Attribute must match taskData being bound");
	taskData.addTaskDataBinder(
	    name, new AttributeAbstractButtonBinder(button, attr));
    }

    /**
     * Called when Attribute we're binding changes.  Select our button
     * if Attribute matches.
     *
     * @param event TaskDataEvent.
     */
    public void taskDataChanged(TaskDataEvent event) {
	if (getTaskData().getAttr(getName()).equals(_attr)) {
	    _button.setSelected(true);
	}
    }

    /**
     * Called when the state of the Button we're binding changes.  Set
     * our Attribute in TaskData if the Button is selected.
     *
     * @param event ChangeEvent.
     */
    public void stateChanged(ChangeEvent event) {
	if (_button.isSelected()) {
	    getTaskData().setAttr(_attr);
	}
    }
}
