//
// BooleanAbstractButtonBinder.java
//
//	Bind Boolean task data to a swing Abstract Button component.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui.taskData;

import javax.swing.*;
import javax.swing.event.*;

/**
 * BooleanAbstractButtonBinder keeps a piece of Boolean TaskData
 * synchronized with an AbstractButton (such as JCheckBox).  If
 * TaskData changes then BooleanAbstractButtonBinder changes the
 * AbstractButton, and if the AbstractButton changes then
 * BooleanAbstractButtonBinder changes the TaskData.
 */
public class BooleanAbstractButtonBinder extends TaskDataBinder
implements ChangeListener {
    private AbstractButton _button;
    private boolean _value;

    /**
     * Construct a BooleanAbstractButtonBinder.  Add this as a
     * ChangeListener to <tt>button</tt> to keep it in sync with
     * TaskData.
     *
     * @param button AbstractButton to keep synchronized.
     */
    public BooleanAbstractButtonBinder(AbstractButton button) {
	this(button, true);
    }

    /**
     * Construct a BooleanAbstractButtonBinder.  Add this as a
     * ChangeListener to <tt>button</tt> to keep it in sync with
     * TaskData.
     *
     * @param button AbstractButton to keep synchronized.
     * @param value Value corresponding to set state of
     *              <tt>button</tt>.  If <tt>value</tt> is false, the
     *              <tt>button</tt> will be selected when the task
     *              data is false.
     */
    public BooleanAbstractButtonBinder(AbstractButton button,
				       boolean value) {
	_button = button;
	_button.addChangeListener(this);
	_value = value;
    }

    /**
     * Bind the Attribute <tt>name</tt> in <tt>taskData</tt> to
     * <tt>button</tt>, so that when <tt>button</tt> changes the
     * Attribute is updated and vice versa.
     *
     * @param taskData TaskData to bind.
     * @param name name of Attribute to bind.
     * @param button AbstractButton to bind.
     *
     * @deprecated Use <CODE>taskData.addTaskDataBinder(name,
     * new BooleanAbstractButtonBinder(button))</CODE>
     */
    static public void bind(TaskData taskData, String name,
			    AbstractButton button) {
	taskData.addTaskDataBinder(
	    name, new BooleanAbstractButtonBinder(button));
    }

    /**
     * Bind the Attribute <tt>name</tt> in <tt>taskData</tt> to
     * <tt>button</tt>, so that when <tt>button</tt> changes the
     * Attribute is updated and vice versa.
     *
     * @param taskData TaskData to bind.
     * @param name name of Attribute to bind.
     * @param button AbstractButton to bind.
     * @param value Value corresponding to set state of <tt>button</tt>.
     *
     * @deprecated Use <CODE>taskData.addTaskDataBinder(name,
     * new BooleanAbstractButtonBinder(button, value))</CODE>
     */
    static public void bind(TaskData taskData, String name,
			    AbstractButton button, boolean value) {
	taskData.addTaskDataBinder(
	    name, new BooleanAbstractButtonBinder(button, value));
    }

    /**
     * Called when our task data changes.  Update our button.
     *
     * @param event TaskDataEvent.
     */
    public void taskDataChanged(TaskDataEvent event) {
	boolean state = event.getTaskData().getBoolean(getName()) == _value;
	if (state != _button.isSelected()) {
	    _button.setSelected(state);
	}
    }

    /**
     * Called when the selection state of our button changes.  Update
     * our task data.
     *
     * @param event ChangeEvent.
     */
    public void stateChanged(ChangeEvent event) {
	getTaskData().setBoolean(getName(),
				 _button.isSelected() ? _value : !_value);
    }
}
