//
// LongJComboBoxBinder.java
//
//	Keep long task data in sync with the items in a JComboBox.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui.taskData;

import javax.swing.*;
import java.awt.event.*;

/**
 * LongJComboBoxBinder is used to keep long task data synchronized
 * with the items in a JComboBox.
 * <p>
 * The idea is that a binder should be created for each item in a
 * JComboBox.  Each binder is created with a "value" different from
 * the other binders for the JComboBox.  When the task data changes,
 * each binder will be notified, and the one whose "value" matches
 * the task data will make its item the current item.  Conversely,
 * when one of the items is selected, its binder updates the task
 * data.  For example:
 * <pre>
 *	JComboBox typeBox = new JComboBox();
 *	String stringItem = rs.getString("DataPage.stringItem");
 *	typeBox.addItem(stringItem);
 *	taskData.addTaskDataBinder(RESKEY_DATATYPE_INT,
 *	    new LongJComboBoxBinder(typeBox, stringItem, STR_TYPE));
 *	String intItem = rs.getString("DataPage.intItem");
 *	typeBox.addItem(intItem);
 *	taskData.addTaskDataBinder(RESKEY_DATATYPE_INT,
 *	    new LongJComboBoxBinder(typeBox, intItem, INT_TYPE));
 * </pre>
 */
public class LongJComboBoxBinder extends TaskDataBinder {

    private JComboBox _box;
    private Object _item;
    private long _value;

    /**
     * Construct a LongJComboBoxBinder.  Add an ItemListener
     * to <tt>box</tt> to keep it in sync with TaskData.  If our task
     * data is set to <tt>value</tt>, we make <tt>item</tt> the
     * selected item in <tt>box</tt>.  When <tt>item</tt> is selected,
     * we set our task data to <tt>value</tt>.
     *
     * @param box JComboBox to keep synchronized.
     * @param item Item to select when task data has <tt>value</tt>
     * @param value Value to set when <tt>item</tt> is selected.
     */
    public LongJComboBoxBinder(JComboBox box, Object item,
			       long value) {
	_box = box;
	_item = item;
	_value = value;
	box.addItemListener(new ItemListener() {
	    public void itemStateChanged(ItemEvent event) {
		String name = getName();
		TaskData taskData = getTaskData();
		if (event.getStateChange() == ItemEvent.SELECTED
		    && event.getItem() == _item
		    && taskData.getLong(name) != _value) {
		    taskData.setLong(name, _value);
		}
	    }
	});
    }

    /**
     * Bind the Attribute <tt>name</tt> in <tt>taskData</tt> to
     * <tt>box</tt>, so that when <tt>item</tt> is selected in
     * <tt>box</tt> the Attribute is changed to <tt>value</tt>, and
     * vice versa.
     *
     * @param taskData TaskData to bind.
     * @param name name of Attribute to bind.
     * @param box JComboBox to bind.
     * @param item Item to select when Attribute takes on <tt>value</tt>.
     * @param value value to set Attribute to when <item> is selected.
     *
     * @deprecated Use <CODE>taskData.addTaskDataBinder(name,
     * new LongJComboBoxBinder(box, item, value))</CODE>
     */
    static public void bind(TaskData taskData, String name,
			    JComboBox box, Object item, long value) {
	taskData.addTaskDataBinder(
	    name, new LongJComboBoxBinder(box, item, value));
    }

    /**
     * Called when our task data changes.  Select our item the task
     * data has our value.
     *
     * @param event TaskDataEvent.
     */
    public void taskDataChanged(TaskDataEvent event) {
	TaskData taskData = event.getTaskData();
	String name = event.getName();
	if (taskData.getLong(name) == _value
	    && _box.getSelectedItem() != _item) {
	    _box.setSelectedItem(_item);
	    _box.repaint();
	}
    }
}
