//
// ReasonItemFinderBinder.java
//
//	Bind Reason String task data to an ItemFinder
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui.taskData;

import com.sgi.sysadm.ui.event.*;
import com.sgi.sysadm.category.*;
import com.sgi.sysadm.ui.*;
import com.sgi.sysadm.util.*;

/**
 * ReasonItemFinderBinder keeps a piece of String TaskData
 * synchronized with an ItemFinder. When the
 * ItemFinder changes then ReasonItemFinderBinder changes the
 * TaskData to reflect the reason that the item was rejected by the
 * filter.  The task data will be set to the empty string if there is
 * no reason given.  If multiple reasons for failure are returned from
 * the ItemFinder, only the first will be used.
 */
public class ReasonItemFinderBinder extends TaskDataBinder
                                      implements ItemFinderListener
{
    private String _noSuchItem;
    private String _noItemChosen;
    private String _reasonStr;
    private ItemFinder _finder;

    /**
     * Construct a ReasonItemFinderBinder.
     *
     * @param finder ItemFinder to keep synchronized.
     * @param noSuchItem The string to put into the task data if the
     *                   reason returned by the ItemFinder is
     *                   NO_SUCH_ITEM
     * @param noItemChosen The string to put into the task data if the
     *                     reason returned by the ItemFinder in
     *                     NO_ITEM_CHOSEN
     * @see com.sgi.sysadm.ui.ItemFinderState
     */
    public ReasonItemFinderBinder(ItemFinder finder,
				  String noSuchItem,
				  String noItemChosen) {
	_finder = finder;
	_noSuchItem = noSuchItem;
	_noItemChosen = noItemChosen;
	_finder.addItemFinderListener(this);
    }

    /**
     * Bind the Attribute <tt>name</tt> in <tt>taskData</tt> to
     * <tt>finder</tt>, so that when <tt>finder</tt> changes the Attribute
     * is updated and vice versa.
     *
     * @param taskData TaskData to bind.
     * @param name name of Attribute to bind.
     * @param finder the ItemFinder to bind.
     * @param noSuchItem The string to put into the task data if the
     *                   reason returned by the ItemFinder is
     *                   NO_SUCH_ITEM
     * @param noItemChosen The string to put into the task data if the
     *                     reason returned by the ItemFinder in
     *                     NO_ITEM_CHOSEN
     *
     * @deprecated Use <CODE>taskData.addTaskDataBinder(name,
     * new ReasonItemFinderBinder(finder, noSuchItem, noItemChosen))</CODE>;
     * </PRE>
     *
     * @see com.sgi.sysadm.ui.ItemFinderState
     */
    static public void bind(TaskData taskData, String name,
			    ItemFinder finder,
			    String noSuchItem,
			    String noItemChosen) {
	ReasonItemFinderBinder binder =
	    new ReasonItemFinderBinder(finder, noSuchItem, noItemChosen);
	taskData.addTaskDataBinder(name, binder);
    }

    /**
     * Called when our TaskData Attribute changes.  Update our
     * ItemFinder. <B>This method does nothing</B>
     *
     * @param event The TaskDataEvent
     */
    public void taskDataChanged(TaskDataEvent event) {
    }

    /**
     * Called when our ItemView changes.  Update our TaskData
     * Attribute.
     *
     * @param event The ItemFinderEvent
     */
    public void itemFinderStateChanged(ItemFinderEvent event) {
	ItemFinderState state = event.getItemFinderState();
	String reasonStr = "";
	if (state.getItem() == null) {
	    Object reason = state.getReasons()[0];
	    Log.assert(reason != null,
		       "The ItemTester returned null as the reason " +
		       "for failure.");
	    if (reason == ItemFinderState.NO_SUCH_ITEM) {
		reasonStr = _noSuchItem;
	    } else if (reason == ItemFinderState.NO_ITEM_CHOSEN) {
		reasonStr = _noItemChosen;
	    } else if (reason == ItemFinderState.HEADER) {
		reasonStr = "";
	    } else {
		reasonStr = reason.toString();
	    }
	}
	TaskData taskData = getTaskData();
	if (taskData == null) {
	    _reasonStr = reasonStr;
	} else {
	    taskData.setString(getName(), reasonStr);
	}
    }

    /**
     * Set the TaskData and name of attribute that we are monitoring.
     *
     * @param taskData the TaskData to bind to.
     * @param name the name of the Attribute to monitor.
     */
    void setTaskData(TaskData taskData, String name) {
	super.setTaskData(taskData, name);
	if (_reasonStr != null) {
	    taskData.setString(name, _reasonStr);
	}
    }
}
