//
// StringJComboBoxBinder.java
//
//	Keep string task data in sync with the entries in a JComboBox.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui.taskData;

import javax.swing.*;
import java.awt.event.*;

/**
 * StringJComboBoxBinder is used to keep string task data synchronized
 * with the items in a JComboBox.
 * <P>
 * The idea is that a binder should be created for each item in a
 * JComboBox.  Each binder is created with a "value" different from
 * the other binders for the JComboBox.  When the task data changes,
 * each binder will be notified, and the one whose "value" matches
 * the task data will make its item the current item.  Conversely,
 * when one of the items is selected, its binder updates the task
 * data.  For example:
 * <pre>
 *	JComboBox heartbeat = new JComboBox();
 *	String hbOn = rs.getString("MainPage.hbOn");
 *	String hbOff = rs.getString("MainPage.hbOff");
 *	heartbeat.addItem(hbOn);
 *	heartbeat.addItem(hbOff);
 *	taskData.addTaskDataBinder(INVISIBLE_HB,
 *	    new StringJComboBoxBinder(heartbeat, hbOn, CLIKeys.TRUE_VALUE));
 *	taskData.addTaskDataBinder(INVISIBLE_HB,
 *	    new StringJComboBoxBinder(heartbeat, hbOff, CLIKeys.FALSE_VALUE));
 * </pre>
 * <P>
 * Note: The item state won't change when the user types into the
 * combobox until the user presses the Enter key or clicks the
 * mouse pointer outside (focus leaves the combobox).  To get
 * more dynamic item state updates, use
 * <a href="com.sgi.sysadm.ui.ItemFinder.html">ItemFinder</a> instead.
 */
public class StringJComboBoxBinder extends TaskDataBinder {

    private JComboBox _box;
    private Object _item;
    private String _value;

    /**
     * Construct a StringJComboBoxBinder.  Add an ItemListener
     * to <tt>box</tt> to keep it in sync with TaskData.  If our task
     * data is set to <tt>value</tt>, we make <tt>entry</tt> the
     * selected entry in <tt>box</tt>.  When <tt>entry</tt> is selected,
     * we set our task data to <tt>value</tt>.
     * <P>
     * Note: The item state won't change when the user types into the
     * combobox until the user presses the Enter key or clicks the
     * mouse pointer outside (focus leaves the combobox).  To get
     * more dynamic item state updates, use ItemFinder instead.
     *
     * @param box JComboBox to keep synchronized.
     * @param entry Entry to select when task data has <tt>value</tt>
     * @param value Value to set when <tt>item</tt> is selected.
     */
    public StringJComboBoxBinder(JComboBox box, Object entry,
				    String value) {
	_box = box;
	_item = entry;
	_value = value;
	box.addItemListener(new ItemListener() {
	    public void itemStateChanged(ItemEvent event) {
		String name = getName();
		TaskData taskData = getTaskData();
		if (event.getStateChange() == ItemEvent.SELECTED
		    && event.getItem() == _item
		    && taskData.getString(name) != _value) {
		    taskData.setString(name, _value);
		}
	    }
	});
    }

    /**
     * Bind the Attribute <tt>name</tt> in <tt>taskData</tt> to
     * <tt>box</tt>, so that when <tt>entry</tt> is selected in
     * <tt>box</tt> the Attribute is changed to <tt>value</tt>, and
     * vice versa.
     *
     * @param taskData TaskData to bind.
     * @param name name of Attribute to bind.
     * @param box JComboBox to bind.
     * @param entry Entry to select when Attribute takes on <tt>value</tt>.
     * @param value value to set Attribute to when <tt>entry</tt> is selected.
     *
     * @deprecated Use <CODE>taskData.addTaskDataBinder(name,
     * new StringJComboBoxBinder(box, entry, value))</CODE>
     */
    static public void bind(TaskData taskData, String name,
			    JComboBox box, Object entry, String value) {
	taskData.addTaskDataBinder(
	    name, new StringJComboBoxBinder(box, entry, value));
    }

    /**
     * Called when our task data changes.  Select our entry if The task
     * data has the <tt>value</tt> that was passed to our constructor.
     *
     * @param event TaskDataEvent.
     */
    public void taskDataChanged(TaskDataEvent event) {
	TaskData taskData = event.getTaskData();
	String name = event.getName();
	if (taskData.getString(name).equals(_value)
	    && _box.getSelectedItem() != _item) {
	    _box.setSelectedItem(_item);
	    _box.repaint();
	}
    }
}
