//
// StringJTextComponentBinder.java
//
//	Bind String task data to a swing text component.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui.taskData;

import javax.swing.text.*;
import javax.swing.event.*;

/**
 * StringJTextComponentBinder keeps a piece of String TaskData
 * synchronized with a JTextComponent.  If the TaskData changes
 * then StringJTextFieldBinder changes the JTextComponent, and if the
 * JTextComponent changes then StringJTextComponentBinder changes the
 * TaskData.
 */
public class StringJTextComponentBinder extends TaskDataBinder
implements DocumentListener {

    private JTextComponent _text;

    // It's very important that we don't do any further notification
    // when we're changing the value of a JTextComponent because when
    // we call JTextComponent.setText() we actually get *two*
    // notifications: 1 for the old text being deleted, the other for
    // the new text being added.  If we try to call setText() from
    // within one of these notifications, JAbstractDocument will
    // deadlock.
    private boolean _notifying = false;

    /**
     * Construct a StringJTextComponentBinder.  Add this as a
     * DocumentListener to <tt>text</tt> to keep it in sync with
     * TaskData.
     *
     * @param text JTextComponent to keep synchronized.
     */
    public StringJTextComponentBinder(JTextComponent text) {
	_text = text;
	_text.getDocument().addDocumentListener(this);
    }

    /**
     * Bind the Attribute <tt>name</tt> in <tt>taskData</tt> to
     * <tt>text</tt>, so that when <tt>text</tt> changes the Attribute
     * is updated and vice versa.
     *
     * @param taskData TaskData to bind.
     * @param name name of Attribute to bind.
     * @param text JTextComponent to bind.
     *
     * @deprecated Use <CODE>taskData.addTaskDataBinder(name,
     * new StringJTextComponentBinder(text))</CODE>
     */
    static public void bind(TaskData taskData, String name,
			    JTextComponent text) {
	taskData.addTaskDataBinder(name,
				     new StringJTextComponentBinder(text));
    }

    /**
     * Called when our TaskData Attribute changes.  Update our
     * JTextComponent.
     *
     * @param event the TaskDataEvent.
     */
    public void taskDataChanged(TaskDataEvent event) {
	String str = event.getTaskData().getString(event.getName());
	if (!_notifying && !str.equals(_text.getText())) {
	    _notifying = true;
	    _text.setText(str);
	    _notifying = false;
	}
    }

    /**
     * Called when our JTextComponent changes.  Update our TaskData
     * Attribute.
     *
     * @param event the DocumentEvent.
     */
    public void changedUpdate(DocumentEvent event) {
	textValueChanged();
    }

    /**
     * Called when our JTextComponent changes.  Update our TaskData
     * Attribute.
     *
     * @param event the DocumentEvent.
     */
    public void insertUpdate(DocumentEvent event) {
	textValueChanged();
    }

    /**
     * Called when our JTextComponent changes.  Update our TaskData
     * Attribute.
     *
     * @param event the DocumentEvent.
     */
    public void removeUpdate(DocumentEvent event) {
	textValueChanged();
    }

    /**
     * Shared code for all of the DocumentListener methods.  Update
     * our TaskData Attribute so that it has the same value as our
     * JTextComponent.
     */
    private void textValueChanged() {
	if (!_notifying) {
	    _notifying = true;
	    getTaskData().setString(getName(), _text.getText());
	    _notifying = false;
	}
    }
}
