//
// TaskDataBinder.java
//
//	Get notification when task data changes.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui.taskData;
import com.sgi.sysadm.util.*;

/**
 * TaskDataBinder is a specialization of AttrListener that adds
 * filtering of notification by "name".  TaskDataBinder also
 * provides accessors for the TaskData being listened to and the name
 * of the piece of TaskData being listened to.  This information is
 * typically needed in subclasses of TaskDataBinder so they know
 * which TaskData to change when they receive notification from UI
 * Components.
 */
public abstract class TaskDataBinder implements AttrListener {

    private TaskData _taskData = null;
    private String _name = null;

    /**
     * Called whenever an Attribute in TaskData changes.  Call
     * taskDataChanged if the key of the Attribute that changed
     * matches our name.
     *
     * @param event specifies which Attribute changed.
     *
     * @see #getName
     */
    public void attrChanged(AttrEvent event) {
	Attribute attr = event.getAttr();
	if (attr.getKey().equals(_name)) {
	    taskDataChanged(new TaskDataEvent(_taskData, _name));
	}
    }

    /**
     * Called whenever an Attribute in TaskData is added.  Calls
     * <tt>Log.fatal</tt> when notified that our named Attribute is
     * added because you must bind to pre-existing task data.
     *
     * @param event specifies which Attribute changed.
     *
     * @see Log#fatal
     */
    public void attrAdded(AttrEvent event) {
	Attribute attr = event.getAttr();
	if (attr.getKey().equals(_name)) {
	    Log.fatal(
		"Can't bind to TaskData attribute that doesn't yet exist");
	}
    }

    /**
     * Called whenever an Attribute in TaskData is removed.  Calls
     * <tt>Log.fatal</tt> when notified that our named Attribute has
     * been removed  because you must not remove a piece of task data
     * that you're bound to.
     *
     * @param event specifies which Attribute changed.
     * @see Log#fatal
     */
    public void attrRemoved(AttrEvent event) {
	Attribute attr = event.getAttr();
	if (attr.getKey().equals(_name)) {
	    Log.fatal("Can't remove a TaskData attribute that is bound.");
	}
    }

    /**
     * Called whenever the Attribute whose key matches our name
     * changes.
     *
     * @param event specifies which Attribute changed.
     * @see #getName
     */
    public abstract void taskDataChanged(TaskDataEvent event);

    /**
     * Accessor for the TaskData we're listening to.
     *
     * @return TaskData we're listening to.
     */
    protected TaskData getTaskData() {
	return _taskData;
    }

    /**
     * Accessor for the name of the piece of TaskData we're listening
     * to.
     *
     * @return name of the piece of TaskData we're listening to.
     */
    protected String getName() {
	return _name;
    }

    /**
     * Set the TaskData and name of attribute that we are monitoring.
     *
     * @param taskData the TaskData to bind to.
     * @param name the name of the Attribute to monitor.
     */
    /*package*/ void setTaskData(TaskData taskData, String name) {
	Log.assert(_taskData == null,
		   "Attempt to install a TaskDataBinder more than once");
	_taskData = taskData;
	_name = name;
    }
}
