//
// TextItemFinderBinder.java
//
//	Bind String task data to an ItemFinder
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui.taskData;

import com.sgi.sysadm.ui.event.*;
import com.sgi.sysadm.category.*;
import com.sgi.sysadm.ui.*;

/**
 * TextItemFinderBinder keeps a piece of String TaskData
 * synchronized with an ItemFinder.  If the TaskData changes
 * then TextItemFinderBinder changes the ItemFinder, and if the
 * ItemFinder changes then TextItemFinderBinder changes the
 * TaskData.
 */
public class TextItemFinderBinder extends TaskDataBinder
                                      implements ItemFinderListener {

    private ItemFinder _finder;

    // It's very important that we don't do any further notification
    // when we're changing the value of an ItemFinder because when
    // we call ItemFinder.setText() we actually get *two*
    // notifications: 1 for the old text being deleted, the other for
    // the new text being added.  If we try to call setText() from
    // within one of these notifications, we'll loop forever
    private boolean _notifying = false;

    /**
     * Construct a TextItemFinderBinder.
     *
     * @param finder ItemFinder to keep synchronized.
     */
    public TextItemFinderBinder(ItemFinder finder) {
	_finder = finder;
	_notifying = true;
	_finder.addItemFinderListener(this);
	_notifying = false;
    }

    /**
     * Bind the Attribute <tt>name</tt> in <tt>taskData</tt> to
     * <tt>finder</tt>, so that when <tt>finder</tt> changes the Attribute
     * is updated and vice versa.
     *
     * @param taskData TaskData to bind.
     * @param name name of Attribute to bind.
     * @param finder the ItemFinder to bind.
     *
     * @deprecated Use <CODE>taskData.addTaskDataBinder(name,
     * new TextItemFinderBinder(finder))</CODE>
     */
    static public void bind(TaskData taskData, String name,
			    ItemFinder finder) {
	taskData.addTaskDataBinder(name,
				     new TextItemFinderBinder(finder));
    }

    /**
     * Called when our TaskData Attribute changes.  Update our
     * ItemFinder.
     *
     * @param event The TaskDataEvent
     */
    public void taskDataChanged(TaskDataEvent event) {
	String str = event.getTaskData().getString(event.getName());
	if (!_notifying && !str.equals(
	    _finder.getItemFinderState().getText())) {
	    _notifying = true;
	    _finder.setTextByName(str);
	    _notifying = false;
	}
    }

    /**
     * Called when our ItemView changes.  Update our TaskData
     * Attribute.
     *
     * @param event The ItemFinderEvent
     */
    public void itemFinderStateChanged(ItemFinderEvent event) {
	if (!_notifying) {
	    _notifying = true;
	    getTaskData().setString(getName(),event.getItemFinderState().getText());
	    _notifying = false;
	}
    }
}
