//
// ItemTreeCellRenderer.java
//
//	Render nodes in the TreeView.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui.treeView;

import com.sgi.sysadm.ui.*;
import javax.swing.*;
import javax.swing.tree.*;
import java.awt.*;

/**
 * ItemTreeCellRenderer renders nodes in the TreeViewPane JTree using FtrIcon.
 *
 * @author	Roger Chickering
 * @author	John Relph
 * @version	$Revision: 1.11 $ $Date: 2000/07/24 21:58:15 $
 */
class ItemTreeCellRenderer extends JLabel
    implements TreeCellRenderer, TreeViewProperties
{
    private Color _textColor;
    private TreeContext _tc;
    private Font _rootFont;
    private Font _childFont;
    private String _prefix;

    /**
     * Construct an ItemTreeCellRenderer.
     *
     * @param tc Tree Context.
     * @param prefix String used to find properties.
     */
    ItemTreeCellRenderer(TreeContext tc, String prefix) {
	_tc = tc;
	_prefix = prefix;
	_textColor = tc._rs.getColor(propNames(TEXT_COLOR));

	// Set the background to the selection color.  We will make
	// ourselves opaque if we're selected.
	setBackground(tc._rs.getColor(propNames(SELECT_COLOR)));

	_rootFont = tc._rs.getFont(propNames(ROOT_FONT));
	_childFont = tc._rs.getFont(propNames(CHILD_FONT));

	int borderHeight = _tc._rs.getPixels(propNames(CELL_BORDER_HEIGHT));
	int borderWidth = _tc._rs.getPixels(propNames(CELL_BORDER_WIDTH));
	setBorder(BorderFactory.createEmptyBorder(
	    borderHeight, borderWidth, borderHeight, borderWidth));
    }

    /**
     * Return a component to render <tt>value</tt>.
     *
     * @param tree JTree to render.
     * @param value DefaultMutableTreeNode to render.
     * @param sel true if <tt>value</tt> is selected.
     * @param expanded true if <tt>value</tt> is expanded.
     * @param leaf true if <tt>value</tt> is a leaf.
     * @param row the row to render.
     * @param hasFocus true if <tt>value</tt> has focus.
     *
     * @return Component to render <tt>value</tt>.
     *
     * @see javax.swing.tree.TreeCellRenderer
     */
    public Component getTreeCellRendererComponent(
	JTree tree,
	Object value,
	boolean sel,
	boolean expanded,
	boolean leaf,
	int row,
	boolean hasFocus)
    {
	try {
	    DefaultMutableTreeNode node = (DefaultMutableTreeNode)value;

	    ItemUserObject itemInfo = (ItemUserObject)node.getUserObject();

	    Icon icon = itemInfo.getIcon();
	    if (icon instanceof FtrIcon) {
		FtrIcon ftrIcon = (FtrIcon)icon;
		ftrIcon.setSelected(sel);
		ftrIcon.setLocated(hasFocus);
	    }
	    setIcon(icon);

	    if (itemInfo.getItem() == _tc._rootItem) {
		setFont(_rootFont);
	    } else {
		setFont(_childFont);
	    }
	    setText(itemInfo.toString());

	} catch (ClassCastException ex) {
	    setIcon(null);
	    setText(value.toString());
	}
	// If we're selected, set opaque so our background color is
	// visible.
	setOpaque(sel);
	setForeground(_textColor);
	return this;
    }

    private String[] propNames(String resource) {
	return new String[] {
	    _prefix + resource,
	    TREEVIEWPANE_PROPERTY_PREFIX + resource
	};
    }
}
