//
// ItemUserObject.java
//
//	User object (in the DefaultMutableTreeNode sense) that
//	represents an item.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui.treeView;

import com.sgi.sysadm.category.*;
import com.sgi.sysadm.ui.*;
import com.sgi.sysadm.util.*;
import javax.swing.*;
import javax.swing.tree.*;
import java.awt.*;
import java.text.*;
import java.util.*;

/**
 * ItemUserObject is used as the user object on DefaultMutableTreeNodes
 * in the TreeViewPane JTree.  It attaches an Item to each node in the
 * tree and handles coming up with a String for displaying each Item.
 *
 * @author	Roger Chickering
 * @author	John Relph
 * @version	$Revision: 1.21 $ $Date: 2000/07/24 21:58:15 $
 *
 * @see com.sgi.sysadm.ui.treeView.TreeViewPane
 * @see javax.swing.JTree
 * @see javax.swing.DefaultMutableTreeNode
 */
public class ItemUserObject implements TreeViewProperties {
    private String _itemSelector;
    private Icon _icon;
    private TreeContext _tc;
    private DefaultMutableTreeNode _node;
    private TreePath _path;
    private String _prefix;
    private String _itemCategory;
    private RenderedObjectListener _iconL;
    private RenderedObjectListener _nameL;
    private IconRenderer _iconRenderer;
    private NameRenderer _nameRenderer;
    private String _name;
    private TreeCategoryListener _listener;
    private Item _item;
    private static String _class = "ItemUserObject";

    /**
     * Construct an ItemUserObject.
     *
     * @param tc TreeContext for display parameters.
     * @param node DefaultMutableTreeNode for this node and Item.
     * @param item Item to display.  The Item can be <tt>null</tt>,
     *             but then no icon or text will be shown.  You can
     *             use the <tt>setItem</tt> method to tell the
     *             ItemUserObject that the Item has changed.
     * @param itemCategory The package-qualified name of the Category of the
     *		           Item to display.
     * @param listener The TreeCategoryListener maintaining the
     *                 children of this node.
     */
    public ItemUserObject(TreeContext tc, DefaultMutableTreeNode node,
			  Item item, String itemCategory,
			  TreeCategoryListener listener) {
	_tc = tc;
	_node = node;
	_path = new TreePath(_node);
	_item = item;
	if (item != null) {
	    _itemSelector = item.getSelector();
	} else {
	    _itemSelector = null;
	}

	_itemCategory = itemCategory;
	_listener = listener;
	_iconL = new RenderedObjectListener() {
	    public void renderedObjectChanged(String selector, Object object) {
		_icon = (Icon)object;
//		Log.debug(_class, "drawing " + selector);
		refreshIcon();
	    }
	};
	_nameL = new RenderedObjectListener() {
	    public void renderedObjectChanged(String selector, Object object) {
		_name = (String)object;
//		Log.debug(_class, "rendering name " + selector + ": " + _name);
		refreshIcon();
	    }
	};

	Log.debug(_class, "looking up renderers in " + _itemCategory);

	_iconRenderer = _tc._hc.getIconRenderer(_itemCategory);
	_nameRenderer = _tc._hc.getNameRenderer(_itemCategory);
	if (_itemSelector != null) {
	    setUpItemListeners();
	}
    }

    private void setUpItemListeners() {
	    _iconRenderer.addIconListener(_itemSelector, null,
					  _tc._iconWidth, _tc._iconHeight,
					  _iconL);
	    _nameRenderer.addNameListener(_itemSelector,
					  GenericItemRenderer.TREE_NODE,
					  _nameL);
    }

    /**
     * Sets the item that's currently being displayed.  The selector
     * of the Item passed must be the same as the one passed to the
     * constructor.  An exception to this is if the Item passed to the
     * contructor was null, then the selector of the Item passed into
     * this method can be anything (but the Item can't be null).
     */
    void setItem(Item item) {
	if (_itemSelector == null ) {
	    Log.assert(item != null, "Can't call setItem with a null Item");
	    _itemSelector = item.getSelector();
	    setUpItemListeners();
	}

	Log.assert(item.getSelector().equals(_itemSelector),
		   "Item's selector changed!");
	_item = item;
    }

    /**
     * Returns the Item currently being displayed
     */
    public Item getItem() {
	return _item;
    }

    /**
     * Render our Item as a String.  ItemTreeCellRenderer calls this
     * when we are to be drawn.
     *
     * @return String to display in JTree for our node.
     */
    public String toString() {
	return _name;
    }

    /**
     * Redraw the node
     */
    private void refreshIcon() {
	DefaultTreeModel model = (DefaultTreeModel)_tc._tree.getModel();
	if (model == null) {
	    return;
	}
	// TreeModel.nodeChanged() doesn't do anything to
	// the root item.
	if (_item == _tc._rootItem) {
	    Rectangle bounds =
		_tc._tree.getPathBounds(_path);
	    if (bounds != null) {
		_tc._tree.repaint(bounds.x, bounds.y,
				  bounds.width, bounds.height);
	    }
	} else {
	    model.nodeChanged(_node);
	}
    }

    /**
     * Get the Icon to be displayed for our node.
     * ItemTreeCellRenderer calls this to get the Icon to display.
     *
     * @return Icon to be displayed for our node.
     */
    Icon getIcon() {
	return _icon;
    }

    /**
     * Dereference IconRenderers and NameRenderers so we can be
     * garbage-collected. Also stop our TreeCategoryListener from
     * listening and clean up any children.
     *
     */
    public void dispose() {
	if (_itemSelector != null) {
	    _iconRenderer.removeIconListener(_itemSelector, _iconL);
	    _nameRenderer.removeNameListener(_itemSelector, _nameL);
	}
	if (_listener != null) {
	    _listener.dispose();
	}
    }
}
