//
// TreeCategoryListener.java
//
//	CategoryListener that keeps a tree populated with nodes
//	corresponding to the items in a category.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui.treeView;

import com.sgi.sysadm.category.*;
import com.sgi.sysadm.ui.*;
import com.sgi.sysadm.util.*;
import javax.swing.*;
import javax.swing.tree.*;
import java.lang.reflect.*;
import java.text.*;
import java.util.*;

/**
 * TreeCategoryListener monitors an Association or Category
 * representing the children of a node in the tree.  Nodes are added
 * to and removed from the tree as items come and go in the Category.
 *
 * @author	Roger Chickering
 * @author	John Relph
 * @version	$Revision: 1.22 $ $Date: 2000/07/24 21:58:15 $
 */
class TreeCategoryListener extends CategoryAdapter
    implements TreeViewProperties
{
    private DefaultMutableTreeNode _parent;
    private TreeContext _tc;
    private Category _category;
    private boolean _expandNeeded;
    private int _changeBlock;
    private TreeLevel _levels[];
    private int _level;
    private String _rootFilter;
    private String _prefix;
    private ItemComparator _compare;
    private ItemTester _itemTester;
    private static String _class =
	ResourceStack.getClassName(TreeCategoryListener.class);

    /**
     * Construct a TreeCategoryListener.
     *
     * @param tc TreeContext.
     * @param parent Node whose children we are maintaining.
     * @param levels Description of the levels in the tree.
     * @param level The level of the nodes that we will be maintaining.
     * @param prefix String property prefix.
     */
    TreeCategoryListener(TreeContext tc,
			 DefaultMutableTreeNode parent,
			 TreeLevel levels[], int level, String prefix) {
	_tc = tc;
	_parent = parent;
	_levels = levels;
	_level = level;
	_prefix = prefix;
	_itemTester = _levels[_level]._itemTester;
	_compare = _levels[_level]._itemComparator;
    }

    /**
     * Create a node for our tree.  If we're not at the deepest level,
     * get the association for the children of the node and install a
     * TreeCategoryListener to keep the children up to date.
     *
     * @param tc TreeContext.
     * @param levels Description of the tree we're displaying.
     * @param level The level of the node we're creating.
     * @param item Item this node should represent.
     * @param prefix String property prefix.
     *
     * @return a tree node representing <tt>item</tt>.
     */
    static DefaultMutableTreeNode createNode(TreeContext tc,
					     TreeLevel levels[], int level,
					     Item item,
					     String prefix) {
	DefaultMutableTreeNode node = new DefaultMutableTreeNode();
	TreeLevel thisLevel = levels[level];
	String itemCategory = thisLevel._category;
	TreeCategoryListener listener = null;
	Category category = null;
	if (level < levels.length - 1) {
	    TreeLevel treeLevel = levels[level + 1];
	    final String categoryName =
		ResourceStack.getClassName(treeLevel._category);

	    if (treeLevel._useAssoc) {
		if (item != null) {
		    category =
			tc._hc.getAssociation(item.getType(),
					      item.getSelector(),
					  categoryName);
		}
	    } else {
		category = tc._hc.getCategory(categoryName);
	    }
	    if (category != null) {
		listener = new TreeCategoryListener(tc, node,
						    levels, level + 1,
						    prefix);
		listener._rootFilter = treeLevel._rootFilter;
		node.setAllowsChildren(true);
		if (level == 0) {
		    listener.expand();
		}
	    }
	}
	ItemUserObject userObj =
	    new ItemUserObject(tc, node, item, itemCategory, listener);
	node.setUserObject(userObj);

	// we have to set the model before setting the category, because
	// the listener uses the model to process added items
	//
	if (level == 0) {	// is this the root node?
	    tc._tree.setModel(new DefaultTreeModel(node));
	}
	if (category != null && listener != null) {
	    listener.setCategory(category);
	}
	return node;
    }

    /**
     * Set the category for the children of our node.
     *
     * @param category category for the children of our node.
     */
    void setCategory(Category category) {
	if (_category != null) {
	    _category.removeCategoryListener(this);
	}
	_category = category;
	_category.addCategoryListener(this, NotificationFilter.ALL_ITEMS);
    }

    /**
     * Expand our children.  We need to postpone this action because
     * JTree expansion doesn't work until we actually have some
     * children.
     */
    void expand() {
	_expandNeeded = true;
    }

    /**
     * Called by Category when a new item arrives.  Create a node in
     * the tree to represent the item.
     *
     * @param item Item that was added.
     */
    public void itemAdded(Item item) {
	if (_itemTester != null &&
	    (! _itemTester.testItem(item).passed())) {
	    return;
	}

	if (_rootFilter != null) {
	    if (_tc._rootItem == null) {
		return;
	    } else if (!item.getString(_rootFilter).
		equals(_tc._rootItem.getSelector())) {
		return;
	    }
	}

	DefaultMutableTreeNode node =
	    createNode(_tc, _levels, _level, item, _prefix);
	DefaultTreeModel model = (DefaultTreeModel)_tc._tree.getModel();

	int childCount = _parent.getChildCount();
	int index = model.getChildCount(_parent);

	for (int ii = 0; ii < childCount; ++ii) {
	    DefaultMutableTreeNode nextNode =
		(DefaultMutableTreeNode)_parent.getChildAt(ii);
	    ItemUserObject itemInfo = (ItemUserObject)nextNode.getUserObject();
	    if (_compare.compareItems(itemInfo.getItem(), item) > 0) {
		index = ii;
		break;
	    }
	}

	model.insertNodeInto(node, _parent, index);

	if (_changeBlock == 0 && (_expandNeeded || _tc._autoExpand)) {
	    _tc._tree.expandPath(new TreePath(
		((DefaultTreeModel)_tc._tree.getModel()).getPathToRoot(node)));
	    _expandNeeded = false;
	}
    }

    /**
     * Called by Category when an item changes.  If we're filtering,
     * see if the change will affect this item's visiblity.
     *
     * @param oldItem old item.
     * @param newItem new item.
     */
    public void itemChanged(Item oldItem, Item newItem) {
	boolean showItem = true;
	if (_itemTester != null &&
	    (! _itemTester.testItem(newItem).passed())) {
	    showItem = false;
	}

	if (showItem == true && _rootFilter != null) {
	    if (_tc._rootItem == null) {
		showItem = false;
	    } else if (! newItem.getString(_rootFilter).equals(_tc._rootItem.getSelector())) {
		showItem = false;
	    }
	}

	DefaultTreeModel model = (DefaultTreeModel)_tc._tree.getModel();
	int childCount = model.getChildCount(_parent);

	// first, find the old item if already in the tree
	//
	DefaultMutableTreeNode child = null;
	ItemUserObject oldInfo = null;
	int oldIndex = -1;
	for (int ii = 0; ii < childCount; ii++) {
	    child = (DefaultMutableTreeNode)model.getChild(_parent, ii);
	    oldInfo = (ItemUserObject)child.getUserObject();
	    if (oldItem == oldInfo.getItem()) { // found it
		oldIndex = ii;
		break;
	    } 
	}

	// did we find it in the tree?  should we show it?
	//
	if (oldIndex != -1) {	// found
	    if (!showItem) {
		model.removeNodeFromParent(child); // so delete
		return;
	    }
	} else {		// not found
	    if (showItem) {
		itemAdded(newItem); // add it
	    }
	    return;		// don't show and don't add
	}

	// now, see if the new item would be somewhere else in the tree
	//
	int index = model.getChildCount(_parent);
	int oldFound = 0;
	for (int ii = 0; ii < childCount; ++ii) {
	    DefaultMutableTreeNode nextNode =
		(DefaultMutableTreeNode)_parent.getChildAt(ii);
	    ItemUserObject itemInfo = (ItemUserObject)nextNode.getUserObject();
	    Item nextItem = itemInfo.getItem();
	    if (oldItem == nextItem) {
		oldFound = 1;
	    } else if (_compare.compareItems(nextItem, newItem) > 0) {
		index = ii;
		break;
	    }
	}
	index -= oldFound;	// fudge for old item in tree

	Log.debug(_class,
		  "new index is " + index + ", oldIndex is " + oldIndex +
		  ", fudge is " + oldFound);

	if (index != oldIndex) {
	    model.removeNodeFromParent(child);
	    itemAdded(newItem);
	} else {
	    oldInfo.setItem(newItem);
	    model.nodeChanged(child);
	}
    }

    /**
     * Called when an item is removed from a category.  Get rid of its
     * node in the tree.
     *
     * @param item Item that was removed.
     */
    public void itemRemoved(Item item) {
	DefaultTreeModel model = (DefaultTreeModel)_tc._tree.getModel();
	int childCount = model.getChildCount(_parent);
	for (int ii = 0; ii < childCount; ii++) {
	    DefaultMutableTreeNode child =
		(DefaultMutableTreeNode)model.getChild(_parent, ii);
	    ItemUserObject info = (ItemUserObject)child.getUserObject();
	    if (item == info.getItem()) {
		model.removeNodeFromParent(child);
		info.dispose();
		break;
	    }
	}
    }

    /**
     * Called when a change block begins.  We keep track of change
     * blocks as part of our expansion logic.
     */
    public void beginBlockChanges() {
	_changeBlock++;
    }

    /**
     * Called when a change block ends.  Expand if we're supposed to
     * and this is the outermost change block.
     */
    public void endBlockChanges() {
	if (--_changeBlock == 0 && (_expandNeeded || _tc._autoExpand)
	    && _parent.getChildCount() > 0) {
	    _tc._tree.expandPath(new TreePath(
		((DefaultTreeModel)_tc._tree.getModel()).
		getPathToRoot(_parent)));
	    _expandNeeded = false;
	}
    }

    /**
     * Dereference so we can be garbage-collected.
     *
     */
    public void dispose() {
	if (_category != null) {
	    _category.removeCategoryListener(this);
	}
	DefaultTreeModel model = (DefaultTreeModel)_tc._tree.getModel();
	int childCount = model.getChildCount(_parent);
	for (int ii = 0; ii < childCount; ii++) {
	    DefaultMutableTreeNode child =
		(DefaultMutableTreeNode)model.getChild(_parent, 0);
	    model.removeNodeFromParent(child);
	    ((ItemUserObject)child.getUserObject()).dispose();
	}
    }
}
