//
// TreeViewPane.java
//
//	Panel containing a scrolling tree view.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui.treeView;

import com.sgi.sysadm.category.*;
import com.sgi.sysadm.ui.*;
import com.sgi.sysadm.util.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.plaf.basic.*;
import javax.swing.tree.*;
import java.awt.*;
import java.awt.event.*;
import java.util.*;

/**
 * <P>TreeViewPane contains a scrolling tree pane.  The pane may display a
 * number of different trees, one at a time.</P>
 * <P>The structure of the tree is defined by properties in the
 * ResourceStack:</P>
 * <P><TT><I>&lt;prefix&gt;</I>.tree<I>&lt;n&gt;</I></TT> specifies an array
 * of tree names,
 * one for each tree structure.  For each <I>treename</I> so specified,
 * <TT><I>&lt;prefix&gt;</I>.<I>&lt;treename&gt;</I>.level<I>&lt;n&gt;</I></TT>
 * specifies an array of Category corresponding to each level in the tree.</P>
 * <P>The Category at level 0 of each tree must match the type of the root
 * Item of the TreeViewPane.
 * Each level of the tree is populated by the members of an Association
 * between the Item we're displaying and the Category at the next level.
 * For each Item in level <I>&lt;n&gt;</I>, we monitor the Association with
 * the Category at level <I>&lt;n+1&gt;</I>.</P>
 * <P>If it is necessary to use a Category instead of an Association at a
 * given level in the tree, specify it by setting the property <PRE>
 *   <I>&lt;prefix&gt;</I>.<I>&lt;treename&gt;</I>.level<I>&lt;n&gt;</I>.useAssoc = false
 </PRE></P>
 *
 * @author	Roger Chickering
 * @author	John Relph
 * @version	$Revision: 1.18 $ $Date: 2000/07/24 21:58:15 $
 *
 * @see com.sgi.sysadm.ui.treeView.TreeViewProperties
 * @see javax.swing.JScrollPane
 */
public class TreeViewPane extends JScrollPane
    implements TreeViewProperties, ExtraCleanup
{
    private TreeContext _tc;
    private Vector _trees = new Vector();
    private Vector _treeInfo = new Vector();
    private Vector _actionListeners = new Vector();
    private CategoryListener _listener;
    private NotificationFilter _notifyFilter;
    private Category _category;
    private String _prefix;
    private ResourceStack _rs = new ResourceStack();
    private int _treeIndex = 0;
    private DefaultMutableTreeNode _rootNode;

    private static final Hashtable _testers = new Hashtable();
    private static final Hashtable _comparators = new Hashtable();
    private static final ItemComparator _defaultComparator =
	new ItemComparator() {
	    public int compareItems(Item first, Item second) {
		return first.getSelector().compareTo(second.getSelector());
	    }
	};

    private String _class =
	ResourceStack.getClassName(TreeViewPane.class);

    private final String TREEVIEWPANE_CLASS =
	"com.sgi.sysadm.ui.treeView.TreeViewPane";

    private class TreeInfoStruct {
	public String name;
	public boolean showRoot;
	public TreeInfoStruct(String name, boolean showRoot)
	{
	    this.name = name;
	    this.showRoot = showRoot;
	}
    }

    /**
     * Construct a TreeViewPane.  Read the specifications of the trees we
     * will display from the ResourceStack.
     *
     * @param uic UIContext for busy state.
     * @param hc HostContext for getting Categories.
     * @param rs ResourceStack for getting tree structure.
     * @param rootItem Item at the root of the tree.
     * @param prefix String used to retrieve properties from the ResourceStack.
     */
    public TreeViewPane(UIContext uic, HostContext hc,
			ResourceStack rs, Item rootItem,
			String prefix) {
	_prefix = prefix;

	_rs.pushBundle(TREEVIEWPANE_CLASS +
		       ResourceStack.BUNDLE_SUFFIX);
	if (rs != null) {
	    _rs.pushStack(rs);
	}

	// create the notification filter and listener
	//
	_listener = new CategoryAdapter() {
	    public void itemAdded(Item item) {
		Log.debug(_class, "itemAdded " + item.getSelector());
		setItem(item);
	    }

	    public void itemRemoved(Item item) {
		Log.debug(_class, "itemRemoved " + item.getSelector());
		_tc._tree.getSelectionModel().clearSelection();
		setItem(null);
	    }

	    private void setItem(Item item) {
		_tc._rootItem = item;
		if (_rootNode == null) { // should never happen
		    return;
		}
		if (item == null) {
		    setTree(_treeIndex);
		    return;
		}
		final TreeInfoStruct treeInfo =
		(TreeInfoStruct)_treeInfo.elementAt(_treeIndex);
		if (treeInfo.showRoot) {
		    _tc._tree.setRootVisible(true);
		}
		((DefaultTreeModel)_tc._tree.getModel()).reload();
		if (_tc._autoExpand) {
		    expandAll();
		}
	    }
	};

	String baseCategory = null;
	_notifyFilter = new NotificationFilter();
	if (rootItem != null) {
	    baseCategory = rootItem.getType();
	    _notifyFilter.monitorItem(rootItem.getSelector());
	}

	// Read our tree structures from the ResourceStack.
	//
	int treeNum = 0;
	while (true) {
	    String label = null;
	    try {
		String treeName = _rs.getString(_prefix + TREE + treeNum);
		final int treeIdx = treeNum++;

		boolean showRoot =
		    _rs.getBoolean(_prefix + "." + treeName + SHOW_ROOT, true);

		Vector levels = new Vector();
		int levelNum = 0;
		while (true) {
		    try {
			String levelString = _prefix + "." + treeName +
			    LEVEL + levelNum++;
			String category = _rs.getString(levelString);
			TreeLevel level = new TreeLevel(category);
			levels.addElement(level);

			// .useAssoc is optional
			//
			level._useAssoc =
			    _rs.getBoolean(levelString + USE_ASSOC, true);

			// .rootFilter is optional
			//
			level._rootFilter =
			    _rs.getString(levelString + ROOT_FILTER_ATTR,
					  null);

			// .itemComparator is optional
			//
			level._itemComparator =
			    getItemComparator(levelString, level);

			// .itemTester is optional
			//
			level._itemTester =
			    getItemTester(levelString);

		    } catch (MissingResourceException ex) {
			break;
		    }
		}
		Log.assert(levels.size() > 0, "Tree: \"" + treeName +
			   "\" specified with no levels");
		TreeLevel tlevels[] = new TreeLevel[levels.size()];
		levels.copyInto(tlevels);
		_trees.addElement(tlevels);
		_treeInfo.addElement(new TreeInfoStruct(treeName, showRoot));

		String categoryName =
		    ResourceStack.getClassName(tlevels[0]._category);
		if (baseCategory == null) {
		    baseCategory = categoryName;
		} else {
		    Log.assert(baseCategory.equals(categoryName),
			       "Root category of tree " + treeIdx +
			       " (" + categoryName +
			       ") does not match type of root item (" +
			       baseCategory + ")");
		}
	    } catch (MissingResourceException ex) {
		break;
	    }
	}

	Log.assert(treeNum > 0, "No tree found in properties");

	Log.debug(_class, "base category is " + baseCategory);

	_category = hc.getCategory(baseCategory);
	_tc = new TreeContext(uic, _rs, hc, new JTree((TreeModel)null), rootItem, _prefix);

	ItemTreeCellRenderer renderer = new ItemTreeCellRenderer(_tc, _prefix);
	_tc._tree.setCellRenderer(renderer);

	String toolTip = _rs.getString(_prefix + TOOLTIP_TEXT, null);
	if (toolTip != null) {
	    renderer.setToolTipText(toolTip);
	}

	_tc._tree.putClientProperty("JTree.lineStyle", "None");
	_tc._tree.setBackground(
	    _tc._rs.getColor(propNames(BACKGROUND)));
	_tc._tree.getSelectionModel().setSelectionMode(
	    TreeSelectionModel.SINGLE_TREE_SELECTION);

	try {
	    BasicTreeUI ui = (BasicTreeUI)_tc._tree.getUI();
	    FtrIcon openArrow = (FtrIcon)_rs.getIcon(propNames(OPENED_ICON));
	    openArrow.setSize(_tc._iconWidth, _tc._iconHeight);
	    FtrIcon closedArrow = (FtrIcon)_rs.getIcon(propNames(CLOSED_ICON));
	    closedArrow.setSize(_tc._iconWidth, _tc._iconHeight);
	    ui.setExpandedIcon(openArrow);
	    ui.setCollapsedIcon(closedArrow);
	} catch (ClassCastException ex) {
	    Log.warning(_class + ":" + _prefix, "JTree has unexpected UI");
	}

	setViewportView(_tc._tree);

	setPreferredSize(
	    new Dimension(_rs.getPixels(propNames(PANE_WIDTH)),
			  _rs.getPixels(propNames(PANE_HEIGHT))));

	setTree(0);

	_tc._tree.addMouseListener(new MouseAdapter() {
	    ToolTipManager _toolTipMgr = ToolTipManager.sharedInstance();
	    long _releaseTime;
	    boolean _active = true;

	    boolean pathFound(MouseEvent event) {
		TreePath path = _tc._tree.getPathForRow(
		    _tc._tree.getRowForLocation(event.getX(),
						event.getY()));
		if (path == null) {
		    return false;
		}
		return true;
	    }
	    public void mouseEntered(MouseEvent event) {
		_active = pathFound(event);
		_toolTipMgr.setEnabled(_active);
		if (_active) {
		    _toolTipMgr.mouseEntered(event);
		}
	    }
	    public void mouseDragged(MouseEvent event) {
		if (_active) {
		    _toolTipMgr.mouseEntered(event);
		}
	    }
	    public void mouseExited(MouseEvent event) {
		if (_active) {
		    _toolTipMgr.mouseExited(event);
		}
		_active = true;
		_toolTipMgr.setEnabled(_active);
	    }
	    public void mouseMoved(MouseEvent event) {
		boolean found = pathFound(event);
		if (found != _active) {
		    _active = found;
		    _toolTipMgr.setEnabled(_active);
		}
		if (_active) {
		    _toolTipMgr.mouseMoved(event);
		}
	    }
	    public void mousePressed(MouseEvent event) {
		if (_active) {
		    _toolTipMgr.mousePressed(event);
		}
	    }
	    public void mouseReleased(MouseEvent event) {
		_releaseTime = System.currentTimeMillis();
	    }
	    public void mouseClicked(MouseEvent event) {
		// If a menu release happens above an item, the mouse
		// up event results in a call to mouseClicked.  We
		// don't want to launch an item view, though.
		// 500 (half a sec) seems to cause the desired effect.
		//
		if (System.currentTimeMillis() - _releaseTime > 500) {
		    return;
		}
		TreePath path = _tc._tree.getPathForRow(
		    _tc._tree.getRowForLocation(event.getX(),
						event.getY()));
		if (path == null) {
		    return;
		}
		ActionEvent action
		    = new ActionEvent(path, ActionEvent.ACTION_PERFORMED,
				      _class + ":" + _prefix);
		notifyAction(action);
	    }
	});

	// Hack to prevent BasicTreeUI from expanding/collapsing nodes
	// which are double-clicked.  BasicTreeUI implements
	// MouseListener and has hard-coded logic for
	// expanding/collapsing when it gets a mousePressed() with a
	// click count of 2.  So we remove BasicTreeUI as a listener
	// and interpose ourselves so we can swallow the double-click
	// events.
	try {
	    final MouseListener ui = (MouseListener)_tc._tree.getUI();
	    _tc._tree.removeMouseListener(ui);
	    _tc._tree.addMouseListener(new MouseListener() {
		public void mouseClicked(MouseEvent ev) {
		    ui.mouseClicked(ev);
		}
		public void mouseEntered(MouseEvent ev) {
		    ui.mouseEntered(ev);
		}
		public void mouseExited(MouseEvent ev) {
		    ui.mouseExited(ev);
		}
		public void mousePressed(MouseEvent ev) {
		    if (ev.getClickCount() <= 1) {
			ui.mousePressed(ev);
		    }
		}
		public void mouseReleased(MouseEvent ev) {
		    if (ev.getClickCount() <= 1) {
			ui.mouseReleased(ev);
		    }
		}
	    });
	} catch (Exception ex) {
	    Log.debug(_class, ex.getMessage());
	}
    }

    /**
     * Choose the tree to display.  The list of trees is
     * specified by the <TT>TREE</TT> property.
     *
     * @param treeIndex Index of the tree to display.  If the specified
     *		tree does not exist, nothing happens.
     *
     * @see com.sgi.sysadm.ui.treeView.TreeViewProperties#TREE
     */
    public void setTree(int treeIndex) {
	TreeLevel levels[] = (TreeLevel[])_trees.elementAt(treeIndex);
	TreeInfoStruct treeInfo =
	    (TreeInfoStruct)_treeInfo.elementAt(treeIndex);

	final boolean showRoot = treeInfo.showRoot;
	Log.debug(_class, "setting tree " + treeIndex + " showRoot: " +
		  showRoot);

	stopMonitoring();
	if (_tc._rootItem != null) {
	    _tc._tree.setRootVisible(showRoot);
	} else {
	    _tc._tree.setRootVisible(false);
	}

	_treeIndex = treeIndex;

	_rootNode = TreeCategoryListener.createNode(_tc, levels, 0,
						    _tc._rootItem, _prefix);

	_category.addCategoryListener(_listener, _notifyFilter);
    }

    /**
     * Choose the tree to display.  The list of trees is
     * specified by the <TT>TREE</TT> property.
     *
     * @param treeName name of the tree to display.  If the tree by that
     *		name cannot be found, nothing happens.
     *
     * @see com.sgi.sysadm.ui.treeView.TreeViewProperties#TREE
     */
    public void setTree(String treeName) {
	for (int ii = 0; ii < _treeInfo.size(); ++ii) {
	    TreeInfoStruct treeInfo = (TreeInfoStruct)_treeInfo.elementAt(ii);
	    if (treeInfo.name.equals(treeName)) {
		setTree(ii);
		return;
	    }
	}

	// caller should not allow this to happen
	//
	Log.assert(false, "Cannot find tree " + treeName);
    }

    /**
     * Change the root of the tree.
     *
     * @param rootItem the new item to be at the root of the tree.
     */
    public void setRoot(Item rootItem)
    {
	TreeLevel levels[] = (TreeLevel[])_trees.elementAt(_treeIndex);

	_notifyFilter = new NotificationFilter();

	if (rootItem != null) {
	    String categoryName =
		ResourceStack.getClassName(levels[0]._category);
	    Log.assert(categoryName.equals(rootItem.getType()),
		       "Type of new item (" + rootItem.getType() +
		       ") does not match root category of tree " + _treeIndex +
		       " (" + categoryName + ")");
	    _notifyFilter.monitorItem(rootItem.getSelector());
	}

	_tc._rootItem = rootItem;

	setTree(_treeIndex);
    }

    /**
     * Get the names of the trees.  The list of trees is
     * specified by the <TT>TREE</TT> property.
     *
     * @return An array of tree names
     *
     * @see com.sgi.sysadm.ui.treeView.TreeViewProperties#TREE
     */
    public String[] getTreeNames() {
	String[] names = new String[_treeInfo.size()];
	for (int ii = 0; ii < _treeInfo.size(); ++ii) {
	    TreeInfoStruct treeInfo = (TreeInfoStruct)_treeInfo.elementAt(ii);
	    names[ii] = treeInfo.name;
	}
	return(names);
    }

    /**
     * Expand all of the nodes in the tree.
     */
    public void expandAll() {
	// It's important that getRowCount() be evaluated each time
	// through the loop.
	for (int row = 0; row < _tc._tree.getRowCount(); row++) {
	    _tc._tree.expandRow(row);
	}
    }

    /**
     * Collapse all of the nodes in the tree.
     */
    public void collapseAll() {
	// It's important that getRowCount() be evaluated each time
	// through the loop.
	// We start at one so that we don't collapse the root node.
	for (int row = 1; row < _tc._tree.getRowCount(); row++) {
	    _tc._tree.collapseRow(row);
	}
    }

    /**
     * Set the auto expand flag.  When <TT>autoExpand</TT> is true,
     * parent nodes automatically expand when children are added, and
     * <TT>expandAll</TT> is called immediately.  By default, nodes
     * are automatically expanded.
     *
     * @param autoExpand Whether or not to automatically expand child
     *                   nodes.
     *
     * @see #expandAll()
     * @see #collapseAll()
     */
    public void setAutoExpand(boolean autoExpand) {
	_tc._autoExpand = autoExpand;
	if (autoExpand) {
	    expandAll();
	} else {
	    collapseAll();
	}
    }

    /**
     * Get the current value of the autoExpand flag.
     *
     * @return Current value of the autoExpand flag.
     */
    public boolean getAutoExpand() {
	return _tc._autoExpand;
    }

    /**
     * Get the JTree.
     *
     * @return The JTree displayed in the TreeViewPane.
     */
    public JTree getJTree() {
	return _tc._tree;
    }

    /**
     * Adds a listener for TreeSelection events.
     *
     * @param listener the TreeSelectionListener that will be notified when
     *            a node is selected or deselected (a "negative selection")
     *
     * @see javax.swing.JTree
     */
    public void addTreeSelectionListener(TreeSelectionListener listener) {
	_tc._tree.addTreeSelectionListener(listener);
    }

    /**
     * Remove a TreeSelection listener.
     *
     * @param listener TreeSelectionListener to remove.
     *
     * @see javax.swing.JTree
     */
    public void removeTreeSelectionListener(TreeSelectionListener listener) {
	_tc._tree.removeTreeSelectionListener(listener);
    }

    /**
     * Add a listener that gets called when an action is performed
     * on a node in the tree.  This event is triggered by the user
     * single-clicking on a node in the tree.
     *
     * @param listener listener to add.
     */
    public void addActionListener(ActionListener listener) {
	_actionListeners.addElement(listener);
    }

    /**
     * Remove an Action listener.
     *
     * @param listener listener to remove.
     */
    public void removeActionListener(ActionListener listener) {
	_actionListeners.removeElement(listener);
    }

    /**
     * Notify ActionListeners that an action has been performed.
     *
     * @param event ActionEvent to distribute.
     */
    private void notifyAction(ActionEvent event) {
	Enumeration enum = _actionListeners.elements();
	while (enum.hasMoreElements()) {
	    ((ActionListener)enum.nextElement()).actionPerformed(event);
	}
    }

    /**
     * Recursively stop monitoring categories for <tt>node</tt> and
     * its descendants.
     *
     * @param node Node to stop monitoring.
     */
    private void stopMonitoring() {
	TreeModel model = _tc._tree.getModel();
	if (model != null) {
	    DefaultMutableTreeNode node =
		(DefaultMutableTreeNode)model.getRoot();
	    try {
		ItemUserObject obj = (ItemUserObject)node.getUserObject();
		obj.dispose();
	    } catch (ClassCastException ex) {
	    }
	    _category.removeCategoryListener(_listener);
	}
    }

    /**
     * Build an array of property names to look up.  First look up
     * the user-specified property, then fall back to the default
     * property name.
     */
    private String[] propNames(String resource) {
	return new String[] {
	    _prefix + resource,
	    TREEVIEWPANE_PROPERTY_PREFIX + resource
	};
    }

    private ItemTester getItemTester(String levelString) {

	String testerClassName =
	    _rs.getString(levelString + ITEM_TESTER, null);

	if (testerClassName == null) {
	    return null;
	}

	Log.debug(_class,
		  "Loading " + testerClassName);

	ItemTester tester =
	    (ItemTester)_testers.get(testerClassName);
	if (tester != null) {
	    return tester;
	}

	Log.debug(_class,
		  (testerClassName == null ? "no tester" :
		   testerClassName));

	try {
	    tester = (ItemTester)SysUtil.createObject(
		testerClassName,
		ItemTester.class,
		null, null);
	    _testers.put(testerClassName, tester);
	    return tester;

	} catch (Exception exception) {
	    Log.error(_class,
		      "Could not load " + testerClassName +
		      " for use as an ItemTester");
	}

	return null;
    }

    /**
     * Get an ItemComparator for this category.  This method first
     * searches for the property {treename}.{categoryname}.itemCompare
     * that describes the ItemComparator class to load.
     * If that class is not found, we return an ItemComparator that
     * compares the selectors.
     *
     * @return An ItemComparator for the category.
     */
    private ItemComparator getItemComparator(String levelString,
					     TreeLevel level) {

	String resName =
	    _rs.getString(levelString + ITEM_COMPARATOR, null);

	ItemComparator comparator = loadItemComparator(resName);
	if (comparator != null) {
	    return comparator;
	}

	resName =
	    _prefix + "." + level._category + ITEM_COMPARATOR;

	Log.debug(_class,
		  "Looking for " + resName);

	String comparatorClassName = _rs.getString(resName, null);

	comparator = loadItemComparator(comparatorClassName);
	if (comparator != null) {
	    return comparator;
	}

	// Since no task-specific ItemComparator exists, return an
	// ItemComparator that compares the selectors
	return _defaultComparator;
    }

    private ItemComparator loadItemComparator(String comparatorClassName) {

	if (comparatorClassName == null) {
	    return null;
	}

	Log.debug(_class,
		  "Loading " + comparatorClassName);

	ItemComparator comparator =
	    (ItemComparator)_comparators.get(comparatorClassName);
	if (comparator != null) {
	    return comparator;
	}

	Log.debug(_class,
		  (comparatorClassName == null ? "no comparator" :
		   comparatorClassName));

	try {
	    comparator = (ItemComparator)SysUtil.createObject(
		comparatorClassName,
		ItemComparator.class,
		null, null);
	    _comparators.put(comparatorClassName, comparator);
	    return comparator;

	} catch (Exception exception) {
	    Log.error(_class,
		      "Could not load " + comparatorClassName +
		      " for use as an ItemComparator");
	}

	return null;
    }

    /**
     * Dereference CategoryListeners so we get garbage-collected.
     *
     * @see com.sgi.sysadm.ui.ExtraCleanup
     */
    public void extraCleanup() {
	stopMonitoring();
    }
}
