//
// AttrBundle.java
//
//	Collection of Attributes with change notification and
//	serialization.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.util;

import java.util.*;
import java.text.Collator;

/**
 * AttrBundle is an aggregation of Attributes, which are typed
 * key/value pairs.  Methods are provided for getting and setting the
 * values of Attributes, for getting notification when Attributes
 * change, and for managing a string representation of an AttrBundle
 * that can be used to transfer an AttrBundle to or from another
 * program.
 * <p>
 * In addition to Attributes, each AttrBundle has a type and a
 * selector, both of which are Strings.  Type and selector are used by
 * subclasses of AttrBundle for uniquely identifying items and for
 * routing packets.
 * <p>	
 * An AttrBundle stores an independent mapping of key names to
 * visibility flags, controlled via the <tt>setAttrVisible</tt>
 * method.  The visibility mapping is used in the <tt>toString</tt>
 * method to prevent invisible Attributes from being part of the
 * String returned.  This facility is used to prevent sensitive
 * information such as passwords from appearing in logging or debug
 * output.
 * @see Attribute
 */
public class AttrBundle {

    private String _type;
    private String _selector;
    private Hashtable _attrs = new Hashtable();
    private Hashtable _invisibleAttrs = new Hashtable();
    private Vector _listeners = new Vector();
    static private Vector _escapeTable = new Vector();
    private static final String VERSION = "A";
    private static final Collator collator = Collator.getInstance();
    

    // Element in our escape table.  The escape table maps certain
    // characters that are significant in the parsing of a serialized
    // AttrBundle to escape sequences.
    static private class Escape {
	private char _toEscape;
	private String _substitution;

	public Escape(char toEscape, String substitution) {
	    _toEscape = toEscape;
	    _substitution = substitution;
	}
    }

    // Initialize our escape table.  If a key or a value contains any
    // of the _toEscape characters in the table, they are replaced
    // with the corresponding _substitution when transferred across a
    // Connection.  We do this so that keys and values can contain
    // characters which are significant in the format we use to
    // transfer keys and values from one process to another.
    static {
        // '!' separates selector from first key and keys from types.
	_escapeTable.addElement(new Escape('!', "@21"));

	// ':' separates type from selector and terminates a key/value.
	// pair
	_escapeTable.addElement(new Escape(':', "@3a"));

        // '=' separates types from values.
	_escapeTable.addElement(new Escape('=', "@3d"));

	// '@' introduces escape strings.
	_escapeTable.addElement(new Escape('@', "@40"));

	// Escape newlines for benefit of Packet class.
	_escapeTable.addElement(new Escape('\n', "@0a"));
    }	

    /**
     * Construct an AttrBundle with <TT>type</TT> and
     * <TT>selector</TT>.
     * 
     * @param type The type of this AttrBundle.  The interpretation of
     *             <tt>type</tt> depends on the application.
     * @param selector The selector of this AttrBundle.  The
     *                 interpretation of <tt>selector</tt> depends on
     *                 the application.
     */
    public AttrBundle(String type, String selector) {
	_type = type;
	_selector = selector;
    }

    /**
     * Construct an AttrBundle with empty type and selector.
     */
    public AttrBundle() {
	_type = "";
	_selector = "";
    }

    /**
     * Construct an AttrBundle from serialized format.
     * @param stream serialized AttrBundle.
     */
    public AttrBundle(String stream) {
	// Serialized format is:
	// "version:type:selector:inviskey1:inviskey2...!key1!type1=value1:key2!type2=value2...".
	// Any embedded characters from the set [!:=@\n] are replaced
	// by their escaped equivalents @21, @3a, @3d, @25, and @0a.
	// The attributes are optional.  If any portion of the string
	// is mal-formed, the corresponding Attribute will not be set.
	// Two examples of serialized AttrBundles are:
	// 
	//  1:url:file@3a//sgi.com/etc/motd
	//        type        url
	//        selector    file://sgi.com/etc/motd
	//        attributes  none
	//  1:url:http@3a//www.sgi.com/canvas.gif!mimeType!s=image/gif
	//        type        url
	//        selector    http://www.sgi.com/canvas.gif
	//       attributes  mimeType (string) =image/gif
	int streamLen = stream.length();
	int start = 0;

	int stop = stream.indexOf(':', start);
	if (stop == -1) {
	    return;
	}
	// For now, we just ignore the version.  If the version is ever
	// revved, we can use the version to be backwards compatible.
	start = stop + 1;
			    
	stop = stream.indexOf(':', start);
	if (stop == -1) {
	    return;
	}
	_type = unescape(stream.substring(start, stop));
	start = stop + 1;

	stop = stream.indexOf(':', start);
	if (stop == -1) {
	    stop = streamLen;
	}
	_selector = unescape(stream.substring(start, stop));
	start = stop + 1;

	int stopInvisibleAttrs = stream.indexOf('!', start);
	if (stopInvisibleAttrs != -1) {
	    while (start < stopInvisibleAttrs) {
		stop = stream.indexOf(':', start);
		if (stop == -1) {
		    break;
		}
		setAttrVisible(unescape(stream.substring(start, stop)), false);
		start = stop + 1;
	    }
	}
	start = stopInvisibleAttrs + 1;
	
	while (start < streamLen) {
	    stop = stream.indexOf('!', start);
	    if (stop == -1) {
		return;
	    }

	    String key = unescape(stream.substring(start, stop));
	    start = stop + 1;

	    stop = stream.indexOf('=', start);
	    if (stop == -1) {
		return;
	    }

	    String type = unescape(stream.substring(start, stop));
	    start = stop + 1;

	    stop = stream.indexOf(':', start);
	    if (stop == -1) {
		stop = streamLen;
	    }

	    String value = unescape(stream.substring(start, stop));
	    start = stop + 1;

	    setAttr(new Attribute(key, type, value));
	}
    }

    /**
     * Construct an AttrBundle that is a copy of <TT>other</TT>.  The
     * list of listeners is not copied.  The visibility flags are
     * copied.
     * 
     * @param other AttrBundle to copy.
     */
    public AttrBundle(AttrBundle other) {
	_type = other._type;
	_selector = other._selector;
	_attrs = (Hashtable)other._attrs.clone();
	_invisibleAttrs = (Hashtable)other._invisibleAttrs.clone();
    }

    /**
     * Create a new AttrBundle that is a copy of this AttrBundle.  The
     * list of listeners is not copied.  The visibility flags are
     * copied.
     * 
     * @return A copy of this AttrBundle
     */
    public Object clone() throws CloneNotSupportedException {
	return new AttrBundle(this);
    }

    /**
     * Add a listener to get notified when Attributes are added,
     * changed, or removed.  <tt>listener</tt> does not get notified
     * about the current state of the AttrBundle.
     * <p>
     * It is safe to add an AttrListener while AttrListeners are being
     * notified.  The added AttrListener will not receive the
     * notification that was being processed when it was added.
     * 
     * @param listener gets notified when Attributes are added,
     *                 changed, or removed.
     */
    public void addAttrListener(AttrListener listener) {
	_listeners.addElement(listener);
    }
	
    /**
     * Remove a listener.
     * <p>
     * It is safe to remove an AttrListener while AttrListeners are
     * being notified.  In this case, however, it is possible for the
     * removed AttrListener to get one last notification even after it
     * has been removed.
     * 
     * @param listener listener to remove.
     */
    public void removeAttrListener(AttrListener listener) {
	_listeners.removeElement(listener);
    }

    /**
     * Get the type of this AttrBundle.
     * 
     * @return type of this AttrBundle.
     */
    public String getType() {
	return _type;
    }

    /**
     * Get the selector of this AttrBundle.
     * 
     * @return selector of this AttrBundle.
     */
    public String getSelector() {
	return _selector;
    }

    /**
     * Return a string representation of this AttrBundle, suitable for
     * transmission to another program.
     * 
     * @return String representation of this AttrBundle.
     */
    public String serialize() {
	StringBuffer stream = new StringBuffer();
	stream.append(escape(VERSION));
	stream.append(':');
	stream.append(escape(_type));
	stream.append(':');
	stream.append(escape(_selector));
	stream.append(':');

	Enumeration invisibleAttrs = _invisibleAttrs.elements();
	while (invisibleAttrs.hasMoreElements()) {
	    String key = (String)invisibleAttrs.nextElement();
	    stream.append(escape(key));
	    stream.append(':');
	}
	stream.append('!');

	Enumeration attrs = _attrs.elements();
	while (attrs.hasMoreElements()) {
	    Attribute attr = (Attribute)attrs.nextElement();
	    stream.append(escape(attr.getKey()));
	    stream.append('!');
	    stream.append(escape(attr.getTypeString()));
	    stream.append('=');
	    stream.append(escape(attr.getValueString()));
	    stream.append(':');
	}
	// Get rid of the last separator.
	stream.setLength(stream.length() - 1);
	return stream.toString();
    }

    /**
     * Return a string representation of this AttrBundle, suitable for
     * human viewing.
     * 
     * @param pad number of spaces to pad the left side of displayed
     *            Attributes.
     * 
     * @return String representation of this AttrBundle.
     */
    public String toString(int pad) {

	// Set up padding.
	StringBuffer padStr = new StringBuffer(pad + 2);
	for (int ii = 0; ii < pad; ii++) {
	    padStr.append(' ');
	}

	// Sort keys. Concat sorted keys and values into stream and return
	// the string.
  	Enumeration keys = _attrs.keys();
	String[] keysArray = new String[_attrs.size()];
	int numVisibleAttrs = 0;

	// Populate keysArray with keys
	while (keys.hasMoreElements()) {
	    String key = (String)keys.nextElement();
	    if (getAttrVisible(key)) {
		keysArray[numVisibleAttrs++] = key;
	    }
	}

	// Sort keysArray
	Sort.stable_sort(keysArray, 0, numVisibleAttrs, new BinaryPredicate() {
	    public boolean apply(Object x, Object y) {
		return collator.compare((String)x, (String)y) < 0;
	    }	
	});

	// Calculate maximum key length.
	StringBuffer stream = new StringBuffer();
	stream.append('\n');
	int maxKeyLength = 0;
	for (int ii = 0; ii < numVisibleAttrs; ii++) {
	    int keyLength = keysArray[ii].length();
	    if (keyLength > maxKeyLength) {
		maxKeyLength = keyLength;
	    }
	}
	
	stream.append(padStr + _type + ": " + _selector);
	padStr.append("  ");

	// For each key, append the key and its value to stream.
	for (int ii = 0; ii < numVisibleAttrs; ii++) {
	    stream.append('\n');
	    String key = keysArray[ii];
	    stream.append(padStr);
	    stream.append(key);
	    
	    for (int j=0; j < maxKeyLength-key.length(); j++) {
		stream.append(' ');
	    }
	    
	    stream.append(": ");

	    // Get value for key
	    Attribute a = (Attribute)_attrs.get(key);
	    Object value = a.getValue();
	    if (value instanceof AttrBundle) {
		stream.append(((AttrBundle)value).toString(pad + 2));
	    } else {
		stream.append(a.getValueString());
	    }
	}
	
	return stream.toString();
    }

    /**
     * Return a string representation of this AttrBundle,
     * suitable for human viewing.
     * 
     * @return String representation of this AttrBundle.
     */
    public String toString() {
	return toString(0);
    }

    /**
     * Set the visibility of Attribute named by <tt>key</tt>.
     * The visibility flags are separate from the Attributes
     * themselves, so a call to <tt>setAttrVisible</tt> permanently
     * affects the visibility of Attributes for <tt>key</tt>,
     * regardless of the order in which and how many times they are
     * added or removed.
     * 
     * @param key Key to set visibility of.
     * @param visible <tt>true</tt> if Attribute is to be
     *                visible, <tt>false</tt> otherwise.
     */
    public void setAttrVisible(String key, boolean visible) {
	if (visible) {
	    _invisibleAttrs.remove(key);
	} else {
	    _invisibleAttrs.put(key, key);
	}
    }
    
    /**
     * Get the visibility of Attribute named by <tt>key</tt>.
     * 
     * @param key Key of Attribute to check visibility of.
     * 
     * @return visibility of Attribute named by "key".
     */
    public boolean getAttrVisible(String key) {
	return _invisibleAttrs.get(key) == null;
    }

    /**
     * Get the Attribute associated with <TT>key</TT>.
     * 
     * @param key key of Attribute to get.
     * 
     * @return Attribute associated with <TT>key</TT>, or null if the
     *         Attribute does not exist.
     */
    public Attribute getAttr(String key) {
	return (Attribute)_attrs.get(key);
    }

    /**
     * Remove the Attribute associated with <TT>key</TT>.
     * 
     * @param key key of Attribute to remove.
     * 
     * @return Attribute associated with <TT>key</TT>, or null if the
     *         Attribute does not exist.
     */
    public Attribute removeAttr(String key) {
	Attribute attr =  (Attribute)_attrs.remove(key);
	notifyRemoved(attr);
	return attr;
    }


    /**
     * Removes all the Attributes in the bundle.
     */
    public void removeAllAttrs() {
	Enumeration keys = _attrs.keys();
	while (keys.hasMoreElements()) {
	    removeAttr((String)(keys.nextElement()));
	}
    }
    
    /**
     * Set an Attribute.  Replaces any existing Attribute that had the
     * same key as <TT>attr</TT>.  Notify AttrListeners that an
     * attribute has changed.
     * <p>
     * It is an error to try to change the type of an Attribute.  If
     * an Attribute already exists with the same key as <TT>attr</TT>
     * but with a different type (e.g. String v. Long), an assertion
     * (<tt>Log.assert</tt>) will fail.
     * <p>
     * If an Attribute with the same key as <TT>attr</TT> already
     * exists with an identical value, the old Attribute is left in
     * place and no change notification occurs.
     * 
     * @param attr Attribute to set.
     * @see Log#assert
     */
    public void setAttr(Attribute attr) {
	Attribute old = getAttr(attr.getKey());
	Log.assert(old == null
		   || attr.getTypeString().equals(old.getTypeString()),
		   "Attempt to change attribute type");
	if (old == null) {
	    _attrs.put(attr.getKey(), attr);
	    notifyAdded(attr);
	} else if (!old.equals(attr)) {
	    _attrs.put(attr.getKey(), attr);
	    notifyChanged(attr);
	}
    }

    /**
     * Convenience method for setting a String attribute.  Everything
     * in the description of <tt>setAttr</tt> applies to this method as
     * well.
     * 
     * @param key key of the attribute to set.
     * @param value String value of the attribute to set.
     * @see #setAttr
     */
    public void setString(String key, String value) {
	setAttr(new Attribute(key, value));
    }

    /**
     * Convenience method for setting a long attribute.  Everything
     * in the description of <tt>setAttr</tt> applies to this method as
     * well.
     * 
     * @param key key of the attribute to set.
     * @param value long value of the attribute to set.
     * @see #setAttr
     */
    public void setLong(String key, long value) {
	setAttr(new Attribute(key, value));
    }

    /**
     * Convenience method for setting a double attribute.  Everything
     * in the description of <tt>setAttr</tt> applies to this method as
     * well.
     * 
     * @param key key of the attribute to set.
     * @param value double value of the attribute to set.
     * @see #setAttr
     */
    public void setDouble(String key, double value) {
	setAttr(new Attribute(key, value));
    }

    /**
     * Convenience method for setting a boolean attribute.  Everything
     * in the description of <tt>setAttr</tt> applies to this method as
     * well.
     * 
     * @param key key of the attribute to set.
     * @param value boolean value of the attribute to set.
     * @see #setAttr
     */
    public void setBoolean(String key, boolean value) {
	setAttr(new Attribute(key, value));
    }

    /**
     * Convenience method for setting an AttrBundle attribute.  Everything
     * in the description of <tt>setAttr</tt> applies to this method as
     * well.
     * 
     * @param key key of the attribute to set.
     * @param value AttrBundle value of the attribute to set.
     * @see #setAttr
     */
    public void setBundle(String key, AttrBundle value) {
	setAttr(new Attribute(key, value));
    }

    /**
     * Convenience method for getting the value of an Attribute of
     * type String.  This method will throw a ClassCastException if
     * an Attribute for <TT>key</TT> exists but is not of type String.
     * This method also asserts (using <tt>Log.assert</tt>) that the
     * Attribute has previously been set.
     * 
     * @param key key of the Attribute to get the String value of.
     * 
     * @return value of Attribute corresponding to key.
     * @see Log#assert
     */
    public String getString(String key) {
	Attribute attr = getAttr(key);
	if (attr == null) {
	    Log.fatal("Attempt to get value of uninitialized Attribute: " + key);
	}
	return attr.stringValue();
    }
    
    /**
     * Convenience method for getting the String value of an Attribute of
     * any type. 
     * This method asserts (using <tt>Log.assert</tt>) that the
     * Attribute has previously been set.
     * 
     * @param key key of the Attribute to get the String representation of.
     * 
     * @return value of Attribute corresponding to key.
     * @see Log#assert
     */
    public String getValueString(String key) {
	Attribute attr = getAttr(key);
	Log.assert(attr != null,
		   "Attempt to get value of uninitialized Attribute: " + key);
	return attr.getValueString();
    }

    /**
     * Convenience method for getting the value of an Attribute of
     * type long.  This method will throw a ClassCastException if
     * an Attribute for <TT>key</TT> exists but is not of type long.
     * This method also asserts (using <tt>Log.assert</tt>) that the
     * Attribute has previously been set.
     * 
     * @param key key of the Attribute to get the long value of.
     * 
     * @return value of Attribute corresponding to key.
     * @see Log#assert
     */
    public long getLong(String key) {
	Attribute attr = getAttr(key);
	Log.assert(attr != null,
		   "Attempt to get value of uninitialized Attribute: " + key);
	return attr.longValue();
    }

    /**
     * Convenience method for getting the value of an Attribute of
     * type double.  This method will throw a ClassCastException if
     * an Attribute for <TT>key</TT> exists but is not of type double.
     * This method also asserts (using <tt>Log.assert</tt>) that the
     * Attribute has previously been set.
     * 
     * @param key key of the Attribute to get the double value of.
     * 
     * @return value of Attribute corresponding to key.
     * @see Log#assert
     */
    public double getDouble(String key) {
	Attribute attr = getAttr(key);
	Log.assert(attr != null,
		   "Attempt to get value of uninitialized Attribute: " + key);
	return attr.doubleValue();
    }

    /**
     * Convenience method for getting the value of an Attribute of
     * type boolean.  This method will throw a ClassCastException if
     * an Attribute for <TT>key</TT> exists but is not of type boolean.
     * This method also asserts (using <tt>Log.assert</tt>) that the
     * Attribute has previously been set.
     * 
     * @param key key of the Attribute to get the boolean value of.
     * 
     * @return value of Attribute corresponding to key.
     * @see Log#assert
     */
    public boolean getBoolean(String key) {
	Attribute attr = getAttr(key);
	Log.assert(attr != null,
		   "Attempt to get value of uninitialized Attribute: " + key);
	return attr.booleanValue();
    }
    
    /**
     * Convenience method for getting the value of an Attribute of
     * type AttrBundle.  This method will throw a ClassCastException if
     * an Attribute for <TT>key</TT> exists but is not of type AttrBundle.
     * This method also asserts (using <tt>Log.assert</tt>) that the
     * Attribute has previously been set.
     * 
     * @param key key of the Attribute to get the AttrBundle value of.
     * 
     * @return value of Attribute corresponding to key.
     * @see Log#assert
     */
    public AttrBundle getBundle(String key) {
	Attribute attr = getAttr(key);
	Log.assert(attr != null,
		   "Attempt to get value of uninitialized Attribute: " + key);
	return attr.bundleValue();
    }

    /**
     * Get the number of Attributes in this AttrBundle.
     * 
     * @return Number of Attributes.
     */
    public int size() {
	return _attrs.size();
    }

    /**
     * Get an Enumeration of the Attributes in this AttrBundle.
     * 
     * @return Enumeration of Attributes.
     */
    public Enumeration getAttrEnum() {
	return _attrs.elements();
    }

    /**
     * Determine whether <tt>other</tt> is equal to this AttrBundle.
     * AttrBundles are equal if they have the same type, the same
     * selector, the same set of keys, and the Attribute for each key
     * is of the same type and has the same value.  The visibility
     * flags are not considered.  If <tt>other</tt> is <tt>null</tt>,
     * <tt>equals</tt> returns <tt>false</tt>.
     * 
     * @param other AttrBundle to compare.
     * 
     * @return <tt>true</tt> if AttrBundles are equal, <tt>false</tt>
     *         otherwise.
     */
    public boolean equals(AttrBundle other) {
	if (other == null) {
	    return false;
	}
	if (!_type.equals(other._type)) {
	    return false;
	}
	    
	if (!_selector.equals(other._selector)) {
	    return false;
	}

	if (_attrs.size() != other.size()) {
	    return false;
	}
	
	Enumeration attrs = other._attrs.elements();
	Attribute attr, myAttr;
	while (attrs.hasMoreElements()) {
	    attr = (Attribute) attrs.nextElement();
	    myAttr = (Attribute) _attrs.get(attr.getKey());
	    if (myAttr == null || !myAttr.equals(attr)) {
		return false;
	    }
	}
	return true;
    }

    /**
     * Create a copy of <TT>raw</TT>, with special characters replaced
     * with their escape strings.
     * 
     * @param raw raw string.
     * 
     * @return copy of <TT>raw</TT> with special characters escaped.
     */
    private String escape(String raw) {
	StringBuffer escaped = new StringBuffer();
	for (int i = 0; i < raw.length(); i++) {
	    int match = 0;
	    for (; match < _escapeTable.size(); match++) {
		Escape escape = (Escape)_escapeTable.elementAt(match);
		if (raw.charAt(i) == escape._toEscape) {
		    escaped.append(escape._substitution);
		    break;
		}
	    }

	    if (match == _escapeTable.size()) {
	        escaped.append(raw.charAt(i));
	    }
	}
	return escaped.toString();
    }

    /**
     * Create a copy of <TT>raw</TT>, with escape strings replaced
     * with the special characters they represent.
     * 
     * @param raw raw string.
     * 
     * @return copy of <TT>raw</TT> with escape sequences turned back
     *         into special characters.
     */
    private String unescape(String raw) {
	StringBuffer unescaped = new StringBuffer();
	for (int i = 0; i < raw.length();) {
	    if (raw.charAt(i) != '@') {
		unescaped.append(raw.charAt(i++));
		continue;
	    }
	    int match = 0;
	    for (; match < _escapeTable.size(); match++) {
		Escape escape = (Escape)_escapeTable.elementAt(match);
		if (raw.regionMatches(i, escape._substitution,
				      0, escape._substitution.length())) {
		    unescaped.append(escape._toEscape);
		    i += escape._substitution.length();
		    break;
		}
	    }
	    Log.assert(match < _escapeTable.size(),
		       "Unrecognized escape sequence parsing AttrBundle");
	}
	return unescaped.toString();
    }

    /**
     * Notify listeners that an Attribute has changed.
     * 
     * @param attr the attribute that changed.
     */
    private void notifyChanged(Attribute attr) {
	AttrEvent event = new AttrEvent(this, attr);
	// clone _listeners so that listener adds or removes during
	// notification do not mess things up.
	Enumeration enum = ((Vector)_listeners.clone()).elements();
	while (enum.hasMoreElements()) {
	    AttrListener listener = (AttrListener)enum.nextElement();
	    listener.attrChanged(event);
	}
    }
    
    /**
     * Notify listeners that an Attribute has been deleted
     * 
     * @param attr the attribute that was deleted
     */
    private void notifyRemoved(Attribute attr) {
	AttrEvent event = new AttrEvent(this, attr);
	// clone _listeners so that listener adds or removes during
	// notification do not mess things up.
	Enumeration enum = ((Vector)_listeners.clone()).elements();
	while (enum.hasMoreElements()) {
	    AttrListener listener = (AttrListener)enum.nextElement();
	    listener.attrRemoved(event);
	}
    } 

    /**
     * Notify listeners that an Attribute has been added
     * 
     * @param attr the attribute that was deleted
     */
    private void notifyAdded(Attribute attr) {
	AttrEvent event = new AttrEvent(this, attr);
	// clone _listeners so that listener adds or removes during
	// notification do not mess things up.
	Enumeration enum = ((Vector)_listeners.clone()).elements();
	while (enum.hasMoreElements()) {
	    AttrListener listener = (AttrListener)enum.nextElement();
	    listener.attrAdded(event);
	}
    }
}
