//
// Attribute.java
//
//	Typed key/value pair.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.util;

/**
 * Attribute is a typed key/value pair.  The set of types
 * which the value of an Attribute can take on is closed; an
 * Attribute is of type <tt>String</tt>, <tt>long</tt>,
 * <tt>boolean</tt>, <tt>double</tt>, or <tt>AttrBundle</tt>.
 * <p>
 * Attributes can be aggregated into AttrBundles.
 * @see AttrBundle
 */
public class Attribute {

    private String _key;
    private Object _value;

    private static final String STRING = "s";
    private static final String LONG = "l";
    private static final String BOOLEAN = "b";
    private static final String DOUBLE = "d";
    private static final String BUNDLE = "a";

    /**
     * Construct an Attribute of type long.
     * 
     * @param key Key for this Attribute.
     * @param value long value for this Attribute.
     */
    public Attribute(String key, long value) {
	_key = key;
	_value = new Long(value);
    }

    /**
     * Construct an Attribute of type boolean.
     * 
     * @param key Key for this Attribute.
     * @param value boolean value for this Attribute.
     */
    public Attribute(String key, boolean value) {
	_key = key;
	_value = new Boolean(value);
    }

    /**
     * Construct an Attribute of type String.
     * 
     * @param key Key for this Attribute.
     * @param value String value for this Attribute.
     */
    public Attribute(String key, String value) {
	_key = key;
	_value = value;
    }

    /**
     * Construct an Attribute of type double.
     * 
     * @param key Key for this Attribute.
     * @param value double value for this Attribute.
     */
    public Attribute(String key, double value) {
	_key = key;
	_value = new Double(value);
    }

    /**
     * Construct an Attribute of type AttrBundle
     * 
     * @param key Key for this Attribute.
     * @param bundle AttrBundle value for this Attribute.
     */
    public Attribute(String key, AttrBundle bundle) {
	_key = key;
	_value = new AttrBundle(bundle);
    }

    /**
     * Construct an Attribute from String representations of type and
     * value.  This is used by AttrBundle to construct Attributes out
     * of a serialized stream representation.
     * 
     * @param key key for this Attribute.
     * @param type string representation of the type of this Attribute.
     * @param value string representation of the value of this Attribute.
     */
    public Attribute(String key, String type, String value) {
	_key = key;
	if (type.equals(STRING)) {
	    _value = value;
	} else if (type.equals(LONG)) {
	    _value = new Long(value);
	} else if (type.equals(BOOLEAN)) {
	    _value = new Boolean(value);
	} else if (type.equals(DOUBLE)) {
	    _value = new Double(value);
	} else if (type.equals(BUNDLE)) {
	    _value = new AttrBundle(value);
	} else {
	    Log.fatal("Illegal Attribute type");
	}
    }

    /**
     * Construct an attribute from a name and a value object
     *
     * @param key key for this Attribute.
     * @param type string representation of the type of this Attribute.
     */
    public Attribute(String key, Object value) {
	_key = key;
	_value = value;
    }
    
    /**
     * Compare two Attributes to see if they are equal.  Two
     * Attributes are equal if they have the same key, type,
     * and value.
     * 
     * @param other Attribute to check for equality.
     * 
     * @return <tt>true</tt> if Attributes are equal,
     *         <tt>false</tt> otherwise.
     */
    public boolean equals(Attribute other) {
	if (other == null) {
	    return false;
	}
	return _key.equals(other._key) && _value.equals(other._value);
    }

    /**
     * Get the key of this Attribute.
     * 
     * @return key of this Attribute.
     */
    public String getKey() {
	return _key;
    }

    /**
     * Get the value of this Attribute.
     * 
     * @return value of this Attribute.
     */
    public Object getValue() {
	return _value;
    }

    /**
     * Get the value of this Attribute as a String.  This method will
     * throw a ClassCastException if this Attribute is not of type
     * String.
     * 
     * @return String value of this Attribute.
     */
    public String stringValue() {
	return (String)_value;
    }

    /**
     * Get the value of this Attribute as a long.  This method will
     * throw a ClassCastException if this Attribute is not of type
     * long.
     * 
     * @return long value of this Attribute.
     */
    public long longValue() {
	return ((Long)_value).longValue();
    }

    /**
     * Get the value of this Attribute as a boolean.  This method will
     * throw a ClassCastException if this Attribute is not of type
     * boolean.
     * 
     * @return boolean value of this Attribute.
     */
    public boolean booleanValue() {
	return ((Boolean)_value).booleanValue();
    }

    /**
     * Get the value of this Attribute as a double.  This method will
     * throw a ClassCastException if this Attribute is not of type
     * double.
     * 
     * @return double value of this Attribute.
     */
    public double doubleValue() {
	return ((Double)_value).doubleValue();
    }

    /**
     * Get the value of this Attribute as an AttrBundle.  This method
     * will throw a ClassCastException if this Attribute is not of
     * type AttrBundle.
     * 
     * @return AttrBundle value of this Attribute.
     */
    public AttrBundle bundleValue() {
	return (AttrBundle)_value;
    }

    /**
     * Get a string representation of the type of this Attribute.
     * This is used by AttrBundle to construct a stream representation
     * of this Attribute.
     * 
     * @return String representation of the type of this Attribute.
     */
    public String getTypeString() {
	if (_value instanceof String) {
	    return STRING;
	} else if (_value instanceof Long) {
	    return LONG;
	} else if (_value instanceof Boolean) {
	    return BOOLEAN;
	} else if (_value instanceof Double) {
	    return DOUBLE;
	} else if (_value instanceof AttrBundle) {
	    return BUNDLE;
	} else {
	    Log.fatal("Illegal type in Attribute");
	    return null;
	}
    }

    /**
     * Return a string representation of the value of this Attribute.
     * This is used by AttrBundle to construct a stream representation
     * of this Attribute.
     * 
     * @return String representation of the type of this Attribute.
     */
    public String getValueString() {
	if (_value == null) {
	    return "";
	}
	if (_value instanceof AttrBundle) {
	    return ((AttrBundle)_value).serialize();
	}
	return _value.toString();
    }
}


