//
// Log.java
//
//	Debugging and assertion support.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.util;

import java.io.*;
import java.util.Hashtable;
import java.util.StringTokenizer;

/**
 * The Log class provides debugging and assertion support.
 *
 * Debug messages may be sent at one of several different levels.
 * A fine level of control over which debugging messages to display is
 * provided via two features.  First, there are several methods for logging
 * messages according to the type of message.  Second, each non-terminating
 * logging method takes a <i>module</i> name as its
 * first argument.  A module in this context is simply a String identifying
 * a logically related set of code.  It is arbitrarily determined by the
 * programmer and is typically a class name or related group of classes.
 * <p>
 * To control the messages printed, you can set the
 * debugging levels (levels correspond to the logging methods) from the
 * SystemProperties.  The Property used is Log.debugLevel.
 * The recognized values are:
 * <ul>
 *   <li>ALL:  Enable all levels
 *   <li>NONE: Disable all levels (except for FATAL)
 *   <li>Any combination of
 *                   <ul>
 *                    <li>    T (Trace)
 *                    <li>    D (Debug)
 *                    <li>    I (Info)
 *                    <li>    W (Warning)
 *                    <li>    E (Error)
 *                    <li>    A (Assert)
 *                   </ul>
 * </ul>
 * All of the above option letters and strings are case insensitive.
 * <p>
 * For example <code>java -DLog.debugLevel=ALL test</code>  will
 * turn on all logging, while <code>java -DLog.debugLevel=WE
 * test</code> will turn on warnings and errors.
 * If the property Log.debugLevel is not set,
 * Log defaults to displaying messages of type Assert, Fatal, and Error.
 * <p>
 * After the level indication there may optionally follow a list of
 * modules.  By default messages from all modules are displayed.
 * If one or more modules are specified, the output is restricted to
 * only messages from those modules.
 * <p>
 * The modules specification is a colon followed by a comma-separated
 * list of modules.  For example
 * <code>java -DLog.debugLevel=ALL:alpha,beta test</code>
 * displays messages of any level from only the alpha and beta modules.
 */

public final class Log {
    // The values chosen here match those chosen in sysadm's C++ Log
    // class so that levels may be easily shared.  Keep them in sync.

    /** Used with leveOn()/levelOff() to enable/disable display of messages
        from the trace(), traceIn(), and traceOut() logging methods. */
    public static final int TRACE	= 0x01;	// Execution tracing
    /** Used with leveOn()/levelOff() to enable/disable display of messages
        from the debug() logging method. */
    public static final int DEBUG	= 0x02;	// Debug-only message
    /** Used with leveOn()/levelOff() to enable/disable display of messages
        from the info() logging method. */
    public static final int INFO	= 0x04;	// Informational message
    /** Used with leveOn()/levelOff() to enable/disable display of messages
        from the warning() logging method. */
    public static final int WARNING	= 0x08;	// Warning message
    /** Used with leveOn()/levelOff() to enable/disable display of messages
        from the error() logging method. */
    public static final int ERROR	= 0x10;	// Non-fatal error message
    private static final int FATAL	= 0x20;	// Fatal error
    /** Used with leveOn()/levelOff() to enable/disable display of messages
        from the assert() logging method. */
    public static final int ASSERT	= 0x40; // Conditional state checks
    private static final int ALL =
	TRACE | DEBUG | INFO | WARNING | ERROR | FATAL | ASSERT;

    // Bit mask of currently enabled levels.
    private static int _level;

    /**
     * Returns a boolean that tells if a certain debug level is turned
     * on.  This can be useful so that client code can avoid creating
     * expensive debug messages if that level isn't on.
     *
     * @param level The level the client is interested in (e.g Log.DEBUG)
     * @return Whether that level is on or not.
     */
    public static boolean isLevelOn(int level) {
	return (_level & level) != 0;
    }
    
    // _modules stores the names of current modules of interest
    private static Hashtable _modules;

    Log() {}	// keep the default constructor from showing up in Javadoc

    static {
	String level =
	    SysUtil.getSystemProperty("Log.debugLevel","EA");
	setDebugLevel(level);
    }

    // The various level methods call log() so output redirection
    // and level and module checking may be handled in one place.
    // assert() is handled separately because it doesn't have a module.
    private static void log(int level, String levelStr, String module,
			    String msg) {
	// recordLogs(3);
	if ((_level & level) != 0) {
	    if (_modules.isEmpty() || _modules.containsKey(module)) {
		String message = levelStr + module + ": " + msg;
		System.out.println(message);
	    }
	}
	// Future work: add writing to file or implementing log listeners.
    }

    // establish an initial set of modules from the given property.
    // moduleList is interpreted as a comma-separated list of modules.
    private static void setModules(String moduleList) {
	StringTokenizer st = new StringTokenizer(moduleList, ",");

	while (st.hasMoreTokens()) {
	    String token = st.nextToken();
	    addModule(token);
	}
    }

    private static void setDebugLevel(String level) {
	String all = "all";	// case insensitive
	String none = "none";	// case insensitive
	if (level.regionMatches(true, 0, all, 0, all.length())) {
	    _level = ALL;
	} else if (level.regionMatches(true, 0, none, 0, none.length())) {
	    _level = 0;
	} else {
	  loop: for (int ii = 0 ; ii < level.length() ; ii++) {
	      switch (Character.toUpperCase(level.charAt(ii))) {
	      case 'T' : _level |= TRACE;
		  break;
	      case 'D': _level |= DEBUG;
		  break;
	      case 'I': _level |= INFO;
		  break;
	      case 'W': _level |= WARNING;
		  break;
	      case 'E': _level |= ERROR;
		  break;
	      case 'A': _level |= ASSERT;
		  break;
	      case ':':
		  if (ii == 0) {	// no levels given; just modules
			_level = ERROR | ASSERT;   // use default
		  }
		  break loop;
	      default:
		  System.out.println(
			"Bad value of System Property Log.debugLevel.  " +
			"Using the default value.");
		  _level = ERROR | ASSERT;
		  break loop;
	      }
	  }
	}
	_modules = new Hashtable();
	int colon = level.indexOf(':');
	if (colon++ != -1) {
	    if (level.length() > colon) {	// colon isn't last char
		// Everything after a colon is part of a modules list
		setModules(level.substring(colon));
	    }
	}
    }

/***
    //
    //  Notify a new listener of the current state of each level.
    //
    private static void sendInitialLevels(LogListener listener) {
	for (int candidate = 1; candidate > 0; candidate <<= 1) {
	    if ((ALL & candidate) != 0) {	// this is a valid level
		listener.levelChanged(candidate, (_level & candidate) != 0);
	    }
	}
    }
***/

    /**
     ** Add a module to the set of modules to display.
     **
     ** @param module the module name to add
     **/
    public static void addModule(String module) {
	_modules.put(module, module);	// don't care what the value is
    }

    /**
     ** Remove a module from the set of modules to display.
     **
     ** @param module the module name to remove
     **/
    public static void removeModule(String module) {
	_modules.remove(module);
    }

    // Reject undefined levels.
    private static void checkLevel(int level) {
	Log.assert ((level & ~ALL) == 0, "invalid log level");
    }

    /**
     ** Enable display of a logging level.
     **
     ** @param level the level to enable
     **/
    public static void levelOn(int level) {
	checkLevel(level);
	if ((_level & level) != 0) {	// level is already on
	    return;
	}
	_level |= level;
    }

    /**
     ** Disable display of a logging level.
     **
     ** @param level the level to disable
     **/
    public static void levelOff(int level) {
	checkLevel(level);
	if ((_level & level) == 0) {	// level is already off
	    return;
	}
	_level &= ~level;
    }

    /**
     ** Log a tracing message.
     **
     ** @param module the module the message is from
     ** @param msg the message to log
     **/
    public static void trace(String module, String msg) {
	log(TRACE, "Trace    ", module, msg);
    }

    /**
     ** Log a method entry message.
     **
     ** @param module the module the message is from
     ** @param msg the message to log.  Should be the method name,
     **		and may also include other information such as parameters.
     **/
    public static void traceIn(String module, String msg) {
	log(TRACE, "TraceIn  ", module, "Entering " + msg);
    }

    /**
     ** Log a method exit message.
     **
     ** @param module the module the message is from
     ** @param msg the message to log.  Should be the method name,
     **		and may also include other information.
     **/
    public static void traceOut(String module, String msg) {
	log(TRACE, "TraceOut ", module, "Leaving " + msg);
    }

    /**
     ** Log a debug level message.
     **
     ** @param module the module the message is from
     ** @param msg the message to log
     **/
    public static void debug(String module, String msg) {
	log(DEBUG, "Debug    ", module, msg);
    }

    /**
     ** Log an informational level message.
     **
     ** @param module the module the message is from
     ** @param msg the message to log
     **/
    public static void info(String module, String msg) {
	log(INFO, "Info     ", module, msg);
    }

    /**
     ** Log a warning message.
     **
     ** @param module the module the message is from
     ** @param msg the message to log
     **/
    public static void warning(String module, String msg) {
	log(WARNING, "Warning  ", module, msg);
    }

    /**
     ** Log an error message.
     **
     ** @param module the module the message is from
     ** @param msg the message to log
     **/
    public static void error(String module, String msg) {
	log(ERROR, "Error    ", module, msg);
    }

    /**
     ** Log a message and terminate the program.
     **
     ** The message will be logged, a stack dump generated, and then
     ** <code>HostContext.abort</code> will be called.
     ** @param msg the message to log
     ** @see com.sgi.sysadm.ui.HostContext#abort
     **/
    public static void fatal(String msg) {
	System.out.println("Fatal    " + msg);
	Thread.dumpStack();
	com.sgi.sysadm.ui.HostContext.abort();
    }

    /**
     ** Verify some condition is true; if not log a message and terminate
     ** the program.
     **
     ** If the boolean condition is false (and the ASSERT level is enabled)
     ** the message will be logged, a stack dump generated, and then
     ** <code>HostContext.abort</code> will be called.
     **
     ** @param cond a condition which is expected to always be true
     ** @param msg the message to log
     **/
    public static void assert(boolean cond, String msg) {
	// recordLogs(2);	
	if ((_level & ASSERT) != 0) {
	    if (!cond) {
		System.out.println("Assertion failed: " + msg);
		Thread.dumpStack();
		com.sgi.sysadm.ui.HostContext.abort();
	    }
	}
    }

// --------------------------------------------
/*    private static Hashtable counts = new Hashtable();
      private static int counter = 0; */

    /**
     * A debugging method for recording the number of times other methods
     * call the loging methods. 
     *
     * We get a copy of the stack trace, pop the first few off to see
     * who called the Log message, and store that call in an hashtable
     * along with a counter.  Every so often, we dump the contents of
     * the hashtable.
     *
     * Leave this method commented out in checked in code!
     */
/*    private static void recordLogs(int levelsToPop) {
	String trace = SysUtil.stackTraceToString(new Exception());
	if (!trace.startsWith("java.lang.Exception")) {
	    System.out.println("Got bad trace:\n" + trace);
	    return;
	}
	int start = -1;
	// Skip some stack frames to get to where we want to look.
	for (int ii = 0; ii < levelsToPop+1; ii++) {
	    start = trace.indexOf('\n',start+1);
	}
	int stop = trace.indexOf('\n', start+1);

	//  5 characters gets us past the tab, the "at" and the space"
	String call = trace.substring(start+5,stop);
//	System.out.println("\ntrace:\n" + trace);
//	System.out.println("call:\n" +call);
	Integer c = (Integer)counts.get(call);
	if (c == null) {
		c = new Integer(0);
	    }
	    counts.put(call, new Integer((c.intValue())+1));
	    if (++counter % 500 == 0) {
		java.util.Enumeration enum = counts.keys();
		System.out.println("\n---------------------");
		while (enum.hasMoreElements()) {
		    String key = (String)enum.nextElement();
		    System.out.println( counts.get(key) + " :" + key);
		}
	    }
    }
*/

}
