//
// PrivBroker.java
//
//	Interface for checking, adding, and running privileges.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.util;

/**
 * PrivBroker provides an interface that clients use to access
 * privileged commands.  Commands can be checked, added, and run.
 * <p>
 * For more information on privileges, see the following Irix man
 * pages:
 * <ul>
 * <li><i>runpriv</i>(1M)</li>
 * <li><i>checkpriv</i>(1M)</li>
 * <li><i>privilege</i>(4)</li>
 * </ul>
 * <p>
 * PrivBroker instances are normally obtained via HostContext.
 * <p>
 * All of the <tt>runPriv</tt> methods return a
 * <tt>java.lang.Process</tt>.  All further interaction with the
 * privileged command, including error handling, happens with the
 * Process.
 * <p>
 * <tt>checkPrivs</tt> will succeed if the user is privileged or if
 * the correct root password was passed to <tt>setPassword</tt>.
 * Similarly, <tt>runPriv</tt> will actually run the privileged
 * command if the user is privileged or if the correct root password
 * was passed to <tt>setPassword</tt>.  If neither of these conditions
 * are met, the <i>runpriv</i>(1M) command will exit with a non-zero
 * exit status without executing the privileged command.  The caller
 * can detect this situation by monitoring the Process returned by
 * <tt>runPriv</tt>.
 * <p>
 * <tt>addPrivs</tt> will only succeed if the correct root password
 * was passed to <tt>setPassword</tt>.
 * 
 * @see com.sgi.sysadm.ui.HostContext
 * @see java.lang.Process
 */
public interface PrivBroker {

    /**
     * When the user is not privileged to perform a requested
     * operation, <b>REASON_NOT_PRIVILEGED</b> is returned by
     * <tt>ResultEvent.getResult</tt> in <tt>failed</tt> callbacks to
     * the ResultListener passed to <tt>checkPrivs</tt> or
     * <tt>addPrivs</tt>.
     */
    static public final Object REASON_NOT_PRIVILEGED = new Object();

    /**
     * When the user has referenced a privilege that does not exist,
     * <b>REASON_NO_SUCH_PRIV</b> is returned by 
     * <tt>ResultEvent.getResult</tt> in <tt>failed</tt> callbacks to
     * the ResultListener passed to <tt>checkPrivs</tt> or
     * <tt>addPrivs</tt>.
     */
    static public final Object REASON_NO_SUCH_PRIV = new Object();

    /**
     * Set the password to be used in subsequent <tt>runPriv</tt>,
     * <tt>checkPrivs</tt>, or <tt>addPrivs</tt> calls.
     * <p>
     * <tt>password</tt> is the plain-text root password.  It is used
     * in conjunction with the <tt>-auth</tt> argument to
     * <tt>runpriv</tt> (see the <i>runpriv</i>(1M) Irix man page).
     * <p>
     * <tt>setPassword</tt> does not check the validity of
     * <tt>password</tt>.  Use <tt>checkPassword</tt> to determine
     * whether <tt>password</tt> is valid.
     * 
     * @param password Password to be used in subsequent calls.
     */
    public void setPassword(String password);

    /**
     * Check that the current password (set via <tt>setPassword</tt>) is
     * valid.
     *
     * @param listener Gets notified with the result of the check.
     */
    public void checkPassword(ResultListener listener);

    /**
     * Run a privileged command with arguments parsed from a String.
     * 
     * @param privilege Privilege to run.
     * @param args Arguments to the privileged command.
     * 
     * @return Process representing the privileged command.
     */
    public Process runPriv(String privilege, String args);

    /**
     * Run a privileged command with an array of String arguments.
     * 
     * @param privilege Privilege to run.
     * @param args Arguments to the privileged command.
     * 
     * @return Process representing the privileged command.
     */
    public Process runPriv(String privilege, String args[]);

    /**
     * Run a privileged command with arguments derived from an
     * AttrBundle.  The command line will look like this:
     * <p>
     * <tt>runpriv <i>privilege</i> <i>key0</i>=<i>value0</i> <i>key1</i>=<i>value1</i> ...</tt>
     * <p>
     * where the <i>key</i> and <i>value</i> arguments are the keys
     * and values of the Attributes in the AttrBundle <tt>args</tt>.
     * <p>
     * For more detail on how command line arguments are passed, see
     * the C header file:
     * <br>
     * <tt>/usr/include/sysadm/SaParam.h</tt>
     * 
     * @param privilege Privilege to run.
     * @param args Arguments to the privileged command.
     * 
     * @return Process representing the privileged command.
     */
    public Process runPriv(String privilege, AttrBundle args);
    
    /**
     * Check to see if we have a set of privileges.
     * <tt>checkPrivs</tt> will fail if any of the <tt>privs</tt> is
     * missing from the user's set of privileges.
     * 
     * @param privs Set of privileges to check.
     * @param listener Gets notified of the result of the check.
     */
    public void checkPrivs(String privs[], ResultListener listener);
    
    /**
     * Add a set of privileges for the <tt>user</tt>.  After privileges
     * have been added, the <tt>user</tt> will no longer need to
     * provide the root password to execute the privileged commands
     * corresponding to the privileges which were added.
     * 
     * @param privs Set of privileges to add.
     * @param user user name to add privileges for.
     * @param listener Gets notified of the result of the add.
     */
    public void addPrivs(String privs[], String user,
			 ResultListener listener);
}
