//
// ProcessEvent.java
//
//	Event generated by a ProcessWatcher.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//
package com.sgi.sysadm.util;

import com.sgi.sysadm.util.*;
import java.util.EventObject;

/**
 * ProcessEvent is an event generated by a ProcessWatcher in response to
 * the activities of a Process.
 *
 * @see java.lang.Process
 * @see com.sgi.sysadm.util.ProcessListener
 * @see com.sgi.sysadm.util.ProcessWatcher
 */
public class ProcessEvent extends EventObject {
    private int _type;
    private byte[] _bytes;
    private int _exitCode;
    
    /**
     * Events of type ProcessEvent.OUTPUT_DATA are generated when the Process
     * writes data to its standard output stream.  Use ProcessEvent.getBytes()
     * to get the output data.
     */
    public static final int OUTPUT_DATA = 1;
    
    /**
     * Events of type ProcessEvent.ERROR_DATA are generated when the Process
     * writes data to its standard error stream.  Use ProcessEvent.getBytes()
     * to get the error data.
     */
    public static final int ERROR_DATA = 2;
    
    /**
     * Events of type ProcessEvent.EXITED are generated when the Process
     * terminates.  Use ProcessEvent.getExitCode() to get the exit code.
     */
    public static final int EXITED = 3;
    
    /**
     * Construct a ProcessEvent.
     * 
     * @param source Process associated with this event.
     * @param type Type of event.  One of ProcessEvent.OUTPUT_DATA,
     *             ProcessEvent.ERROR_DATA, or ProcessEvent.EXITED.
     */
    public ProcessEvent(Process source, int type) {
	super(source);
	
	Log.assert(type == OUTPUT_DATA || type == ERROR_DATA || type == EXITED,
		   "Illegal ProcessEvent type.  It should be one of: " +
		   "ProcessEvent.OUTPUT_DATA, ProcessEvent.ERROR_DATA, or " +
		   "ProcessEvent.EXITED");
	_type = type;
    }
    
    /**
     * Get the Process associated with this event.
     *
     * @return The Process associated with this event.
     */
    public Process getProcess() {
	return (Process)getSource();
    }
    
    /**
     * Set the byte array associated with an OUTPUT_DATA or ERROR_DATA event.
     *
     * @param bytes The array of bytes generated by the Process.
     */
    public void setBytes(byte[] bytes) {
	Log.assert(_type == OUTPUT_DATA ||
		   _type == ERROR_DATA, "ProcessEvent type is " +
	           "not ProcessEvent.OUTPUT_DATA or ProcessEvent.ERROR_DATA");
	_bytes = bytes;
    }
    
    /**
     * Get the byte array associated with an OUTPUT_DATA or ERROR_DATA
     * event.  Will call Log.fail() if the event is of the wrong type.
     *
     * @return The array of bytes generated by the Process.
     */
    public byte[] getBytes() {
	Log.assert(_type == OUTPUT_DATA ||
		   _type == ERROR_DATA, "ProcessEvent type is " +
	           "not ProcessEvent.OUTPUT_DATA or ProcessEvent.ERROR_DATA");
	return _bytes;
    }
    
    /**
     * Set the exit code associated with an EXITED event.
     *
     * @param exitCode The Process exit code.
     */
    public void setExitCode(int exitCode) {
	Log.assert(_type == EXITED, 
		   "ProcessEvent type is not ProcessEvent.EXITED.");
	_exitCode = exitCode;
    }
    
    /**
     * Get the exit code associated with an EXITED event.  Will call
     * Log.fail() if the event is of the wrong type.
     *
     * @return The Process exit code.
     */
    public int getExitCode() {
	Log.assert(_type == EXITED,
		   "ProcessEvent type is not ProcessEvent.EXITED.");
	return _exitCode;
    }
}
