//
// ProcessWatcher.java
//
//	Spawns threads to deal with a Process.  Notifies ProcessListeners
//	when input arrives or the Process exits.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//
package com.sgi.sysadm.util;

import com.sgi.sysadm.util.*;
import javax.swing.*;
import java.io.*;
import java.util.*;

/**
 * ProcessWatcher spawns threads to deal with a running Process.  Notifies
 * ProcessListeners when input arrives or the Process exits.
 */
public class ProcessWatcher {
    private Process _proc;
    private Vector _listeners = new Vector();
    private boolean _watching;
    
    // Class used to notify a ProcessListener of an event.
    private interface ProcessNotifier {
	public void notify(ProcessListener listener);
    }
    
    /**
     * Constructor.
     *
     * @param proc Process to watch.
     */
    public ProcessWatcher(Process proc) {
	_proc = proc;
    }
	
    /**
     * Add a listener interested in ProcessEvents.  This method must be
     * called before startWatching() is called.
     *
     * @param listener A listener interested in ProcessEvents.
     */
    public void addProcessListener(ProcessListener listener) {
	Log.assert(!_watching, "Attempt to add ProcessListener after call " +
		   "to startWatching()");
	_listeners.addElement(listener);
    }
    
    /**
     * Remove a listener no longer interested in ProcessEvents.
     *
     * @param listener A listener previously interested in ProcessEvents.
     */
    public synchronized void removeProcessListener(ProcessListener listener) {
	_listeners.removeElement(listener);
    }
    
    /**
     * Initiate monitoring of the process.  A call to startWatching()
     * indicates that all ProcessListeners have been added and they are
     * ready to receive events.
     */
    public void startWatching() {
	Log.assert(!_watching, "Illegal duplicate call to startWatching().");
	_watching = true;
	
	// Watch the standard output stream of the process.
	final Thread inputThread = new Thread("ProcessWatcherInput") {
	    public void run() {
		readAndNotify(_proc.getInputStream(),
			      ProcessEvent.OUTPUT_DATA);
	    }
	};
	inputThread.start();

	// Watch the standard error stream of the process.
	final Thread errorThread = new Thread("ProcessWatcherError") {
	    public void run() {
		readAndNotify(_proc.getErrorStream(),
			      ProcessEvent.ERROR_DATA);
	    }
	};
	errorThread.start();

	// Wait for the process to exit in a separate thread so this
	// method does not block.
	final Thread waitForThread = new Thread("ProcessWatcherExit") {
	    public void run() {
		final ProcessEvent event =
		    new ProcessEvent(_proc, ProcessEvent.EXITED);
		    
		try {
		    _proc.waitFor();
		    inputThread.join();
		    errorThread.join();
		    event.setExitCode(_proc.exitValue());
		} catch (InterruptedException ex) {
		    ex.printStackTrace();
		    event.setExitCode(-1);
		}
		
		// Notify listeners that the process has exited.
		notifyProcessListeners(new ProcessNotifier() {
		    public void notify(ProcessListener listener) {
			listener.processExited(event);
		    }
		});
	    }
	};
	waitForThread.start();
    }
    
    /**
     * Attempt to read a Process InputStream.  When data becomes available
     * on this stream, notify ProcessListeners.
     *
     * @param stream The Process InputStream to read.
     * @param eventType The type of ProcessEvent to generate when data becomes
     *                  available on this stream.
     */
    private void readAndNotify(InputStream stream, int eventType) {
	final int type = eventType;
	try {
	    while (true) {
		// First attempt to read a single byte, which will block until
		// data is available on the input stream or the stream is
		// closed.  Next read the rest of the bytes available without
		// blocking.  This strategy enables us to allocate a byte
		// array of the exact length we need, which simplifies
		// listener input handling code.
		
		// Block until at least one byte is available or EOF.
		byte[] firstByte = new byte[1];
		int len = stream.read(firstByte);
		if (len == -1) {
		    break;
		}
		
		// Read the rest of the bytes available without blocking.
		int byteCount = stream.available() + 1;
		final byte[] allBytes = new byte[byteCount];
		allBytes[0] = firstByte[0];
		stream.read(allBytes, 1, byteCount-1);
		
		// Notify ProcessListeners that data has arrived.
		notifyProcessListeners(new ProcessNotifier() {
		    public void notify(ProcessListener listener) {
			ProcessEvent event =
			    new ProcessEvent(_proc, type);
			event.setBytes(allBytes);
			if (type == ProcessEvent.OUTPUT_DATA) {
			    listener.processOutputData(event);
			} else {
			    listener.processErrorData(event);
			}
		    }
		});
	    }
	    stream.close();
	} catch (IOException ex) {
	    Log.error("ProcessWatcher", "IOEception in errorThread.");
	    ex.printStackTrace();
	}
    }
    
    /**
     * Notify all of the ProcessListeners of a particular event.  Notification
     * will always take place in the event dispatch thread.
     *
     * @param notifier Notifier that knows what event has occurred.
     */
    private synchronized void notifyProcessListeners(ProcessNotifier notifier) {
	final ProcessNotifier ntf = notifier;
	Enumeration enum = _listeners.elements();
	while (enum.hasMoreElements()) {
	    final ProcessListener listener =
		(ProcessListener)enum.nextElement();
	    SwingUtilities.invokeLater(new Runnable () {
		public void run() {
		    ntf.notify(listener);
		}
	    });
	}
    }
}
