//
// ResourceStack.java
//
//	Maintains a stack of ResourceBundles.
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//
package com.sgi.sysadm.util;

import com.sgi.sysadm.ui.*;
import javax.swing.*;
import java.awt.*;
import java.net.URL;
import java.text.*;
import java.util.*;
import java.io.*;

/**
 * ResourceStack maintains a stack of ResourceBundles.  Clients can
 * easily search for a property in the whole stack. 
 * A new ResourceBundle may be pushed onto the top of
 * the stack using ResourceStack.pushBundle().
 * <p>
 * Many of the get* methods in this class have four versions.  For
 * example purposes, consider getString.
 * <dl>
 * <dt>getString(String key)
 * <dd>Looks up the specified key in the ResourceStack and return the
 * associated value.  If the key is missing, a
 * MissingResourceException will be thrown.
 * <dt>getString(String[] keys)
 * <dd>Looks up the specifed keys in the ResourceStack, one after
 * another, until a key is found in the ResourceStack.  The value
 * associated with that key is then returned.  If none of the keys are
 * found, a MissingResourceException is thrown.  This version is
 * useful if you want to lookup a particular resource, but then fall
 * back to another resource if the first resource is not found.
 * <dt>getString(String key, String defaultVal)
 * <dd>The same as the getString(String key) case, but instead of
 * returning a MissingResourceException if the key is not found, the
 * defaultVal will be returned.  This is useful in cases where the
 * calling method wants to lookup a resource, but has a default value
 * that it can use if the lookup fails.  This reduces the overhead of
 * having to catch the MissingResourceException.
 * <dt>getString(String[] keys, String defaultVal)
 * <dd>The same as the getString(String[] keys) case, but it takes a
 * default value.  See the case above for more info about the
 * defaultValue.
 * </dl>
 *<p>
 * ResourceStack supports macros in Properties files as
 * follows:  Any key or value that is loaded into a ResourceStack
 * via the pushBundle method will be parsed for ${<i>MACRO</i>}.  The
 * resource <i>MACRO</i> will be looked up and substituted for
 * ${<i>MACRO</i>}.  
 * Macros in resource keys are resolved when the resource file is
 * loaded - for that reason macros in keys must be defined in the same
 * property file.  Macros in resource values are resolved when the
 * lookup (with getString(), etc) is made.  For this reason, the
 * definition of a macro in a value can be in the same propery file or
 * in any resource bundle on the stack when the lookup is made.
 *<p> 
 * 
 */
public class ResourceStack implements Cloneable {

    /**
     * The Resource <i>&lt;Icon key&gt;.iconWidth</i> is an integer
     * that specifies the width, in points, of the FtrIcon identified
     * by &lt;Icon key&gt;.  For example, to specify the width and
     * height of an FtrIcon, put the following in the ResourceFile:
     * <pre>
     * Icon = com.sgi.sysadm.MyIcon
     * Icon.iconWidth = 5
     * Icon.iconHeight = 5
     * </pre>
     * If this resource is missing, the default width for an FtrIcon
     * will be used.
     */
    public static final String ICON_WIDTH = ".iconWidth";

    /**
     * The Resource <i>&lt;Icon key&gt;.iconHeight</i> is an integer
     * that specifies the height, in points, of the FtrIcon identified
     * by &lt;Icon key&gt;.  
     * See <tt>ICON_WIDTH</tt> for an example.
     * If this resource is missing, the default height for an FtrIcon
     * will be used.
     * @see #ICON_WIDTH
     */
    public static final String ICON_HEIGHT = ".iconHeight";

    /**
     * The suffix to append to all property files
     * For example, myProperties&lt;BUNDLE_SUFFIX&gt;.properties.
     * The value is "P".
     */
    public static final String BUNDLE_SUFFIX = "P";

    private static Object[] _rbAndArray;
    private final static String _class = "ResourceStack";
    private final static String _noPropertyLookup = 
        _class + ".Error.noSuchProperty";
    
    private static Hashtable _missingBundles = new Hashtable();
    private static Hashtable _bundles = new Hashtable();
    private static Hashtable _bundleArrays = new Hashtable();
    
    private static final String CLASS_NAME = "ResourceStack";

    // Holds the names of the bundles on the stack.
    private Vector _stack = new Vector();

    // Holds the resources
    private Hashtable _properties = new Hashtable(1000);
    private Hashtable _arrays = new Hashtable();
    
// XXX DEBUG    private static int count = 1;
    
    /**
     * Create a ResourceStack with only
     * com.sgi.sysadm.util.SysadmUtilP on it
     */
    public ResourceStack() {
// XXX DEBUG System.out.println(count++);
	if (_rbAndArray == null) {
	    _rbAndArray = getBundle(
		ResourceStack.getPackageName(ResourceStack.class.getName()) +
		"SysadmUtil" + BUNDLE_SUFFIX);
	}
	pushProperties(_rbAndArray);
    }
    
    /**
     * Makes an exact copy of the ResourceStack.
     *
     * @return The copy.
     * @see java.lang.Object#clone()
     */
    public Object clone() {
	ResourceStack o = null; 
	try {
	    o = (ResourceStack)super.clone();
	    o._properties = (Hashtable)_properties.clone();
	    o._arrays = (Hashtable)_arrays.clone();
	    o._stack = (Vector)_stack.clone();
	    return o;
	} catch (CloneNotSupportedException e) {
	    throw new InternalError();
	}
    }

    private MissingResourceException createMissingResourceException(
	String[]keys) {
	return new MissingResourceException( 
	    MessageFormat.format(
		(String)((Hashtable)_rbAndArray[0]).get(_noPropertyLookup),
		new Object[] { keys[keys.length - 1] , toString()}),
	    _class,
	    keys[keys.length - 1]);
    }

    /**
     * getBundle() loads a properties file.  Since we localize on a
     * package by package basis and set CLASSPATH conditionally based
     * on locale, and since all of our resource bundles are in
     * properties files, we don't need all the complexity that
     * ResourceBundle.getBundle() has.
     *
     * @param bundleName Name of the properties file to get.
     * 
     * @return an Object[]. The element at 0 will be a Hashtable of the
     * properties, and the element at 1 will be a Hashtable of arrays
     * that were found in the bundle, with their lenghts as the values.
     */
    private Object[] getBundle(String bundleName) {
	StringBuffer fileNameBuf = new StringBuffer(
	    bundleName.replace('.', '/')).append(".properties");
	if (fileNameBuf.charAt(0) != '/') {
	    fileNameBuf.insert(0, '/');
	}
	String fileName = fileNameBuf.toString();
	Object[] bundleAndArray
	    = (Object[])_bundles.get(fileName);
	if (bundleAndArray != null) {
	    return bundleAndArray;
	}
	if (_missingBundles.containsKey(fileName)) {
	    throw new MissingResourceException(
		"Can't find resource bundle " + bundleName, bundleName, "");
	}
	InputStream input = ResourceStack.class.getResourceAsStream(fileName);
	if (input != null) {
	    input = new BufferedInputStream(input);
	    try {
		bundleAndArray =
		    expandBundle(new PropertyResourceBundle(input));
		_bundles.put(fileName, bundleAndArray);
		return bundleAndArray;
	    } catch (IOException ex) {
	    }
	}
	_missingBundles.put(fileName, fileName);
	throw new MissingResourceException(
	    "Can't find resource bundle " + bundleName, bundleName, "");
    }  
    
    private Object[] expandBundle(ResourceBundle rb) {
	Vector arrayBases = new Vector();
	Enumeration keys = rb.getKeys();
	Hashtable tempHash = new Hashtable();
	Hashtable arrays = new Hashtable();
	while (keys.hasMoreElements()) {
	    String key = (String)keys.nextElement();
	    String resolvedKey = expandString(key, rb);
	    int keyLength = resolvedKey.length();
	    if (keyLength >= 2 && 
		resolvedKey.charAt(keyLength - 1) == '0' &&
		!Character.isDigit(resolvedKey.charAt(keyLength-2))) {
		// If the last character is a '0' and second to last
		// character isn't a digit, then we know that this
		// could be the first element of a String array.
		arrayBases.addElement(
		    resolvedKey.substring(0, keyLength - 1));
	    }
	    tempHash.put(resolvedKey, rb.getObject(key));
	}
	// Now take care of any possible string arrays we identified.
	int counter;
	String element;
	StringBuffer buf = new StringBuffer();
	int len;
	for (int ii = 0; ii < arrayBases.size(); ii++) {
	    counter = 0;
	    buf.setLength(0);
	    buf.append((String)arrayBases.elementAt(ii));
	    len = buf.length();
	    while (true) {
		buf.setLength(len);
		buf.append(++counter);
		if (tempHash.get(buf.toString()) == null) {
		    arrays.put((String)arrayBases.elementAt(ii), 
				new Integer(counter));
		    break;
		}
	    }
	}
	return new Object[] {tempHash, arrays };
    }
    
    /**
     * Convenience method for getting the name of a class without the
     * package prefix.  Classes that wish to prefix all of their property
     * names with the class name will find this method useful.
     *
     * @param classObj Class object from which to get the class name.
     * @return class name without package portion of name.
     */
    public static String getClassName(Class classObj) {
	String fullName = classObj.getName();
	int lastDot = fullName.lastIndexOf('.');
	if (lastDot == -1){  // If there is no package part
	    return fullName;
	} else {  // If there is package part of the name
	    return fullName.substring(lastDot+1);
	}
    }

    /**
     * Convenience method for getting the name of a class without the
     * package prefix.  Classes that wish to prefix all of their property
     * names with the class name will find this method useful.
     *
     * @param className String containing a fully qualified class name.
     * @return class name without package portion of name.
     */
    public static String getClassName(String className) {
	int lastDot = className.lastIndexOf('.');
	if (lastDot == -1){  // If there is no package part
	    return className;
	} else {  // If there is package part of the name
	    return className.substring(lastDot+1);
	}
    }

    /**
     * Convenience method for getting the package name of a class.
     *
     * @param class Class object from which to get the package name.
     * @return package name of <TT>class</TT>, including trailing '.'
     */
    public static String getPackageName(Class classObj) {
	String fullName = classObj.getName();
	int lastDot = fullName.lastIndexOf('.');
	if (lastDot == -1) {
	    return "";
	} else {
	    return fullName.substring(0, lastDot+1);
	}
    }

    /**
     * Convenience method for getting the package name of a class.
     *
     * @param fullClassName Fully qualified class name from which to
     *                      get the package name.
     * @return package name of class <TT>fullClassName</TT>,
     *         including trailing '.'
     */
    public static String getPackageName(String fullClassName) {
	int lastDot = fullClassName.lastIndexOf('.');
	if (lastDot == -1) {
	    return "";
	} else {
	    return fullClassName.substring(0, lastDot+1);
	}
    }

    /**
     * Push a new ResourceBundle onto the bundle stack.  This bundle will
     * be searched before any bundles already on the bundle stack.
     *
     * @param bundleName Name of ResourceBundle to push onto the stack.
     *
     * @exception java.util.MissingResourceException Thrown if bundleName
     *            is not found.
     */
    public void pushBundle(String bundleName) throws MissingResourceException {
	_stack.removeElement(bundleName);
	pushProperties(getBundle(bundleName));
	_stack.addElement(bundleName);
    }

    /**
     * Push the ResourceBundles from another ResourceStack onto the
     * bundle stack.  Bundles from <tt>other</tt> will be searched
     * before any bundles already on the bundle stack.  Bundles from
     * <tt>other</tt> will not be pushed if they are already on the
     * stack, and those Bundles will remain in their old location on
     * the stack.
     * <p>
     * If <tt>other</tt> is null, pushStack does nothing.
     *
     * @param other ResourceStack providing ResourceBundles to be
     *              pushed onto the bundle stack.
     */
    public void pushStack(ResourceStack other) {
	if (other == null) {
	    return;
	}
	String bundle;
	boolean alreadyOnStack;
	for (int ii = 0; ii < other._stack.size(); ii++) {
	    alreadyOnStack = false;
	    bundle = (String)other._stack.elementAt(ii);
	    for (int jj = 0; jj < _stack.size(); jj++) {
		if (bundle.equals(_stack.elementAt(jj))) {
		    alreadyOnStack = true;
		    break;
		}
	    }
	    if (!alreadyOnStack) {
		pushBundle(bundle);
	    }
	}
    }

    /**
     * Search the package path for package ResourceBundles.  The
     * search starts from the root of the path and works
     * downwards.  For example, if the class name is
     * "com.sgi.product.doSomethingTask", the search will first
     * look for com.PackageP, then com.sgi.PackageP, etc. up through
     * com.sgi.product.doSomethingTask.PackageP. There
     * can be zero or more such package properties files, each
     * being pushed onto the stack as it is found.  The result is
     * that package properties files closer to the package root
     * will be overridden by package properties files farther from
     * the package root.
     *
     * @param objectName The package-qualified name of the object for
     *                   which package bundles should be loaded.
     */
    public void pushPackageBundles(String objectName) {
	if (!objectName.endsWith(".")) {
	    objectName += ".";
	}
	int dotIndex = objectName.indexOf('.');
	while (dotIndex > 0) {
	    // Attempt to load the package properties from this level
	    // of the package path
	    try {
		pushBundle(objectName.substring(0, dotIndex+1) +
			   "Package" + BUNDLE_SUFFIX);
	    } catch (MissingResourceException exception) {
		// The package properties files are optional.
	    }

	    dotIndex = objectName.indexOf('.', dotIndex+1);
	}
    }

    /**
     * Return the names of the ResourceBundles in this stack in the
     * order in which they will be searched for resources.
     */
    public String toString() {
	StringBuffer result = new StringBuffer();
	for (int ii = _stack.size() - 1; ii >= 0; ii--) {
	    result.append((String)_stack.elementAt(ii)).append("\n");
	}
	return result.toString();
    }

    private void pushProperties(Object[] rbAndArray) {
	Hashtable bundle = (Hashtable)rbAndArray[0];
	Hashtable arrays = (Hashtable)rbAndArray[1];
//	_properties.put(bundle);
//	_arrays.put(arrays);
        Enumeration keys = bundle.keys();
        String key;
        while (keys.hasMoreElements()) {
            key = (String)keys.nextElement();
            _properties.put(key, bundle.get(key));
        }
        keys = arrays.keys();
        while (keys.hasMoreElements()) {
            key = (String)keys.nextElement();
            _arrays.put(key, arrays.get(key));
        }
    }
    
    
    private String expandString(String input, ResourceBundle bundle) {
	if (input == null) {
	    return null;
	}
	StringBuffer out;
	int close;
	int open =  input.indexOf("${", 0);
	int n;
	int i;
	if (open == -1) {
	    return input;
	} else {
	    out = new StringBuffer(input.length()*2);
	    for (i = 0; i < open; i++) {
		out.append(input.charAt(i));
	    }
	}
	while (open < input.length() && open >= 0) {
	    open += 2;
	    close = input.indexOf('}', open);
	    if (close < 0) {
		// A ${ sequence was found, but with no closing }.  We
		// should just treat the ${ as if it were normal.
		out.append(input.substring(open-2));
		break;
	    }
	    String lookup = input.substring(open, close);
	    if (bundle == null) {
		out.append(getString(lookup, ""));
	    }
	    else {
		try {
		    out.append(expandString(bundle.getString(lookup),
					    bundle));
		    }
		catch (MissingResourceException mre) {
		    Log.fatal("While trying to resolve the string: \""
			     + input + "\" ResourceStack failed to find\n" +
			     "a resource: \"" + lookup + "\".  Definitions " +
			     "of Macros that are used in resource keys\n" +
			     "must be in the same file as the key that " +
			     "references them.");

		}
		
	    }
	    open = input.indexOf("${", close + 1);
	    if (open == -1) {
		// no more found
		n = input.length();
		for (i = close+1; i < n; i++) {
		    out.append(input.charAt(i));
		}
	    } else {
		for (i = close+1; i < open; i++) {
		    out.append(input.charAt(i));
		}
	    }
	}
	String result = out.toString();
//	    System.out.println("In: " + input + " Out: " + result);
	return result;
    }
    

    //------------------------ getObject -----------------------

    /**
     * Search for a resource in the resource stack
     *
     * @param key Name of the resource to retrieve.
     *
     * @exception java.util.MissingResourceException if the resource
     *            is not found.
     *
     * @return The resource described by <TT>key</TT>.
     */
    public Object getObject(String key) throws MissingResourceException {
	return getObject(new String[] { key });
    }

    /**
     * Search for a resource in the resource stack.
     *
     * @param keys An array of Strings containing the resource names to
     *             look up.  The names will be looked up, in order, and
     *             the first resource found will be returned.
     *
     * @exception java.util.MissingResourceException if the property
     *            is not found.
     *
     * @return The first resource that is found matching a key in
     *         <TT>keys</TT>
     */
    public Object getObject(String[] keys) throws MissingResourceException {
	Object object = getObject(keys, null);
	if (object != null) {
	    return object;
	}
	throw createMissingResourceException(keys);
    }

    /**
     * Search for a resource in the resource stack.
     *
     * @param key Name of the resource to retrieve.
     *
     * @param defaultVal The resource to return if no resource
     *                   matching <tt>key</tt> is found.
     *
     * @return The resource described by <TT>key</TT>, or
     *         <tt>defaultVal</tt> if no value is found.
     */
    public Object getObject(String key, Object defaultVal) {
	return getObject(new String[] { key }, defaultVal);
    }

    /**
     *  Search for a resource in the resource stack.
     *
     * @param keys An array of Strings containing the resource names to
     *             look up.  The names will be looked up, in order, and
     *             the first resource found will be returned.
     *
     * @param defaultVal The object to return if no matching properties
     *                   are found.
     *
     * @return The first resource that is found matching a key in
     *         <TT>keys</TT>, or <tt>defaultVal</tt> if no value is found.
     */
    public Object getObject(String[] keys, Object defaultVal) {
	Log.assert(_stack.size() > 0,
		   "Nothing has been added to the ResourceStack");
	for (int ii = 0; ii < keys.length; ii++) {
	    Object object = _properties.get(keys[ii]);
	    if (object != null) {
		return object;
	    }
	}
	return defaultVal;
    }

    // ------------------------- getIcon --------------------------

    /**
     * Get an icon resource.
     *
     * @param key Name of icon.  This can either be the class name of
     *            an FtrIcon subclass, or the full path name of an
     *            image.  If <tt>key</tt> is the name of an FtrIcon,
     *            the resources <i>&lt;key&gt;.iconWidth</i> and
     *            <i>&lt;key&gt;.iconHeight</i> may also be present.
     *            Default values will be used if these resources are missing.
     *            These two resources specify the width and height in
     *            points of the FtrIcon.
     *
     * @exception MissingResourceException if resource is missing.
     * @see #ICON_WIDTH
     * @see #ICON_HEIGHT
     * @return Icon resource
     */
    public Icon getIcon(String[] keys) throws MissingResourceException {
	FtrIcon icon = null;
	String iconName = null;
	String key = null;
	for (int ii = 0 ; ii < keys.length; ii++) {
	    iconName = getString(keys[ii], null);
	    if (iconName != null) {
		key = keys[ii];
		break;
	    }
	}
	if (iconName == null) {
	    throw createMissingResourceException(keys);
	}
	try {
	    icon = FtrIcon.createIcon(iconName);
	    icon.setSize(getPixels(key + ICON_WIDTH),
			 getPixels(key + ICON_HEIGHT));
	    return icon;
	} catch (IllegalArgumentException ex) {
	} catch (MissingResourceException mre) {
	    //ignore this error.  We'll just get default icon size.
	    return icon;
	}
	
	try {
	    InputStream in = ResourceStack.class.getResourceAsStream(iconName);
	    if (in == null) {
		throw new IOException();
	    }
	    byte[] buf = new byte[4096];
	    ByteArrayOutputStream out = new ByteArrayOutputStream();
	    int n;
	    while ((n = in.read(buf)) != -1) {
		    out.write(buf, 0, n);
	    }
	    return new ImageIcon(out.toByteArray());
	} catch (IOException io) {
	    throw new MissingResourceException(
		MessageFormat.format(
		    (String)((Hashtable)_rbAndArray[0]).get(
			_class + ".Error.resourceNotFound"),
		    new Object[] { iconName, keys[keys.length-1] }),
		"java.util.MissingResourceException",
		keys[keys.length-1]);
	}
    }

    /**
     * Get an icon resource.
     *
     * @param keys An array of possible names of the icon.  These can
     *             either be the class names of FtrIcon subclasses, or
     *             the full path names of images.  If a <tt>key</tt>
     *             is the name of an FtrIcon, the resources
     *             <i>&lt;key&gt;.iconWidth</i> and
     *             <i>&lt;key&gt;.iconHeight</i> may also be
     *             present. These two resources specify the width and
     *             height in points of the FtrIcon.  Default values
     *             will be used if these resources are missing. The
     *             array will be searched, in order, until a Icon
     *             resource is found.
     *
     * @exception MissingResourceException if resource is missing.
     * @see #ICON_WIDTH
     * @see #ICON_HEIGHT
     *
     * @return Icon resource
     */
    public Icon getIcon(String key) throws MissingResourceException {
	return getIcon(new String[] { key });
    }

    //---------------------------- getString ----------------------

    /**
     * Search for a String resource in the resource stack
     *
     * @param key Name of the resource to retrieve.
     *
     * @exception java.util.MissingResourceException if the resource
     *            is not found.
     *
     * @return The String described by <TT>key</TT>.
     */
    public String getString(String key) throws MissingResourceException {
	return expandString((String)getObject(key), null);
    }

    /**
     * Search for a String resource in the resource stack.
     *
     * @param keys An array of Strings containing the resource names to
     *             look up.  The names will be looked up, in order, and
     *             the first resource found will be returned.
     *
     * @exception java.util.MissingResourceException if the property
     *            is not found.
     *
     * @return The first String resource that is found matching a key in
     *         <TT>keys</TT>
     */
    public String getString(String[] keys) throws MissingResourceException {
	return expandString((String)getObject(keys), null);
    }

    /**
     * Search for a String resource in the resource stack.
     *
     * @param key Name of the String resource to retrieve.
     *
     * @param defaultVal The resource to return if no resource
     *                   matching <tt>key</tt> is found.
     *
     * @return The String resource described by <TT>key</TT> or
     *         <tt>defaultVal</tt> if no value is found.
     */
    public String getString(String key, String defaultVal) {
	return expandString((String)getObject(key, defaultVal), null);
    }

    /**
     *  Search for a String resource in the resource stack.
     *
     * @param keys An array of Strings containing the resource names to
     *             look up.  The names will be looked up, in order, and
     *             the first resource found will be returned.
     *
     * @param defaultVal The object to return if no matching properties
     *                    are found.
     *
     * @return The first String resource that is found matching a key in
     *         <TT>keys</TT> or <tt>defaultVal</tt> if no value is found.
     */
    public String getString(String[] keys, String defaultVal) {
	return expandString((String)getObject(keys, defaultVal), null);
    }

    // --------------------------- getStringArray -------------------

     /**
     * Search for a set of properties that defines a string array.
     * For example, if the <tt>baseKey</tt> is "myKey", then the
     * strings should be of the form:
     * <pre>
     * myKey0: valueA
     * myKey1: valueB
     * </pre>
     *
     * @param baseKey Base name of array property to retrieve.
     *
     * @exception java.util.MissingResourceException if the set of properties
     *            is not found.
     *
     * @return String[] containing array of strings described
     *         by <TT>baseKey</TT>.
     */
    public String[] getStringArray(String baseKey)
	    throws MissingResourceException {
	return getStringArray(new String[] { baseKey });
    }

    /**
     * Search for a set of properties that defines a string array.
     * For example, if one of the <tt>baseKeys</tt> is "myKey", then the
     * strings should be of the form:
     * <pre>
     * myKey0: valueA
     * myKey1: valueB
     * </pre>
     *
     * @param baseKeys An array of Strings containing the resource names to
     *                 look up.  The names will be looked up, in order, and
     *                 the first resource found will be returned.
     *
     * @exception java.util.MissingResourceException if the set of properties
     *            is not found.
     *
     * @return The first resource that is found matching a key in
     *         <TT>keys</TT>
     */
    public String[] getStringArray(String[] baseKeys) 
	throws MissingResourceException {
	String[] array = getStringArray(baseKeys, null);
	if (array != null) {
	    return array;
	}
	throw createMissingResourceException(baseKeys);
    }

    /**
     * Search for a set of properties that defines a string array.
     * For example, if one of the <tt>baseKeys</tt> is "myKey", then the
     * strings should be of the form:
     * <pre>
     * myKey0: valueA
     * myKey1: valueB
     * </pre>
     *
     * @param baseKeys Base names of array property to search.  The
     *                 names will be looked up, in order, until a
     *                 stringArray resource is found.
     *
     *
     * @return String[] containing array of strings described
     *         by <TT>baseKey</TT> or <tt>defaultVal</tt> if no value is found.
     */
    public String[] getStringArray(String[] baseKeys, String[] defaultVal) {
	for (int jj = 0; jj < baseKeys.length; jj++) {
	    String val = getString(baseKeys[jj] + "0", null);
	    if (val == null) {
		// Didn't find the first element, so can't be that baseKey
		continue;
	    }
	    int numKeys = ((Integer)_arrays.get(baseKeys[jj])).intValue();
	    String[] result = new String[numKeys];
	    result[0] = val;
	    for (int ii = 1; ii < numKeys ; ii++) {
		val = getString(baseKeys[jj] + ii, null);
		if (val == null) {
		    Log.fatal("ran off the end of the array: " +
			      baseKeys[jj]);
		}
		result[ii] = val;
	    }
	    return result;
	}
	return defaultVal;
    }

    /**
     * Search for a set of properties that defines a string array.
     * For example, if the <tt>baseKey</tt> is "myKey", then the
     * strings should be of the form:
     * <pre>
     * myKey0: valueA
     * myKey1: valueB
     * </pre>
     *
     * @param baseKey An array of Strings containing the resource names to
     *                look up.  The names will be looked up, in order, and
     *                the first resource found will be returned.
     *
     * @param defaultVal The object to return if no matching properties
     *                   are found.
     *
     * @return The first resource that is found matching a key in
     *         <TT>keys</TT> or <tt>defaultVal</tt> if no value is found.
     */
    public String[] getStringArray(String baseKey, String[] defaultVal) {
	return getStringArray(new String[] { baseKey }, defaultVal);
    }

    //-------------------- getInt ------------------------

    /**
     * Search for an int resource in the resource stack
     *
     * @param key Name of the resource to retrieve.
     *
     * @exception java.util.MissingResourceException if the resource
     *            is not found.
     *
     * @return The int resource described by <TT>key</TT>.
     */
    public int getInt(String key) throws MissingResourceException {
	return Integer.parseInt(getString(key));
    }

    /**
     * Search for an int resource in the resource stack.
     *
     * @param keys An array of Strings containing the resource names to
     *             look up.  The names will be looked up, in order, and
     *             the first resource found will be returned.
     *
     * @exception java.util.MissingResourceException if the property
     *            is not found.
     *
     * @return The first int resource that is found matching a key in
     *         <TT>keys</TT>
     */
    public int getInt(String[] keys) throws MissingResourceException {
	return Integer.parseInt(getString(keys));
    }

    /**
     * Search for an int resource in the resource stack.
     *
     * @param key Name of the resource to retrieve.
     *
     * @param defaultVal The resource to return if no resource
     *                   matching <tt>key</tt> is found.
     *
     * @return The int resource described by <TT>key</TT> or
     *         <tt>defaultVal</tt> if no value is found.
     */
    public int getInt(String key, int defaultVal) {
	return getInt(new String[] { key }, defaultVal);
    }

    /**
     *  Search for an int resource in the resource stack.
     *
     * @param keys An array of Strings containing the resource names to
     *             look up.  The names will be looked up, in order, and
     *             the first resource found will be returned.
     *
     * @param defaultVal The object to return if no matching properties
     *                   are found.
     *
     * @return The first int resource that is found matching a key in
     *          <TT>keys</TT> or <tt>defaultVal</tt> if no value is found.
     */
    public int getInt(String[] keys, int defaultVal) {
	String string = getString(keys, null);
	return string != null ? Integer.parseInt(string) : defaultVal;
    }

    // ------------------ getPixels ---------------------------

    /**
     * Search for a pixel resource in the resource stack.  The
     * resource will be interpreted as an integer representing a
     * distance in points (1 point = 1/72 of an inch), and the points
     * will be converted to pixels.
     *
     * @param key Name of the resource to retrieve.
     *
     * @exception java.util.MissingResourceException if the resource
     *            is not found.
     *
     * @return The pixel resource described by <TT>key</TT>.
     */
    public int getPixels(String key) throws MissingResourceException {
	return SysUtil.pixelsFromPoints(getInt(key));
    }

     /**
     * Search for a pixel resource in the resource stack.  The
     * resource will be interpreted as an integer representing a
     * distance in points (1 point = 1/72 of an inch), and the points
     * will be converted to pixels
     *
     * @param keys An array of Strings containing the resource names to
     *             look up.  The names will be looked up, in order, and
     *             the first resource found will be returned.
     *
     * @exception java.util.MissingResourceException if the property
     *            is not found.
     *
     * @return The first pixel resource that is found matching a key in
     *         <TT>keys</TT>
     */
    public int getPixels(String[] keys) throws MissingResourceException {
	return SysUtil.pixelsFromPoints(getInt(keys));
    }

    /**
     * Search for a pixel resource in the resource stack.  The
     * resource will be interpreted as an integer representing a
     * distance in points (1 point = 1/72 of an inch), and the points
     * will be converted to pixels
     *
     * @param key Name of the resource to retrieve.
     *
     * @param defaultVal The resource to return if no resource
     *                   matching <tt>key</tt> is found. The int will
     *                   be converted from points to pixels
     *
     * @return The pixel resource described by <TT>key</TT> or
     *         <tt>defaultVal</tt> if no value is found.
     */
    public int getPixels(String key, int defaultVal) {
	return SysUtil.pixelsFromPoints(getInt(key, defaultVal));
    }

    /**
     * Search for a pixel resource in the resource stack.  The
     * resource will be interpreted as an integer representing a
     * distance in points (1 point = 1/72 of an inch), and the points
     * will be converted to pixels
     *
     * @param keys An array of Strings containing the resource names to
     *             look up.  The names will be looked up, in order, and
     *             the first resource found will be returned.
     *
     * @param defaultVal The object to return if no matching properties
     *                   are found.  The int will be converted from
     *                   points to pixels
     *
     * @return The first pixel resource that is found matching a key in
     *         <TT>keys</TT> or <tt>defaultVal</tt> if no value is found.
     */
    public int getPixels(String[] keys, int defaultVal) {
	return SysUtil.pixelsFromPoints(getInt(keys, defaultVal));
    }

    //---------------------------- getFloat ----------------------

    /**
     * Search for a float resource in the resource stack
     *
     * @param key Name of the resource to retrieve.
     *
     * @exception java.util.MissingResourceException if the resource
     *            is not found.
     *
     * @return The resource described by <TT>key</TT>.
     */
    public float getFloat(String key) throws MissingResourceException {
	return new Float(getString(key)).floatValue();
    }

     /**
     * Search for a float resource in the resource stack.
     *
     * @param keys An array of Strings containing the resource names to
     *             look up.  The names will be looked up, in order, and
     *             the first resource found will be returned.
     *
     * @exception java.util.MissingResourceException if the property
     *            is not found.
     *
     * @return The first resource that is found matching a key in
     *         <TT>keys</TT>
     */
    public float getFloat(String[] keys) throws MissingResourceException {
	return new Float(getString(keys)).floatValue();
    }

     /**
     * Search for a float resource in the resource stack.
     *
     * @param key Name of the resource to retrieve.
     *
     * @param defaultVal The resource to return if no resource
     *                   matching <tt>key</tt> is found.
     *
     * @return The resource described by <TT>key</TT> or
     *         <tt>defaultVal</tt> if no value is found.
     */
    public float getFloat(String key, float defaultVal) {
	return getFloat(new String[] {key}, defaultVal);
    }

     /**
     *  Search for a float resource in the resource stack.
     *
     * @param keys An array of Strings containing the resource names to
     *             look up.  The names will be looked up, in order, and
     *             the first resource found will be returned.
     *
     * @param defaultVal The object to return if no matching properties
     *                   are found.
     *
     * @return The first resource that is found matching a key in
     *         <TT>keys</TT> or <tt>defaultVal</tt> if no value is found.

     */
    public float getFloat(String[] keys, float defaultVal)  {
	String str = getString(keys, null);
	return str != null ? new Float(str).floatValue() : defaultVal;
    }

    // ---------------------- getFont ---------------------------

     /**
     * Search for a Font resource in the resource stack.  The font
     * must be specifed in a way that can be understood by Font.decode()
     *
     * @param key Name of the resource to retrieve.
     *
     * @exception java.util.MissingResourceException if the resource
     *            is not found.
     *
     * @return The resource described by <TT>key</TT>.
     * @see java.awt.Font#decode(String)
     */
    public Font getFont(String key) throws MissingResourceException {
	return SysUtil.loadFont(getString(key));
    }
     /**
     * Search for a Font resource in the resource stack. The font
     * must be specifed in a way that can be understood by Font.decode()
     *
     * @param keys An array of Strings containing the resource names to
     *             look up.  The names will be looked up, in order, and
     *             the first resource found will be returned.
     *
     * @exception java.util.MissingResourceException if the property
     *            is not found.
     *
     * @return The first resource that is found matching a key in
     *         <TT>keys</TT>
     * @see java.awt.Font#decode(String)
     */
    public Font getFont(String[] keys) throws MissingResourceException {
	return SysUtil.loadFont(getString(keys));
    }

    /**
     * Search for a Font resource in the resource stack. The font
     * must be specifed in a way that can be understood by Font.decode()
     *
     * @param key Name of the resource to retrieve.
     *
     * @param defaultVal The resource to return if no resource
     *                   matching <tt>key</tt> is found.
     *
     * @return The resource described by <TT>key</TT> or
     *         <tt>defaultVal</tt> if no value is found.
     * @see java.awt.Font#decode(String)
     */
    public Font getFont(String key, Font defaultVal) {
	return getFont(new String[]{key} , defaultVal);
    }

    /**
     *  Search for a Font resource in the resource stack. The font
     * must be specifed in a way that can be understood by Font.decode()
     *
     * @param keys An array of Strings containing the resource names to
     *             look up.  The names will be looked up, in order, and
     *             the first resource found will be returned.
     *
     * @param defaultVal The object to return if no matching properties
     *                   are found.
     *
     * @return The first resource that is found matching a key in
     *         <TT>keys</TT> or <tt>defaultVal</tt> if no value is found.
     * @see java.awt.Font#decode(String)
     */
    public Font getFont(String[] keys, Font defaultVal) {
	String str = getString(keys,null);
	return str != null ? SysUtil.loadFont(str) : defaultVal;
    }

    //--------------------------- getColor ------------------------

    /**
     * Search for a Color resource in the resource stack.  The color
     * must be specifed in a way that can be understood by Color.decode()
     *
     * @param key Name of the resource to retrieve.
     *
     * @exception java.util.MissingResourceException if the resource
     *            is not found.
     *
     * @return The resource described by <TT>key</TT>.
     * @see java.awt.Color#decode(String)
     */
    public Color getColor(String key) throws MissingResourceException {
	return Color.decode(getString(key));
    }
     /**
     * Search for a Color resource in the resource stack. The color
     * must be specifed in a way that can be understood by Color.decode()
     *
     * @param keys An array of Strings containing the resource names to
     *             look up.  The names will be looked up, in order, and
     *             the first resource found will be returned.
     *
     * @exception java.util.MissingResourceException if the property
     *            is not found.
     *
     * @return The first resource that is found matching a key in
     *         <TT>keys</TT>
     * @see java.awt.Color#decode(String)
     */
    public Color getColor(String[] keys) throws MissingResourceException {
	return Color.decode(getString(keys));
    }

    /**
     * Search for a Color resource in the resource stack. The color
     * must be specifed in a way that can be understood by Color.decode()
     *
     * @param key Name of the resource to retrieve.
     *
     * @param defaultVal The resource to return if no resource
     *                   matching <tt>key</tt> is found.
     *
     * @return The resource described by <TT>key</TT> or
     *         <tt>defaultVal</tt> if no value is found.
     * @see java.awt.Color#decode(String)
     */
    public Color getColor(String key, Color defaultVal) {
	return getColor(new String[]{key} , defaultVal);
    }

    /**
     *  Search for a Color resource in the resource stack. The color
     * must be specifed in a way that can be understood by Color.decode()
     *
     * @param keys An array of Strings containing the resource names to
     *             look up.  The names will be looked up, in order, and
     *             the first resource found will be returned.
     *
     * @param defaultVal The object to return if no matching properties
     *                   are found.
     *
     * @return The first resource that is found matching a key in
     *         <TT>keys</TT> or <tt>defaultVal</tt> if no value is found.
     * @see java.awt.Color#decode(String)
     */
    public Color getColor(String[] keys, Color defaultVal) {
	String str = getString(keys,null);
	return str != null ? Color.decode(str) : defaultVal;
    }

    //------------------------ getBoolean -------------------------

    private Boolean getBooleanImpl(String[] keys, Boolean defaultVal) {
	String value = getString(keys, null);
	if (value == null) {
	    return defaultVal;
	}
	if (value.equalsIgnoreCase("true")) {
	    return Boolean.TRUE;
	} else if (value.equalsIgnoreCase("false")) {
	    return Boolean.FALSE;
	} else {
	    Log.warning(CLASS_NAME,
			"A resource was set to \"" + value + 
			"\", which was not recognized as a boolean value." +
			"  Returning false.");
	    return Boolean.FALSE;
	}
    }
    
    /**
     * Search for a boolean resource in the resource stack.  Set the
     * resource to "true" to represent true, anything else for false.
     *
     * @param key Name of the resource to retrieve.
     *
     * @exception java.util.MissingResourceException if the resource
     *            is not found.
     *
     * @return The resource described by <TT>key</TT>.
     */
    public boolean getBoolean(String key) throws MissingResourceException {
	return getBoolean(new String[] {key});
    }

    /**
     * Search for a boolean resource in the resource stack.  Set the
     * resource to "true" to represent true, anything else for false.
     *
     * @param keys An array of Strings containing the resource names to
     *             look up.  The names will be looked up, in order, and
     *             the first resource found will be returned.
     *
     * @exception java.util.MissingResourceException if the property
     *            is not found.
     *
     * @return The first resource that is found matching a key in
     *         <TT>keys</TT>
     */
    public boolean getBoolean(String[] keys) throws MissingResourceException {
	Boolean returnVal = getBooleanImpl(keys, null);
	if (returnVal != null) {
	    return returnVal.booleanValue();
	}
	throw createMissingResourceException(keys);
    }

    /**
     * Search for a boolean resource in the resource stack.  Set the
     * resource to "true" to represent true, anything else for false.
     *
     * @param key Name of the resource to retrieve.
     *
     * @param defaultVal The resource to return if no resource
     *                   matching <tt>key</tt> is found.
     *
     * @return The resource described by <TT>key</TT> or
     *         <tt>defaultVal</tt> if no value is found.
     */
    public boolean getBoolean(String key, boolean defaultVal) {
	return getBoolean(new String[] {key}, defaultVal);
    }

    /**
     *  Search for a boolean resource in the resource stack.  Set the
     * resource to "true" to represent true, anything else for false.
     *
     * @param keys An array of Strings containing the resource names to
     *             look up.  The names will be looked up, in order, and
     *             the first resource found will be returned.
     *
     * @param defaultVal The object to return if no matching properties
     *                   are found.
     *
     * @return The first resource that is found matching a key in
     *         <TT>keys</TT> or <tt>defaultVal</tt> if no value is found.
     */
    public boolean getBoolean(String[] keys, boolean defaultVal) {
	return getBooleanImpl(
	    keys, defaultVal ? Boolean.TRUE : Boolean.FALSE).booleanValue();
    }

    /**
     * Convert a String to a Byte, based on a mapping.
     * <tt>strings</tt> will be searched for <tt>input</tt>, and the
     * corresponding Byte from <tt>bytes</tt> will be returned.  
     * <p>
     * For example, a call to the following will return a Byte set to 1:
     * <pre>
     * mapStringToBytes("NO", 
     *                  new String[] {"YES", "NO", "MAYBE"},
     *                  new Byte[] {0, 1, 2));
     * </pre>
     *
     *
     * @param input The String to convert
     * @param strings The list of Strings
     * @param bytes The list of bytes
     */
    public static byte mapStringToByte(String input,
				       String[] strings,
				       byte[] bytes) {
	for (int ii = 0; ii < strings.length; ii++) {
	    if (input.equals(strings[ii])) {
		return bytes[ii];
	    }
	}
	throw new IndexOutOfBoundsException(
	    "String \"" + input + "\" not found in string array");
    }

    
    /**
     * Search for an array of TaskLoaders in the resource stack.
     *
     * For example, if a resource file contained:
     * <pre>
     * myTasks0 = com.sgi.myPackage.CreateTask
     * myTasks1 = com.sgi.myPackage.ModifyTask
     * myTasks2 = com.sgi.myPackage.DeleteTask
     * </pre>
     * The a call to <tt>getTaskLoaders("myTasks", _hostContext)</tt>
     * would return an array of three TaskLoaders corresponding to the
     * Tasks mentioned.
     * 
     * @param baseKey The base key of the TaskLoaders to load.  The
     * format of the resources should be the same as the
     * getStringArray, and the values should be the fully qualified
     * names of Tasks.
     *
     * @exception java.util.MissingResourceException if the resource
     *            is not found.
     *
     * @return An array of TaskLoaders corresponding to the Tasks
     * described by <TT>key</TT>
     * @see #getStringArray(String)
     */
    public TaskLoader[] getTaskLoaders(String baseKey, HostContext hc) {
	return getTaskLoaders(new String[]{baseKey}, hc);
    }
    
    /**
     * Search for an array of TaskLoaders in the resource stack. 
     *
     * See the version of getTaskLoaders that takes a single baseKey
     * for more information.
     *
     * @param baseKeys An array of Strings containing the resource names to
     *                 look up.  The names will be looked up, in order, and
     *                 the first resource found will be returned.  The
     *                 format of the resources should be the same as
     *                 the getStringArray, and the values should be
     *                 the fully qualified names of Tasks.
     *
     * @exception java.util.MissingResourceException if the property
     *            is not found.
     *
     * @return The first resource that is found matching a key in
     *         <TT>keys</TT>
     * @see #getTaskLoaders(String, HostContext)
     * @see #getStringArray(String[])
     */
    public TaskLoader[] getTaskLoaders(String[] baseKeys, HostContext hc) {
	String[] taskNames = getStringArray(baseKeys);
	Vector loaderList = new Vector(taskNames.length);
	TaskLoader taskLoader;
	for (int ii = 0; ii < taskNames.length; ii++) {
	    try {
		taskLoader = new TaskLoader(hc, taskNames[ii]);
	    } catch (MissingResourceException exception) {
		Log.warning(_class, exception.getMessage());
		continue;
	    }
	    loaderList.addElement(taskLoader);
	}
	TaskLoader[] loaderArray = new TaskLoader[loaderList.size()];
	loaderList.copyInto(loaderArray);
	return loaderArray;
    }
}
