//
//    BinaryPredicate.java
//
//	Interface for comparing to Objects.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//
package com.sgi.sysadm.util;

/**
 * A class that encapsulates mutating sequence algorithms on one
 * and two arrays.  All methods are static and all variables are
 * static and final, so this class has no constructors.
 *
 *<P>
 * Most methods operate on a range of elements.  A range is described
 * by the index of its first element and an index that is 
 * <strong>one past</strong> its last element.  So, for example,
 * <code>[n, n+1)</code> is a range that contains one element,
 * <code>[n, n)</code> is a range that contains zero elements,
 * and <code>[n, n-1)</code> is not a valid range.
 *
 * <P>
 * Unless otherwise specified, the test for equality uses
 * the <code>==</code> operator by default.  Any different notion of
 * equality may be represented as a BinaryPredicate.  You can use the
 * predefined class Equals, which implements BinaryPredicate, if you want 
 * to use the <code>Object.equals()</code> method.
 *
 * @author Matthew Austern (austern@mti.sgi.com)
 * @author Alexander Stepanov (stepanov@mti.sgi.com)
 * @see BinaryPredicate
 */

public final class Sort {
    private static final int stableSortCutoff = 9;

  /**
   * Sort a range of elements by a user-supplied comparison function.
   * The sort is stable---that is, the relative order of equal elements
   * is unchanged.  Worst case performance is <code>N (log N)^2</code>.
   * @param array       Array containing the range.
   * @param first       Beginning of the range.
   * @param last        One past the end of the range.
   * @param comp        Comparison function.
   */
  public static void stable_sort(Object[] array, int first, int last,
                                 BinaryPredicate comp)
    {
      if (last - first < stableSortCutoff) 
        insertion_sort(array, first, last, comp);
      else {
        int middle = first + (last - first) / 2;
        stable_sort(array, first, middle, comp);
        stable_sort(array, middle, last, comp);
        inplace_merge(array, first, middle, last, comp);
      }
    }

 /**
   * Sort a range of elements by a user-supplied comparison function.
   * Uses the insertion sort algorithm.  This is a quadratic
   * algorithm, but it is useful for sorting small numbers of elements.
   * @param array       Array containing the range.
   * @param first       Beginning of the range.
   * @param last        One past the end of the range.
   * @param comp        Comparison function.  
   */
  public static void insertion_sort(Object[] array, int first, int last,
                                    BinaryPredicate comp)
    {
      for (int current = first; ++current < last; /* */ ) { 
        Object tmp = array[current];
        int i = current;
        for (Object tmp1 = array[i - 1];
             comp.apply(tmp, tmp1); 
             tmp1 = array[--i - 1] ) {
          array[i] = tmp1;
          if (first == i - 1) {
            --i;
            break;
          }
        }
        array[i] = tmp;
      }
    }

 /**
   * Transforms two consecutive sorted ranges into a single sorted 
   * range.  The initial ranges are <code>[first, middle)</code>
   * and <code>[middle, last)</code>, and the resulting range is
   * <code>[first, last)</code>.  
   * Elements in the first input range will precede equal elements in the 
   * second.
   * Sorting is by a user-supplied comparison function.
   * @param array    Array containing the ranges.
   * @param first    Beginning of the first range.
   * @param middle   One past the end of the first range, and beginning
   *                 of the second.
   * @param last     One past the end of the second range.
   * @param comp     Comparison function.
   */
  public static void inplace_merge(Object[] array, 
                                   int first, int middle, int last,
                                   BinaryPredicate comp)
    {

      if (first >= middle || middle >= last)
        return;

      if (last - first == 2) {
        if (comp.apply(array[middle], array[first])) {
          Object tmp = array[first];
          array[first] = array[middle];
          array[middle] = tmp;
        }
        return;
      }

      int firstCut;
      int secondCut;

      if (middle - first > last - middle) {
        firstCut = first + (middle - first) / 2;
        secondCut = lower_bound(array, middle, last, array[firstCut], comp);
      }
      else {
        secondCut = middle + (last - middle) / 2;
        firstCut = upper_bound(array, first, middle, array[secondCut], comp);
      }

      rotate(array, firstCut, middle, secondCut);
      middle = firstCut + (secondCut - middle);

      inplace_merge(array, first, firstCut, middle, comp);
      inplace_merge(array, middle, secondCut, last, comp);
    }

  /**
   * Performs a binary search on an already-sorted range: finds the first
   * position where an element can be inserted without violating the ordering.
   * Sorting is by a user-supplied comparison function.
   * @param array    Array containing the range.
   * @param first    Beginning of the range.
   * @param last     One past the end of the range.
   * @param x        Element to be searched for.
   * @param comp     Comparison function.
   * @return         The largest index i such that, for every j in the
   *                 range <code>[first, i)</code>, 
   *                 <code>comp.apply(array[j], x)</code> is
   *                 <code>true</code>.
   */
  public static int lower_bound(Object[] array, int first, int last,
                                Object x,
                                BinaryPredicate comp)
    {
      int len = last - first;
      while (len > 0) {
        int half = len / 2;
        int middle = first + half;
        if (comp.apply(array[middle], x)) {
          first = middle + 1;
          len -= half + 1;
        } else
          len = half;
      }
      return first;
    }

  /**
   * Performs a binary search on an already-sorted range: finds the last
   * position where an element can be inserted without violating the ordering.
   * Sorting is by a user-supplied comparison function.
   * @param array    Array containing the range.
   * @param first    Beginning of the range.
   * @param last     One past the end of the range.
   * @param x        Element to be searched for.
   * @param comp     Comparison function.
   * @return         The largest index i such that, for every j in the
   *                 range <code>[first, i)</code>, 
   *                 <code>comp.apply(x, array[j])</code> is 
   *                 <code>false</code>.
   */
  public static int upper_bound(Object[] array, int first, int last,
                                Object x,
                                BinaryPredicate comp)
  {
      int len = last - first;
      while (len > 0) {
        int half = len / 2;
        int middle = first + half;
        if (comp.apply(x, array[middle]))
          len = half;
        else {
          first = middle + 1;
          len -= half + 1;
        }
      }
      return first;
    }

  /** 
   * Reverses a sequence of elements.
   * @param array      Array containing the sequence
   * @param first      Beginning of the range
   * @param last       One past the end of the range
   * @exception        ArrayIndexOutOfBoundsException If the range
   *                   is invalid.
   */
  static public void reverse(Object[] array, int first, int last)
    {
      while (first < --last) {
        Object tmp = array[first];
        array[first++] = array[last];
        array[last] = tmp;
      }
    }

  /**
   * Rotate a range in place: <code>array[middle]</code> is put in
   * <code>array[first]</code>, <code>array[middle+1]</code> is put in
   * <code>array[first+1]</code>, etc.  Generally, the element in position
   * <code>i</code> is put into position 
   * <code>(i + (last-middle)) % (last-first)</code>.
   * @param array    Array containing the range
   * @param first    Beginning of the range
   * @param middle   Index of the element that will be put in
   *                 <code>array[first]</code>
   * @param last     One past the end of the range
   */
  public static void rotate(Object[] array, int first, int middle, 
                                 int last)
    {
      if (middle != first && middle != last) {
        reverse(array, first, middle);
        reverse(array, middle, last);
        reverse(array, first, last);
      }
    }
}
