//
// SysUtil.java
//
//	System utilities
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//
package com.sgi.sysadm.util;

import java.net.*;
import java.util.*;
import java.io.*;
import java.awt.*;
import java.lang.reflect.*;
import javax.swing.plaf.FontUIResource;

/**
 * System utility functions.
 */
public class SysUtil {
    //
    // The timer methods of this class implement a stop watch facility
    // which can be used to collect runtime performance speed data.  The
    // idea is that for each section of code to be measured, a named timer
    // is started before running the code and stopped after running the
    // code.  The code being measured can be very fine-grained, and a
    // timer can be started and stopped many times to accumulate execution
    // time over the course of a long run of a program.  The value of a
    // timer is accessed by calling <tt>printTimer</tt>.
    //
    // The timer methods are all package protected because we don't
    // want to support the API.
    //
    private static boolean _propertyException = false;
    private static Hashtable _timers = new Hashtable();
    private static final String CLASS_NAME = "SysUtil";

    // Stuff for pixelsFromPoints and sizeFromWidth.
    private static float _scaleFactor;
    private static float _goldenRatio = (float)8.5/(float)11.0;
    static {
	Toolkit tk = Toolkit.getDefaultToolkit();
	Font font = new Font("SansSerif", Font.PLAIN, 14);
	FontMetrics metrics = tk.getFontMetrics(font);

	_scaleFactor = (float)metrics.getHeight()/(float)font.getSize();
    }

    /**
     * Package visible constructor so it won't show up in javadoc.
     */
    SysUtil() {
    }

    /**
     * Timer keeps track of time accumulation, for measuring execution
     * time.
     */
    private static class Timer {
	long _startTime = 0;
	long _totalTime = 0;
	boolean _running = false;
    }

    /**
     * Get a named timer.  Public API deals with timers only by name;
     * the Timer structure is private to SysUtil.
     *
     * @param timerName Name of timer to get.
     *
     * @return Timer for <tt>timerName</tt>
     */
    private static Timer getTimer(String timerName) {
	Timer timer = (Timer)_timers.get(timerName);
	if (timer == null) {
	    timer = new Timer();
	    _timers.put(timerName, timer);
	}
	return timer;
    }

    /**
     * Start the timer named by <tt>timerName</tt>.
     *
     * <B>Note:</B> This is indended for internal Rhino development
     * and is subject to change.
     *
     * @param timerName Name of timer to start.
     */
    public static void startTimer(String timerName) {
	Timer timer = getTimer(timerName);
	Log.assert(!timer._running,
		   "Timer: " + timerName + " is already running");
	timer._running = true;
	timer._startTime = System.currentTimeMillis();
    }

    /**
     * Stop the timer named by <tt>timerName</tt>.
     *
     * <B>Note:</B> This is indended for internal Rhino development
     * and is subject to change.
     *
     * @param timerName Name of timer to stop.
     */
    public static void stopTimer(String timerName) {
	Timer timer = getTimer(timerName);
	Log.assert(timer._running,
		   "Timer: " + timerName + " is not running");
	timer._running = false;
	timer._totalTime += System.currentTimeMillis() - timer._startTime;
    }

    /**
     * Stop the timer named by <tt>timerName</tt>.
     *
     * <B>Note:</B> This is indended for internal Rhino development
     * and is subject to change.
     *
     * @param timerName Name of timer to stop.
     */
    public static void resetTimer(String timerName) {
	Timer timer = getTimer(timerName);
	timer._startTime = 0;
	timer._totalTime = 0;
	timer._running = false;
    }

    /**
     * Print timer information for <tt>timerName</tt>.
     *
     * <B>Note:</B> This is indended for internal Rhino development
     * and is subject to change.
     *     
     * @param timerName Name of timer to print.
     */
    public static void printTimer(String timerName) {
	Timer timer = getTimer(timerName);
	System.out.println(timerName + ": " + timer._totalTime);
    }

    /**
     * Get the timer information for <tt>timerName</tt>.
     *
     * <B>Note:</B> This is indended for internal Rhino development
     * and is subject to change.
     *
     * @param timerName Name of timer to print.
     * @return The number of milliseconds that the timer ran.
     */
    public static long getTime(String timerName) {
	return  getTimer(timerName)._totalTime;
    }
    /**
     * Reset timer for <tt>timerName</tt>, and restart it.
     *
     * <B>Note:</B> This is indended for internal Rhino development
     * and is subject to change.
     *
     * @param timerName Name of timer to reset and start.
     */
    public static void resetAndStartTimer(String timerName) {
	resetTimer(timerName);
	startTimer(timerName);
    }

    /**
     * Stop timer for <tt>timerName</tt> and print its value.
     *
     * <B>Note:</B> This is indended for internal Rhino development
     * and is subject to change.
     *
     * @param timerName Name of timer to stop and print.
     */
    public static void stopAndPrintTimer(String timerName) {
	stopTimer(timerName);
	printTimer(timerName);
    }

    /**
     * <tt>getSystemProperty</tt> is a wrapper for
     * <tt>System.getProperty</tt> that does not cause a torrent of
     * exceptions when called from an untrusted Applet.  This version
     * returns null if the property does not exist.
     *
     * @param key Key of property to get.
     *
     * @return value of property, or null if no such property exists.
     * @see java.lang.System#getProperty
     */
    public static String getSystemProperty(String key) {
	return getSystemProperty(key, null);
    }

    /**
     * <tt>getSystemProperty</tt> is a wrapper for
     * <tt>System.getProperty</tt> that does not cause a torrent of
     * exceptions when called from an untrusted Applet.  This version
     * returns <tt>defValue</tt> if the property does not exist.
     *
     * @param key Key of property to get.
     * @param defValue Default value to return if property does not
     *                 exist.
     *
     * @return value of property, or <tt>defValue</tt> if no such
     *         property exists.
     * @see java.lang.System#getProperty
     */
    public static String getSystemProperty(String key, String defValue) {
	if (!_propertyException) {
	    try {
		return System.getProperty(key, defValue);
	    } catch (Exception ex) {
		_propertyException = true;
	    }
	}
	return defValue;
    }

    /**
     * Read one line of input from <tt>in</tt>.
     *
     * @param in InputStream to read from
     *
     * @exception IOException if an I/O problem occurs.
     *
     * @return String representing the line of input.
     */
    public static String readLine(InputStream in) throws IOException {
	final int READ_LENGTH = 512;
	byte[] buf = new byte[READ_LENGTH];
	int length = 0;
	while (true) {
	    int c = in.read();
	    if (c == -1) {
		throw new IOException();
	    }
	    if (c == '\r' || c == '\n') {
		if (length == 0) {
		    continue;
		}
		break;
	    }
	    // Make our buffer bigger if necessary.
	    if (length == buf.length) {
		byte[] newBuf = new byte[buf.length + READ_LENGTH];
		System.arraycopy(buf, 0, newBuf, 0, length);
		buf = newBuf;
	    }
	    buf[length++] = (byte)c;
	}

	return new String(buf, 0, length);
    }

    /**
     * A static method to load a class and instantiate an object.
     * This method instantiates the class named by <tt>className</tt>,
     * passing the <tt>params</tt> to the constructor, and makes sure
     * that what it loads is of type <tt>returnType</tt>.  If the
     * <tt>returnType</tt> does not match, or if an exception is
     * thrown during object load or instantiation, then a
     * ClassLoadException is thrown.
     * <p>
     * This method is a convenience method so that callers don't have
     * to handle the messy details of loading a class, instantiating
     * an object, and handling all the Exceptions that could be thrown.
     *
     * @param className The name of the class to load
     * @param returnType The type of the Object to return.  If the
     *                   Class that is loaded does not match this
     *                   type, a <tt>ClassLoadException</tt> is thrown.
     * @param paramTypes The types of the arguments to be passed to
     *                   the constructor
     * @param params The arguments to be passed to the constructor
     *
     * @exception ClassNotFoundException if Class.forName method fails
     *            when passed <TT>className</TT>.
     * @exception SysUtil.ClassLoadException if there are any exceptions
     *            thrown during load or instantiation of the Object.
     */
    public static Object createObject(String className,
				      Class returnType,
				      Class[] paramTypes,
				      Object[] params)
	throws ClassNotFoundException, ClassLoadException {
	Class newClass;
	try {
	    newClass = Class.forName(className);
	} catch (ClassNotFoundException cnfe) {
	    Log.debug(CLASS_NAME,"Couldn't load class " +
		      className + ".  The error was: " +
		      "ClassNotFoundException");
	    throw cnfe;
	} catch (Error e) {
	     Log.debug(CLASS_NAME,"Couldn't load class " +
		       className);
	     throw e;
	}

	// if we're here, then newClass is good.
	Constructor constructor;
	try {
	    constructor = newClass.getConstructor(paramTypes);
	    Object obj = constructor.newInstance(params);
	    if (!returnType.isInstance(obj)) {
		throw new ClassCastException("Tried to cast " +
					     obj.getClass().getName()
					     + " to a " +
					     returnType.getName());
	    }
	    return obj;
	} catch (ClassCastException exception) {
	    Log.debug(CLASS_NAME, "An exception of type " +
		      exception.getClass().getName() +
		      " with message: \"" +
		      exception.getMessage() +
		      "\" was thrown while trying to load " +
		      className);
	    throw new ClassLoadException(className, exception.getMessage());
	} catch (RuntimeException exception) {
	    throw exception;
	} catch (Exception exception) {
	    Log.error(CLASS_NAME, "An exception of type " +
		      exception.getClass().getName() +
		      " with message: \"" +
		      exception.getMessage() +
		      "\" was thrown while trying to load " +
		      className + ".  The stack trace was: \n" +
		      stackTraceToString(exception));
	    if (exception instanceof InvocationTargetException) {
		Throwable origException =
		    ((InvocationTargetException)exception).
		    getTargetException();
		Log.error(CLASS_NAME,
			  "Cause of InvocationTargetException was " +
			  origException.toString());
		origException.printStackTrace();
	    }
	    throw new ClassLoadException(className, exception.getMessage());
	}
    }

    /**
     * A class that gets thrown in a Class can't be loaded and
     * instantiated.  It encapulates all the Exceptions that are thrown
     * by a class load and instantiation.
     * <p>
     * Calling <tt>getClassName</tt> will return the name of the Class
     * that was being loaded, and <tt>getErrorString</tt> will return an
     * error message (not localized) about the nature of the error.
     *
     */
    public static class ClassLoadException extends Exception {
	private String _className;

	/**
	 * A constructor that takes a className and errorString
	 * @param className The name of the class that was being loaded
	 * @param errorString A string describing the error
	 */
	public ClassLoadException(String className, String errorStr) {
	    super(errorStr);
	    _className = className;
	}

	/**
	 * A constructor that takes a className.
	 * @param className The name of the class that was being loaded*
	 */
	public ClassLoadException(String className) {
	    super();
	    _className = className;
	}

	/**
	 * Returns the <code>className</code> passed to the constructor
	 */
	public String getClassName() {
	    return _className;
	}

	/**
	 * Returns the <code>errorStr</code> passed to the constructor
	 */
	public String getErrorString() {
	    return getMessage();
	}
    }

    /**
     * Get the stack trace from <tt>th</tt> as a String.
     *
     * @param th Throwable from which to get a stack trace.
     *
     * @return String containing stack trace.
     */
    public static String stackTraceToString(Throwable th) {
	StringWriter sw = new StringWriter();
	th.printStackTrace(new PrintWriter(sw));
	return sw.getBuffer().toString();
    }

    /**
     * Compares the IP addresses of the 2 hosts to see if they are the
     * same.  <tt>hostsEqual</tt> attempts to resolve both host names
     * and compare the resulting IP Addresses.  If a SecurityException
     * occurs because we're running in an untrusted Applet,
     * <tt>hostsEqual</tt> falls back to doing a string compare.
     * <p>
     * If either hostname cannot be resolved for non-security reasons,
     * <tt>false</tt> is returned, even if the hostnames are the same.
     *
     * <B>Warning:</B> This call can block for a long time (many
     * seconds) while the names are resolved.  Don't call this from a
     * thread that can't block.
     *
     * @param  hostname1 A Hostname.
     * @param  hostname2 Another Hostname.
     * @return <tt>true</tt> if IP addresses of the hostnames
     *         exist and are the same. Returns <tt>false</tt>
     *         otherwise.
     */
    public static boolean hostsEqual(String hostname1, String hostname2) {

	String ipAddr1 = null;
	boolean equal = hostname1.equals(hostname2);
	if (equal) {
	    return true;
	}
	
	try {
	    try {
		InetAddress addr1 = InetAddress.getByName(hostname1);
		ipAddr1 = addr1.getHostAddress();
	    } catch(UnknownHostException e) {
		return false;
	    }

	    String ipAddr2 = null;
	    try {
		InetAddress addr2 = InetAddress.getByName(hostname2);
		ipAddr2 = addr2.getHostAddress();
	    } catch(UnknownHostException e) {
		return false;
	    }

	    if (ipAddr2.equals(ipAddr1)) {
		return true;
	    } else {
		return false;
	    }
	} catch (Throwable ex) {
	    // We get here if we're an applet.  Catch Throwable
	    // instead of SecurityException to work with more JVMs.
	    return false;
	}
    }

    /**
     * Convert points to pixels.  Experimentation has revealed that
     * there is no deterministic relationship between the screen
     * resolution returned by java.awt.Toolkit and and the number of
     * pixels in a point, even though a point is supposed to be 1/72
     * of an inch.  So we determine our scale factor by picking a font
     * and dividing its pixel height by its point size.
     *
     * @param points points to convert to pixels.
     *
     * @return points converted to pixels.
     */
    public static int pixelsFromPoints(int points) {
	return (int)(points * _scaleFactor);
    }

    /**
     * Given a width in points, calculates the height based on the "golden"
     * UI ratio and returns a Dimension in pixels.  The golden ratio
     * is 8.5:11.
     *
     * @param widthInPoints The width of an interface, in points.
     *
     * @return A Dimension in pixels where height has been calculated from
     *         widthInPoints using the "golden" UI ratio.
     */
    public static Dimension sizeFromWidth(int widthInPoints) {
	int widthInPixels = pixelsFromPoints(widthInPoints);
	int heightInPixels = (int)(widthInPixels*_goldenRatio);

	return new Dimension(widthInPixels, heightInPixels);
    }

    /**
     * Works around a bug in the code that loads classes for applets in
     * Java 1.2 (Java bug 4191520). Sometimes instead of throwing a
     * ClassNotFoundException, the class loader incorrectly throws a
     * ClassFormatError instead.  This method catches the
     * ClassFormatError and then tries to determine if the app is
     * running as an applet by looking at the stack trace.  If it's
     * determined that it was in an applet, then a warning will be
     * printed to the console and a ClassNotFoundException will be
     * thrown.  Otherwise it's probably a real ClassFormatException,
     * and the method re-throws the original exception.
     */
    public static Class forName(String className)
	throws ClassNotFoundException {

	try {
	    return Class.forName(className);
	} catch (ClassFormatError cfe) {
	    String stack = stackTraceToString(cfe);
	    if ((stack.indexOf("sun.applet.AppletClassLoader.findClass") > 0)) {
		System.out.println(
		    "Got a ClassFormatError while loading a class.\n" +
		    "This is most likely a result of Java bug 4191520,\n" +
		    "and is caused by a missing class.  Will re-throw a\n" +
		    "ClassNotFoundException.  The original error was\n" +
		    cfe.getMessage());
		throw new ClassNotFoundException(className);
	    }
	    else {
		throw cfe;
	    }
	}
    }

    private static Hashtable _fonts = new Hashtable();

    /**
     * Returns a font that matches the description given.  All fonts
     * loaded via this method are cached, and subsequent requests for
     * a font are returned from the cache.
     *
     * @param name The name of the Font
     * @param style The style of the font (e.g. Font.BOLD)
     * @param size The size of the font
     *
     * @return A font matching the desctription.
     */
    public static Font loadFont(String name, int style, int size) {
	String fontKey;
	fontKey = name.toLowerCase() + style + size;
	Font font = (Font)_fonts.get(fontKey);
	if (font != null) {
	    return font;
	}
	else {
	    // Create a new FontUIResource (which is just a Font
	    // subclass for Swing) so that we can use the Fonts both
	    // in normal circumstances and as a UIDefs in RApp,
	    // RApplet.
	    font = new FontUIResource(name, style, size);
	    _fonts.put(fontKey, font);
	    return font;
	}
    }
    /**
     * Returns a font that matches the description given.  All fonts
     * loaded via this method are cached, and subsequent requests for
     * a font are returned from the cache.
     *
     * @param description A desciption of a font in the form
     * &lt;name&gt;-&lt;style&gt;-&lt;size&gt; (The same format taken
     * by Font.decode())
     *
     * @return A font matching the desctription.
     */
    public static Font loadFont(String description) {
	String fontName = description;
	int fontSize = 12;
	int fontStyle = Font.PLAIN;

	int i = description.indexOf('-');
	if (i >= 0) {
	    fontName = description.substring(0, i);
	    description = description.substring(i+1);
	    if ((i = description.indexOf('-')) >= 0) {
		if (description.startsWith("bold-")) {
		     fontStyle = Font.BOLD;
		} else if (description.startsWith("italic-")) {
		     fontStyle = Font.ITALIC;
		} else if (description.startsWith("bolditalic-")) {
		     fontStyle = Font.BOLD | Font.ITALIC;
		}
		description = description.substring(i + 1);
	    }
	    try {
		fontSize = Integer.parseInt(description);
	    } catch (NumberFormatException e) {
	    }
	}
	return loadFont(fontName, fontStyle, fontSize);
    }
}
