//
// mkrhinoism.c++
//
//	mkrhinoism privileged command.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2 of the GNU General Public License as
//  published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

#ident "$Revision: 1.5 $"

#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <limits.h>
#include <sysadm/privdefs.h>
#include <sysadm/SaParam.h>
#include <sysadm/format.h>

static const char* gCategory;
static const char* gIsm;
static const char* gProduct;
static const char* gDir;
static const char* gRoot;
static const char* gToolroot;

static void CreateRhinoIsm()
{
    printf("Creating Rhino ISM...\n");
    fflush(stdout);
    pid_t pid = fork();
    if (pid == -1) {
	perror("fork");
	exit(SaFIRSTAPPCODE);
    }
    if (pid == 0) {
	setuid(getuid());
	setgid(getgid());
	execl("/usr/sbin/mkrhinoism", "mkrhinoism",
	      "-o", gCategory, "-p", gProduct, "-d", gDir, "-i", gIsm, NULL);
	perror("/usr/sbin/mkrhinoism");
	exit(1);
    }
    int status;
    while (waitpid(pid, &status, 0) == -1) {
	if (errno != EINTR) {
	    perror("waitpid");
	    exit(SaFIRSTAPPCODE);
	}
    }
    if (WEXITSTATUS(status) != 0) {
	exit(SaFIRSTAPPCODE);
    }
}

static void RunInst(const char* selectionsFile, const char* root)
{
    if (root == NULL) {
	return;
    }
    printf("Installing software into %s...\n", root);
    fflush(stdout);
    if (strlen(root) < 2 || root[0] != '/') {
	fprintf(stderr,
		"mkrhinoism: %s is not a full pathname other than \"/\".\n",
		root);

	exit(SaFIRSTAPPCODE);
    }
    int pipes[2];
    if (pipe(pipes) == -1) {
	perror("pipe");
	exit(SaFIRSTAPPCODE);
    }
    pid_t pid = fork();
    if (pid == -1) {
	perror("fork");
	exit(SaFIRSTAPPCODE);
    }
    if (pid == 0) {
	setuid(geteuid());
	setgid(getegid());
	dup2(pipes[1], 1);
	close(pipes[0]);
	close(pipes[1]);
	execl("/usr/sbin/inst", "inst", "-a", "-F", selectionsFile,
	      "-r", root, "-M", NULL);
	perror("/usr/sbin/inst");
	exit(1);
    }

    // Indent inst's output.
    close(pipes[1]);
    FILE* fp = fdopen(pipes[0], "r");
    if (fp != NULL) {
	char buf[1000];
	while (fgets(buf, sizeof buf, fp) != NULL) {
	    fprintf(stdout, "    %s", buf);
	    fflush(stdout);
	}
	fclose(fp);
    }

    int status;
    while (waitpid(pid, &status, 0) == -1) {
	if (errno != EINTR) {
	    perror("waitpid");
	    exit(SaFIRSTAPPCODE);
	}
    }

    // The "6" is the exit status inst uses when nothing was installed
    // because everything selected for installation was already
    // installed.
    if (WEXITSTATUS(status) != 0 && WEXITSTATUS(status) != 6) {
	fprintf(stderr, "Error: inst exited with non-zero exit code.\n");
	exit(SaFIRSTAPPCODE);
    }
}

static void CreateToolroot()
{
    RunInst("http://ethyl.engr.sgi.com/rhino/ismtools/1.1/troot.selections",
	    gToolroot);
}

static void CreateRoot()
{
    // Exitops from ROOT need TOOLROOT.
    if (gToolroot != NULL) {
	int len = strlen("TOOLROOT=") + strlen(gToolroot) + 1; 
	char buf[len];
	SaStringFormat(buf, len, "TOOLROOT=%s", gToolroot);
	putenv(buf);
    }
    RunInst("http://ethyl.engr.sgi.com/rhino/ismtools/1.1/root.selections",
	    gRoot);
}

static void CreateBuildScript()
{
    printf("Creating build script...\n");
    fflush(stdout);
    setuid(getuid());
    setgid(getgid());
    char buf[PATH_MAX];
    SaStringFormat(buf, sizeof buf, "%s/makeme", gDir);
    FILE* fp = fopen(buf, "w");
    if (fp == NULL) {
	perror(buf);
	exit(1);
    }
    fprintf(fp, "#!/bin/sh\n");
    fprintf(fp, "#\n#This script is used to build the %s ism.\n", gIsm);
    fprintf(fp, "WORKAREA=%s\n", gDir);
    fprintf(fp, "export WORKAREA\n");
    if (gRoot != NULL) {
	fprintf(fp, "ROOT=%s\n", gRoot);
	fprintf(fp, "export ROOT\n");
    }
    if (gToolroot != NULL) {
	fprintf(fp, "TOOLROOT=%s\n", gToolroot);
	fprintf(fp, "export TOOLROOT\n");
    }
    fprintf(fp, "make rawidb && make startversion && make images\n");
    fclose(fp);
    chmod(buf, 0755);

    // Create a census file so that "make images" will work.
    SaStringFormat(buf, sizeof buf, "%s/census", gDir);
    close(open(buf, O_CREAT | O_WRONLY, 0644));
}

void Usage()
{
    fprintf(stderr, "usage: mkrhinoism key=value ...\n");
    fprintf(stderr, "\tobject=object_type_name\n"
	    "\tproduct=product_name\n"
	    "\tdir=dest_directory\n"
	    "\tism=ism_name\n"
	    "\t[root=root_location]\n"
	    "\t[troot=toolroot_location]\n"
	    "key=value pairs enclosedin [ ] are optional.\n");
    exit(SaPRIVUSAGE);
}

void main(int argc, char* argv[])
{
    SaParam* param = SaParamCreate();
    if (param == NULL) {
	fprintf(stderr, "Out of memory!\n");
	exit(SaPRIVNOMEM);
    }
    
    if (SaParamParseArgs(param, argc, argv) == -1) {
	Usage();
    }
    
    gCategory = SaParamGet(param, "object");
    gProduct = SaParamGet(param, "product");
    gDir = SaParamGet(param, "dir");
    gIsm = SaParamGet(param, "ism");
    gRoot = SaParamGet(param, "root");
    gToolroot = SaParamGet(param, "troot");

    if (gCategory == NULL || gProduct == NULL
	|| gDir == NULL || gIsm == NULL) {
	Usage();
    }
    CreateRhinoIsm();
    CreateToolroot();
    CreateRoot();
    CreateBuildScript();
    printf("New Rhino ISM successfully created.\n");
    fflush(stdout);
    exit(0);
}
