//
// httpServer.c++
//
//	http protocol plugin for sysadmd.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

#ident "$Revision: 1.3 $"

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <syslog.h>
#include <fcntl.h>

#include <sysadm/ByteStreamConnection.h>
#include <sysadm/AppContext.h>
#include <sysadm/io.h>

USING_NAMESPACE(sysadm);

class HttpConnection : public ByteStreamConnection {
  public:
    HttpConnection(int input, int output, const String& fifo,
		   long long key);
    ~HttpConnection();

  protected:

    // Send data to our peer.
    virtual void sendData(const void* data, size_t nbytes);

    // Receive data from our peer.
    virtual ssize_t receiveData(void* data, size_t nbytes);

    // distributePacket() finds the PacketListener for this type of
    // Packet and calls its receivePacket() method.
    virtual void distributePacket(const Packet& packet);

  private:

    // Called when some input is available.
    static void monitorCallback(void* clientData, int id, int fd);

    int _input;
    int _output;
    int _monitor;
    long long _key;
    String _fifo;
};

HttpConnection::HttpConnection(int input, int output, const String& fifo,
			       long long key)
: _input(input), _output(output), _fifo(fifo), _key(key)
{
    AppContext& app(AppContext::getAppContext());
    _monitor = app.registerMonitor(input, monitorCallback, this);
}

HttpConnection::~HttpConnection()
{
    AppContext& app(AppContext::getAppContext());
    app.unregisterMonitor(_monitor);
    ::close(_input);
    ::close(_output);
    ::unlink(_fifo);
}

//
//  void HttpConnection::sendData(const void* data, size_t nbytes)
//
//  Description:
//      Send data to our peer.
//
//  Parameters:
//      data    The data to send.
//      nbytes  The number of bytes to send.
//
void HttpConnection::sendData(const void* data, size_t nbytes)
{
    if (SaWriteFully(_output, data, nbytes) == -1) {
	perror("SaWriteFully");
	exit(1);
    }
}

//
//  ssize_t HttpConnection::receiveData(void* data, size_t nbytes)
//
//  Description:
//      Receive data from our peer.
//
//  Parameters:
//      data    Buffer to read data into.
//      nbytes  Maximum number of bytes to read.
//
//  Returns:
//	The actual number of bytes read.
//
ssize_t HttpConnection::receiveData(void* data, size_t nbytes)
{
    return SaRead(_input, data, nbytes);
}


    // distributePacket() finds the PacketListener for this type of
    // Packet and calls its receivePacket() method.
void HttpConnection::distributePacket(const Packet& packet)
{
    if (packet.getAttr("key").longValue() != _key) {
	syslog(LOG_WARNING,
	       "Someone is trying to send an unauthorized packet to sysadmd.");
	exit(1);
    }
    ByteStreamConnection::distributePacket(packet);
}

//
//  void HttpConnection::monitorCallback(void* clientData, int, int)
//
//  Description:
//      Called by AppContext when some input is available on our
//      fifo.
//
//  Parameters:
//      clientData   Connection* (this is a static method)
//
void HttpConnection::monitorCallback(void* clientData, int, int)
{
    ((HttpConnection*)clientData)->handleInput();
}

extern "C" Connection* CreateClientConnection(const char*)
{
    char fifoName[100];
    strcpy(fifoName, "/tmp/sysadmdXXXXXX");
    if (mktemp(fifoName) == NULL) {
	return NULL;
    }
    if (mkfifo(fifoName, 0600) == -1
	|| chmod(fifoName, 0600) == -1) {
	return NULL;
    }
    int fd = open(fifoName, O_RDWR);
    if (fd == -1) {
	return NULL;
    }
    struct timeval tv;
    gettimeofday(&tv, NULL);
    srandom(tv.tv_usec + tv.tv_sec);
    long long key = ((long long)random() << 32) | random();
    char buf[150];
    snprintf(buf, sizeof buf, "%s %lld\n", fifoName, key);
    int len = strlen(buf);
    if (SaWriteFully(1, buf, len) != len) {
	return NULL;
    }
    return new HttpConnection(fd, 1, fifoName, key);
}
