//
// tcpmuxServer.c++
//
//	tcpmux protocol plugin for sysadmd.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

#ident "$Revision: 1.12 $"

#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <string.h>
#include <sysadm/io.h>
#include <sysadm/Log.h>
#include <sysadm/AppContext.h>
#include <sysadm/SocketConnection.h>

USING_NAMESPACE(sysadm);

// Can't use in_addr_t because we need to compile with 6.2 $ROOT.
typedef unsigned int InetAddr;

#define TCPMUX "tcpmux"

//
//  static bool matches(InetAddr addr, const char* elem)
//
//  Description:
//      Check to see if addr maches "elem", where "elem" could be the
//      keyword "all", a domain name, a host name, an IP Address, or a
//      network.
//
//  Parameters:
//      addr  client address to test.
//      elem  element to match against.
//
//  Returns:
//	true if addr matches, false otherwise.
//
static bool matches(InetAddr addr, const char* elem)
{
    // Keyword all matches all hosts.
    if (strcmp(elem, "all") == 0) {
	Log::trace(TCPMUX, "matches \"all\"");
	return true;
    }

    // Match domain.
    if (elem[0] == '.') {
	hostent* clientEnt = gethostbyaddr((const char *)(&addr), sizeof addr,
					   AF_INET);
	if (clientEnt != NULL) {
	    const char* pc = strchr(clientEnt->h_name, '.');
	    if (pc != NULL && strcmp(pc, elem) == 0) {
		Log::trace(TCPMUX, "matches domain: %s", elem);
		return true;
	    }
	}
	// If elem starts with '.' and it's not a suffix of the
	// hostname, there's no need to continue.
	return false;
    }

    // Match host name.
    hostent* ent = gethostbyname(elem);
    if (ent != NULL && ent->h_length == sizeof addr) {
	for (char** paddr = ent->h_addr_list; *paddr != NULL; paddr++) {
	    if (memcmp(*paddr, &addr, sizeof addr) == 0) {
		Log::trace(TCPMUX, "matches host: %s", elem);
		return true;
	    }
	}
	return false;
    }    

    // Match IP Address.
    in_addr elemAddr;
    if (inet_aton(elem, &elemAddr)
	&& memcmp(&elemAddr, &addr, sizeof addr) == 0) {
	Log::trace(TCPMUX, "matches IP Address: %s", elem);
	return true;
    }

    // Match network.
    char elemBuf[strlen(elem) + 1];
    strcpy(elemBuf, elem);
    char* slash = strchr(elemBuf, '/');
    unsigned long clientNet;
    if (slash != NULL) {
	*slash = '\0';
	in_addr maskAddr;
	if (!inet_aton(slash + 1, &maskAddr)) {
	    return false;
	}
	clientNet = ntohl(addr) & ntohl(maskAddr.s_addr);
    } else {
	in_addr clientAddr;
	clientAddr.s_addr = addr;
	clientNet = inet_netof(clientAddr);
	if (clientNet == INADDR_NONE) {
	    return false;
	}
    }
    unsigned long elemNet = inet_network(elemBuf);
    if (elemNet == INADDR_NONE) {
	Log::trace(TCPMUX, "Can't turn %s into a network", elemBuf);
	return false;
    }
    if (elemNet == clientNet) {
	Log::trace(TCPMUX, "matches network: %s", elem);
	return true;
    }

    return false;
}

//
//  static bool isInList(const String& list, InetAddr addr)
//
//  Description:
//      Check to see if "addr" matches one of the elements of "list".
//
//      I'd really like this method to be static.  The C++ compiler
//      crashes if it's static.
//
//  Parameters:
//      list  List of elements to match.
//      addr  Address to match.
//
//  Returns:
//	true if "addr" matches an element in "list", false otherwise.
//
bool isInList(const String& list, InetAddr addr)
{
    String listCopy(list);

    char* lasts = NULL;
    char* buf = listCopy.writableCharPtr();
    for (char* elem = strtok_r(buf, ", ", &lasts);
	 elem != NULL; elem = strtok_r(NULL, ", ", &lasts)) {
	if (matches(addr, elem)) {
	    return true;
	}
    }
    return false;
}

//
//  static bool checkAccess(int sock)
//
//  Description:
//      Determine whether "sock" is a socket that we should talk to.
//      Access control is implemented by two parameters in
//      /var/sysadm/sysadm.conf:
//      
//        tcpmux.allowHosts - list of hosts that are allowed access.
//        tcpmux.denyHosts - list of hosts that are denied access.
//          
//      If "denyHosts" comes before "allowHosts" in sysadm.conf, then
//      hosts are allowed access by default, and denied if on the
//      "denyHosts" list and not on the "allowHosts" lists.  If
//      "allowHosts" comes first, then hosts are denied by default,
//      and allowed if on the "allowHosts" list and not on the
//      "denyHosts" list.
//          
//      allowHosts and denyHosts are comma and/or space-delimited
//      lists of hostnames, IP addresses in either a.b.c.d or hex
//      notation, domain names (any string that starts with a "."),
//      and networks.  A network can be specified as either a partial
//      IP address, or a full IP address, a slash, and another full IP
//      address.  In the last case, the IP Address before the slash
//      is the network and the IP Address after the slash is the
//      netmask.  All hosts can be matched using the keyword "all".
//
//  Parameters:
//      sock  Socket to check for access
//
//  Returns:
//	true if client should be allowed access, false otherwise.
//
static bool checkAccess(int sock)
{
    sockaddr_in addr;
    socklen_t nameLen = sizeof addr;
    if (getpeername(sock, (struct sockaddr*)&addr, &nameLen) == -1
	|| addr.sin_family != AF_INET) {
	return false;
    }
    AppContext& app(AppContext::getAppContext());
    String allowList(app.getProperty("tcpmux.allowHosts"));
    String denyList(app.getProperty("tcpmux.denyHosts"));
    int allowNum(app.getPropertyNum("tcpmux.allowHosts"));
    int denyNum(app.getPropertyNum("tcpmux.denyHosts"));
    bool hostOK;
    if (denyNum != -1 && denyNum < allowNum) {
	hostOK = true;
	if (denyList != String::NUL
	    && isInList(denyList, addr.sin_addr.s_addr)) {
	    hostOK = false;
	}
	if (!hostOK
	    && allowList != String::NUL
	    && isInList(allowList, addr.sin_addr.s_addr)) {
	    hostOK = true;
	}
    } else {
	hostOK = false;
	if (allowList == String::NUL
	    || isInList(allowList, addr.sin_addr.s_addr)) {
	    hostOK = true;
	}
	if (denyList != String::NUL
	    && isInList(denyList, addr.sin_addr.s_addr)) {
	    hostOK = false;
	}
    }
    return hostOK;
}

//
//  extern "C" Connection* CreateClientConnection(const char*)
//
//  Description:
//      Create a SocketConnection.
//
//  Returns:
//	A new SocketConnection.
//
extern "C" Connection* CreateClientConnection(const char*)
{
    // If we leave sock as 1, syslog screws up our protocol.
    int sock = dup(1);
    close(1);
    open("/dev/console", O_WRONLY);

    if (!checkAccess(sock)) {
	AppContext& app(AppContext::getAppContext());
	String reason = String("-") +
	    app.getProperty("tcpmux.denyMessage") + String("\r\n");
	SaWriteFully(sock, (const char*)reason, reason.getLength());
	return NULL;
    }

    // This completes the tcpmux handshake
    char *ackstr = "+\r\n";
    if (SaWriteFully(sock, ackstr, strlen(ackstr)) < strlen(ackstr)) {
	return NULL;
    }

    return  new SocketConnection(sock);
}
