//
// UnixServerAuth.c++
//
//	Server side of unix authentication.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

#ident "$Revision: 1.11 $"

#include <sys/stat.h>
#include <sys/wait.h>

#define _BSD_COMPAT
#include <fcntl.h>
#include <sys/file.h>
#undef _BSD_COMPAT

#include <pwd.h>
#include <crypt.h>

#include <unistd.h>
#include <syslog.h>
#include <sysadm/sysadm.h>
#ifdef HAVE_IA_H
#include <ia.h>
#endif  //  HAVE_IA_H
#include <shadow.h>
#ifdef HAVE_DEFLT_H
#include <deflt.h>
#include <sat.h>  //  for ia_audit
#endif  //  HAVE_DEFLT_H
#include <lastlog.h>
#include <time.h>
#include <dlfcn.h>
#include <unistd.h>
#include <signal.h>

#include <sysadm/i18n.h>
#include <sysadm/Log.h>

#include "UnixServerAuth.h"

#define PASSWORD "password"		// Attribute name for password.
#define AUTH_UNIX "authScheme/unix" 	// Name of our service.
#define RESULT "result"			// Attribute name of result.
#define MAX_ATTEMPTS 3			// Number of failures before
					// we shut down.
#define LOGIN_DEFAULT "login"		// basename("/etc/default/login")

static void count_badlogins(bool outcome, const char *username);

//
// Constructor.
//
UnixServerAuth::UnixServerAuth()
: _listener(NULL),
  _attempts(0),
  _dayNow(time((time_t *)NULL) / (24L * 60 * 60))  /* (seconds since ...
                                                      / (seconds per day)) */
{
}

//
// Destructor.
//
UnixServerAuth::~UnixServerAuth()
{
    getConnection()->orphanPacketListener(AUTH_UNIX);
    delete _listener;
}

//
//  void UnixServerAuth::authenticate(ResultListener* adoptedListener)
//
//  Description:
//      Start the authentication process.  For us this simply means
//      installing a listener for AUTH_UNIX packets.
//
//  Parameters:
//      adoptedListener  Used to return results.
//
void UnixServerAuth::authenticate(ResultListener* adoptedListener)
{
    delete _listener;
    _listener = adoptedListener;
    getConnection()->adoptPacketListener(AUTH_UNIX, this);
}

//
//  void UnixServerAuth::receivePacket(const Packet& packet)
//
//  Description:
//      Get a packet bound for the AUTH_UNIX service.  We
//      expect the packet to specify the login name and password.  If
//      login name/password are a valid combination according to the
//      passwd database, we setuid to the user id specified and return
//      success.
//
//  Parameters:
//      packet  Packet specifying login parameters.
//
void UnixServerAuth::receivePacket(const Packet& packet)
{
    String loginName(packet.getSelector());
    String password(packet.getAttr(PASSWORD).stringValue());

    Log::debug(AUTH_UNIX, "Login request for user: %s",
	       (const char*)loginName);

    Packet reply(packet.getType(), packet.getSelector());
    bool result = false;
    passwd pwbuf;
    char buf[BUFSIZ];
    passwd *pw;
    getpwnam_r(loginName, &pwbuf, buf, sizeof(buf), &pw);
    if (pw != NULL) {
	result = checkPassword(pw, password);
	count_badlogins(result, loginName);
	if (result) {
	    result = setgid(pw->pw_gid) != -1 && setuid(pw->pw_uid) != -1;
	}
    }
    
    if (result) {
	// Set current directory to home directory so we can create a
	// core even if we're not running as root (which will be the
	// case if the user logs in as a user other than root).
	chdir(pw->pw_dir);

	Log::info(AUTH_UNIX, i18n("sysadmd login request for %s succeeded."),
		   (const char*)loginName);
	syslog(LOG_INFO, i18n("sysadmd login request for %s succeeded."),
	       (const char*)loginName);
	_listener->succeeded(ResultEvent());
    } else {
	Log::error(AUTH_UNIX, i18n("sysadmd login request for %s failed."),
		   (const char*)loginName);
	syslog(LOG_ERR, i18n("sysadmd login request for %s failed."),
	       (const char*)loginName);

	// Sleep for awhile to discourage the use of sysadmd as a
	// method for blindly trying login accounts.
	sleep(1);

	if (++_attempts == MAX_ATTEMPTS) {
	    _listener->failed(ResultEvent());
	}
    }

    reply.setAttr(Attribute(RESULT, result));
    getConnection()->sendPacket(reply);
}

//
//  bool UnixServerAuth::passwordRequired()
//
//  Description:
//      Determine whether passwords are required on this system.
//
//  Returns:
//	true if passwords are required, false otherwise.
//
bool UnixServerAuth::passwordRequired()
{
    bool result = false;

#if HAVE_DEFLT_H
    FILE* fp = defopen(LOGIN_DEFAULT);
    if (fp == NULL) {
	return false;
    }
    const char* mandPass = defread(fp, "MANDPASS");
    if (mandPass != NULL && strncasecmp(mandPass, "YES", 3) == 0) {
	result = true;
    }
    if (!result) {
	const char* passReq = defread(fp, "PASSREQ");
	if (passReq != NULL && strncasecmp(passReq, "YES", 3) == 0) {
	    result = true;
	}
    }
    defclose(fp);
#endif // HAVE_DEFLT_H

    return result;
}

//
//  bool UnixServerAuth::dceVerify(passwd* pw, const char* password)
//
//  Description:
//      Perform DCE password verification.
//
//  Parameters:
//      pw        passwd entry to verify.
//      password  Password that the user typed.
//
//  Returns:
//	true if OK to login according to DCE, false otherwise.
//
bool UnixServerAuth::dceVerify(passwd* pw, const char* password)
{
    static void* handle = NULL;
    if (handle == NULL) {
	// Do not dlclose this handle; otherwise sysadmd gets killed,
	// probably due to a signal handler in libdce.
	handle = dlopen("libdce.so", RTLD_LAZY);
    }
    bool result = false;
    if (handle != NULL) {
	typedef int (*VERIFY_FUNC)(char* user, uid_t uid, gid_t gid,
				   char* passwd, char** msg);
	VERIFY_FUNC verify = (VERIFY_FUNC)dlsym(handle, "dce_verify");
	char* dceError = NULL;
	if (verify != NULL) {
	    String passwordCopy(password);
	    result = verify(pw->pw_name, pw->pw_uid, pw->pw_gid,
			    passwordCopy.writableCharPtr(), &dceError) == 0;
	    if (!result && dceError != NULL) {
		Log::debug(AUTH_UNIX, "DCE verify failed: %s",
			   dceError);
	    }
	}
    }
    return result;
}

#ifdef HAVE_IA_H

//
//  bool UnixServerAuth::isInactive(const passwd* pw, const uinfo_t uinfo)
//
//  Description:
//      Determine if an account has been inactive for too long.
//
//  Parameters:
//      pw     passwd entry of account to check for inactivity.
//      uinfo  User info.
//
//  Returns:
//	true if account has been inactive too long, false otherwise.
//
bool UnixServerAuth::isInactive(const passwd* pw, const uinfo_t uinfo)
{
    long inactive = 0;
    ia_get_loginact(uinfo, &inactive);
    if (inactive <= 0) {
	return false;
    }

    String lastLogFile("/var/adm/lastlog/");
    lastLogFile += pw->pw_name;
    int fd = open(lastLogFile, O_RDONLY);
    if (fd == -1) {
	return true;
    }
    struct stat st;
    if (fstat(fd, &st) != 0 || st.st_size != sizeof(struct lastlog)) {
	close(fd);
	return true;
    }
    struct lastlog last;
    if (read(fd, &last, sizeof last) != sizeof last) {
	close(fd);
	return true;
    }
    close(fd);

    if (last.ll_time == 0) {
	return false;
    }

    return last.ll_time / DAY + inactive < _dayNow;
}

//
//  bool UnixServerAuth::isAged(const uinfo_t uinfo)
//
//  Description:
//      Determine whether the password for an account has gotten too
//      old.
//
//  Parameters:
//      uinfo  User info.
//
//  Returns:
//	true if password is too old, false otherwise.
//
bool UnixServerAuth::isAged(const uinfo_t uinfo)
{
    long expire = 0;
    ia_get_logexpire(uinfo, &expire);
    if (expire > 0 && expire < _dayNow) {
	return true;
    }
    long lastChange = 0;
    ia_get_logchg(uinfo, &lastChange);
    long min = 0;
    ia_get_logmin(uinfo, &min);
    long max = 0;
    ia_get_logmax(uinfo, &max);
    if (lastChange == 0 || lastChange > _dayNow
	|| max >= 0 && _dayNow > lastChange + max && max > min) {
	return true;
    }
    return false;
}

#endif  //  HAVE_IA_H

//
//  bool UnixServerAuth::checkPassword(passwd* pw, const char* password)
//
//  Description:
//      Verify that it's OK for user whose passwd entry is "pw" to
//      login after providing "password".
//
//  Parameters:
//      pw        passwd entry of user that wants to log in.
//      password  Password that the user typed.
//
//  Returns:
//	true if OK to login, false otherwise.
//
bool UnixServerAuth::checkPassword(passwd* pw, const char* password)
{
    // Check for no password when passwords are required.
    if (strlen(password) == 0 && passwordRequired()) {
	Log::trace(AUTH_UNIX,
		   "No password provided and passwords are required.");
	return false;
    }

    // Can't log in to restricted account.
    if (strcmp(pw->pw_passwd, "*") == 0) {
	Log::trace(AUTH_UNIX, "Attempt to login to restricted account.");
	return false;
    }

    // DCE verification.
    if (strcmp(pw->pw_passwd, "-DCE-") == 0) {
	Log::trace(AUTH_UNIX, "Using DCE account verification.");
	return dceVerify(pw, password);
    }		

#ifdef HAVE__GETPWENT_NO_SHADOW
    // Check password.
    if (strcmp(crypt(password, pw->pw_passwd), pw->pw_passwd) != 0) {
	Log::trace(AUTH_UNIX, "Wrong password.");
	return false;
    }
#else
    // We have to check for shadowed passwords ourselves.  Is this one?
    if (strcmp(pw->pw_passwd, "x") == 0) {
        struct spwd *spw = getspnam(pw->pw_name);
        if (spw == NULL) {
            //  Not good.  Not actually shadowed?
            Log::debug(AUTH_UNIX, "getspnam() failed");
            return false;
        }
        if (strcmp(crypt(password, spw->sp_pwdp), spw->sp_pwdp) != 0) {
            Log::trace(AUTH_UNIX, "Wrong password.");
	    return false;
        }
    } else {
        // Nope, it's a regular old password.  Check it.
        if (strcmp(crypt(password, pw->pw_passwd), pw->pw_passwd) != 0) {
	    Log::trace(AUTH_UNIX, "Wrong password.");
	    return false;
        }
    }
#endif  //  HAVE__GETPWENT_NO_SHADOW

    bool result = true;

#ifdef HAVE_IA_H
    // Check inactivity and password aging.
    uinfo_t uinfo = NULL;
    if (ia_openinfo(pw->pw_name, &uinfo) || (uinfo == NULL)) {
	Log::trace(AUTH_UNIX, "ia_openinfo failed.");
	return false;
    }
    if (isInactive(pw, uinfo)) {
	Log::trace(AUTH_UNIX, "Account has been inactive too long.");
	result = false;
    }
    if (result && isAged(uinfo)) {
	Log::trace(AUTH_UNIX, "Password has aged too much.");
	result = false;
    }
	
    ia_closeinfo(uinfo);
#endif  //  HAVE_IA_H

    return result;
}

//
// This method is called by sysadmd after dlopening our dso.
//
extern "C" Authenticator* CreateAuthenticator()
{
    return new UnixServerAuth();
}


//
// The rest of the code in this file was adapted from
// isms/eoe/cmd/login/login.c
//
#define BL_WRITE		1
#define BL_LOCK			2
#define BL_BADFILE		3
#define BL_BADLOCK		4
#define BL_LOCK_BADFILE		5

#define	LNAME_SIZE	  32	/* size of logname */

#define NAME_DELIM         32   /* LOCKOUTEXEMPT name delimiter */
#define BADLOGDIR 	"/var/adm/badlogin"  /* used by update_count() */

#define MATCH 1
#define NOMATCH 0
#define SKIP_DELIM(a)            {while (*a && (*a == NAME_DELIM)) a++;}

static int
notlockout(const char *user)
{
#if HAVE_DEFLT_H
    char notlockname[LNAME_SIZE+1];
    const char * name_begin_ptr;
    const char * name_delim_ptr;
    int    namelen=0;
    String exempt;

    FILE* fp = defopen(LOGIN_DEFAULT);
    if (fp != NULL) {
	char* pc = defread(fp, "LOCKOUTEXEMPT");
	if (pc != NULL) {
	    exempt = pc;
	}
	defclose(fp);
    }

    if (exempt == NULL)
	return (NOMATCH);

    notlockname[0]='\0';
    name_begin_ptr = exempt;
    SKIP_DELIM(name_begin_ptr);

    /* Iterate through the list of names until we reach \0 */
    while (*name_begin_ptr != NULL) {
	name_delim_ptr = strchr(name_begin_ptr,NAME_DELIM);
	  
	/* strchr didn't find any more delimiters, but we still
	 * have to deal with the last name in the list
	 */
	if (name_delim_ptr != NULL)
	    namelen = name_delim_ptr - name_begin_ptr;
	else
	    namelen = strlen(name_begin_ptr);

	/* name must be a valid length <= LNAME_SIZE,
	 * we don't want to overflow on the stack, and we
	 * don't want to just chop off the name at LNAME_SIZE 
	 * since that would invalidate the identity.
	 */
	if ((namelen > 0) && (namelen <=LNAME_SIZE)){
	    strncpy(notlockname, name_begin_ptr, namelen);
	    notlockname[namelen]='\0';
	    if (strcmp(user, notlockname) == 0)
		return (MATCH);
	}
	/* Increment the name_begin_ptr, and handle  
	 * termination of the loop is there isn't
	 * a match in the list.
	 */
	if (name_delim_ptr != NULL) {
	    name_begin_ptr=name_delim_ptr;
	    SKIP_DELIM(name_begin_ptr);
	}
	else
	    name_begin_ptr = name_begin_ptr + namelen; 
    }
#endif  // HAVE_DEFLT_H
    return (NOMATCH);
}



/*
 * On successful login (outcome=true), reset count to zero.
 * Otherwise, increment count, test whether count has reached
 * Lockout value, and lock user's account if it has.
 */
static int
update_count(bool outcome, const char *user, long lockout)
{
    char *badlogfile;
    int fd, retval, wstatus;
    bool isNew = false;
    short lockok = 0;
    char count;
    struct passwd *pw;
    struct stat  bl_buf;

    /* Validate that this user exists, we don't use uinfo (iaf)
     * structure here, because we're not guaranteed that it exists
     * when this routine is called. BUG #491422
     */
    pw = getpwnam(user);
	
    /* Don't process LOCKOUT if the user doesn't exist.
     * BUG #491422
     */
    if (pw != NULL) {

	/* 
	 * Open user's badlog file.
	 */
	if (access(BADLOGDIR, 0) < 0) {
	    if (mkdir(BADLOGDIR, 0700) < 0)
		return(BL_BADFILE);
	}
	if ((badlogfile = (char *)malloc(sizeof(BADLOGDIR) + 
					 strlen(user) + 3)) == NULL) {
	    return(BL_BADFILE);
	}
	sprintf(badlogfile, "%s/%s", BADLOGDIR, user);
	/*
	 * If file is new, count is zero.
	 * Use stat instead of access to avoid the real uid/gid
	 * problems when login not started as root.  BUG #506487
	 */
	if (stat(badlogfile, &bl_buf) < 0) {
	    isNew = true;
	    count = '\0';
	}
	fd =open(badlogfile, O_RDWR | O_CREAT, 0600);
	free(badlogfile);
	if (fd < 0 || flock(fd, LOCK_EX) < 0) {
	    close(fd);
	    return(BL_BADFILE);
	}
	if (!outcome) { 	
	    /* 
	     * Failed login: read count and seek back to start.
	     */
	    if (!isNew && read(fd, &count, 1) != 1) {
		close(fd);
		return(BL_BADFILE);
	    }
	    if (lseek(fd, 0, SEEK_SET) < 0) {
		close(fd);
		return(BL_BADFILE);
	    }
	    /*
	     * If updated count has reached Lockout value,
	     * invoke passwd -l and reset count to zero - UNLESS
	     * (UNLESS!) the user is on the LOCKOUTEXEMPT list in
	     * the options file.  Such a user will have his count
	     * updated when a failed login attempt is made, but
	     * will not be locked out when the Lockout count is
	     * exceeded. This feature is the result of Bug 491422,
	     * to address the denial of service attacks possible
	     * when the LOCKOUT option is in use.
	     */
	    if ((++count >= lockout) &&
		(notlockout(user) == NOMATCH)) {  
		retval = fork();
		switch (retval) {
		case -1:
		    close(fd);
		    return (BL_BADLOCK);
		case 0:
		    close(fd);
		    signal(SIGALRM, SIG_DFL);
		    signal(SIGHUP, SIG_DFL);
		    execl("/bin/passwd", "passwd",
			  "-l", user, (char *)0);
		    exit(1);
		}
		if (waitpid(retval, &wstatus, 0) < 0 ||
		    wstatus != 0) {
		    close(fd);
		    return (BL_BADLOCK);
		}
		count = '\0';
		lockok = 1;
	    }
	}
	else { 		
	    /* 
	     * On successful login, reset count to zero. 
	     */
	    count = '\0';
	}
	/*
	 * Write back updated count.
	 */
	if (write(fd, &count, 1) != 1) {
	    close(fd);
	    if (lockok)
		return(BL_LOCK_BADFILE);
	    return(BL_BADFILE);
	}
	else {
	    close(fd);
	    if (lockok)
		return(BL_LOCK);
	    return(BL_WRITE);
	}
    }
    return(BL_LOCK);
}

/*
 *  Call update_count and log result to syslog and system audit trail.
 */
static void
count_badlogins(bool outcome, const char *username)
{
#ifdef HAVE_DEFLT_H
#ifdef HAVE_IA_H
    long lockout = 0;

    FILE* fp = defopen(LOGIN_DEFAULT);
    if (fp != NULL) {
	char* pc = defread(fp, "LOCKOUT");
	if (pc != NULL) {
	    lockout = atoi(pc);
	}
	defclose(fp);
    }

    if (lockout <= 0) {
	return;
    }

    switch(update_count(outcome, username, lockout)) {
    case BL_LOCK:
	syslog(LOG_NOTICE|LOG_AUTH, "Locked %s account\n", username);
	ia_audit("LOGIN", (char*)username, 1, "Locked account");
	break;
    case BL_BADFILE:
	ia_audit("LOGIN", (char*)username, 0, "Can't update badlogin file");
	syslog(LOG_ALERT|LOG_AUTH, "Can't update %s badlogin count", 
	       username);
	break;
    case BL_BADLOCK:
	ia_audit("LOGIN", (char*)username, 0, "Can't lock account");
	syslog(LOG_ALERT|LOG_AUTH, "Can't  lock %s account", username);
	break;
    case BL_LOCK_BADFILE:
	ia_audit("LOGIN", (char*)username, 0, 
		 "Locked account but can't update badlogin file");
	syslog(LOG_ALERT|LOG_AUTH, 
	       "Can't update %s badlogin count", username);
	break;
    default:
	break;
    }
#endif  // HAVE_IA_H
#endif  // HAVE_DEFLT_H
}
