//
// CategoryListenerProxy.c++
//
//	Proxy Class for monitoring changes in a Category.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

#ident "$Revision: 1.14 $"

#include "CategoryListenerProxy.h"

BEGIN_NAMESPACE(sysadm);

#define ITEM "item"
#define ITEMADDED "itemAdded"
#define ITEMCHANGED "itemChanged"
#define NEWITEM "newItem"
#define ITEMREMOVED "itemRemoved"
#define SELECTOR "selector"
#define BEGINBLOCKCHANGES "beginBlockChanges"
#define ENDBLOCKCHANGES "endBlockChanges"
#define ENDEXISTS "endExists"
#define ATTRCHANGED "attrChanged"
#define KEY "key"
#define TYPE "type"
#define VALUE "value"
#define ERROR "error"
#define ERRORSTRING "errorString"

//
//  CategoryListenerProxy Implementation
//
class CategoryListenerProxyImpl {
  public:
    CategoryListenerProxyImpl(Connection& conn,
			      const String& serviceName)
    : _conn(conn), _serviceName(serviceName) { }
    Connection& _conn;
    String _serviceName;
};

//
// Constructor
//
CategoryListenerProxy::CategoryListenerProxy(Connection& conn,
					     const String& serviceName) :
    _impl(new CategoryListenerProxyImpl(conn, serviceName)) {
}

//
// Destructor 
//
CategoryListenerProxy::~CategoryListenerProxy() {
    delete _impl;
}

// 
// CategoryListenerProxy overrides the base class methods to send the 
// notification to the CategoryProxy named by serviceName
//
//
void CategoryListenerProxy::itemAdded(const Item& item) {
    Packet packet(_impl->_serviceName, ITEMADDED);
    packet.setAttr(Attribute(ITEM, item));

    // Send it over the wire
    _impl->_conn.sendPacket(packet);
}

void CategoryListenerProxy::itemChanged(const Item&, 
					const Item& newItem) {
    Packet packet(_impl->_serviceName, ITEMCHANGED);
    packet.setAttr(Attribute(NEWITEM, newItem));

    // Send it over the wire
    _impl->_conn.sendPacket(packet);
}

void CategoryListenerProxy::itemRemoved(const Item& item) {
    Packet packet(_impl->_serviceName, ITEMREMOVED);
    packet.setAttr(Attribute(SELECTOR, item.getSelector()));
    
    // Send it over the wire
    _impl->_conn.sendPacket(packet);
}

void CategoryListenerProxy::beginBlockChanges() {
    _impl->_conn.sendPacket(Packet(_impl->_serviceName, BEGINBLOCKCHANGES));
}

void CategoryListenerProxy::endBlockChanges() {
    _impl->_conn.sendPacket(Packet(_impl->_serviceName, ENDBLOCKCHANGES));
}

void CategoryListenerProxy::endExists() {
    _impl->_conn.sendPacket(Packet(_impl->_serviceName, ENDEXISTS));
}

void CategoryListenerProxy::attrChanged(const AttrEvent& event) {
    Packet packet(_impl->_serviceName, ATTRCHANGED);
 
    const Attribute attr = event.getChangedAttr();

    packet.setAttr(Attribute(KEY, attr.getKey()));
    packet.setAttr(Attribute(TYPE, attr.getTypeString()));
    packet.setAttr(Attribute(VALUE, attr.getValueString()));

    // Send it over the wire
    _impl->_conn.sendPacket(packet);
}

void CategoryListenerProxy::categoryError(CategoryErrorEvent& event) {
     Packet packet(_impl->_serviceName, ERROR);

     packet.setAttr(Attribute(ERRORSTRING, event.getErrorString()));

     // Send it over the wire
    _impl->_conn.sendPacket(packet);
}

END_NAMESPACE(sysadm);
