//
// category.c++
//
//	Category service.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

#ident "$Revision: 1.10 $"

#include <sysadm/Log.h>
#include <sysadm/PacketListener.h>
#include <sysadm/DictionaryOf.h>
#include <sysadm/i18n.h>
#include <sysadm/format.h>
#include <sysadm/CategoryFactory.h>

#include "CategoryPacketListener.h"

#define CATEGORY_SERVICE "category"

USING_NAMESPACE(sysadm);

struct CategoryElem {
    CategoryElem(const String& serviceName, CategoryPacketListener* listener)
    : _serviceName(serviceName), _listener(listener) {
    }

    String _serviceName;
    CategoryPacketListener* _listener;
};

class CategoryServiceListener : public PacketListener {
  public:
    CategoryServiceListener(Connection& conn)
    : _conn(conn) {
    }
    
    void loadCategory(const Packet& packet);
    void unloadCategory(const Packet& packet);
    void error(const Packet& packet, const String& reason);
    void success(const Packet& packet);

    virtual void receivePacket(const Packet& packet);

    Connection& _conn;
    DictionaryOf<CategoryElem> _categories;
};

void CategoryServiceListener::receivePacket(const Packet& packet) {
    if (packet.getSelector() == "load") {
	loadCategory(packet);
    } else if (packet.getSelector() == "unload") {
	unloadCategory(packet);
    }
}

void CategoryServiceListener::loadCategory(const Packet& packet) {
    Log::debug(CATEGORY_SERVICE, "loadCategory");
    String catName(packet.getAttr("name").stringValue());
    
    String errorString;
    Category* cat = CategoryFactory::getCategory(catName, errorString);
    if (cat == NULL) {
	error(packet, errorString);
	return;
    }

    String serviceName(String("category/") + catName);
    CategoryPacketListener* packetListener =
	new CategoryPacketListener(_conn, serviceName, *cat);
    _conn.adoptPacketListener(serviceName, packetListener);
    _categories.add(new CategoryElem(serviceName, packetListener),
		    new String(catName));
     success(packet);
}

void CategoryServiceListener::unloadCategory(const Packet& packet) {
    Log::debug(CATEGORY_SERVICE, "unloadCategory");
    String catName(packet.getAttr("name").stringValue());
    CategoryElem* elem
	= _categories.remove(catName);
    if (elem != NULL) {
	Category* cat = elem->_listener->getCategory();
	delete _conn.orphanPacketListener(elem->_serviceName);
	delete elem;

	// Release this last as CategoryPacketListener destructor
	// might interact with Category.
	String errorString;
	if (!CategoryFactory::releaseCategory(cat, errorString)) {
	    error(packet, errorString);
	    return;
	}
	
    } else {
	Log::warning(CATEGORY_SERVICE, 
		     i18n("unload request for category %s that was not loaded"),
		     (const char*) catName);
    }
    success(packet);
}

//
//  void CategoryServiceListener::error(const Packet& packet,
//                                      const String& reason)
//
//  Description:
//      Return an error for the operation requested by "packet". 
//
//  Parameters:
//      packet  The request that failed.
//      reason  The reason for the failure.
//
void CategoryServiceListener::error(const Packet& packet,
				    const String& reason) {
    Packet reply(packet.getType(), packet.getSelector());
    reply.setAttr(packet.getAttr("cookie"));
    reply.setAttr(Attribute("result", false));
    reply.setAttr(Attribute("reason", reason));
    _conn.sendPacket(reply);
}

//
//  void CategoryServiceListener::success(const Packet& packet)
//
//  Description:
//      Return a successful result for the operation requested by
//      "packet".
//
//  Parameters:
//      packet  The request that succeeded.
//
void CategoryServiceListener::success(const Packet& packet) {
    Packet reply(packet.getType(), packet.getSelector());
    reply.setAttr(packet.getAttr("cookie"));
    reply.setAttr(Attribute("result", true));
    _conn.sendPacket(reply);
}


extern "C" void* CreatePacketListener(Connection& conn, 
				      const String&) {
    return new CategoryServiceListener(conn);
}
