//
// LogService.c++
//
//	Service that copies log messages to the client.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

#ident "$Revision: 1.6 $"

#include <stdio.h>
#include <string.h>

#include <sysadm/PacketListener.h>
#include <sysadm/LogListener.h>
#include <sysadm/Connection.h>
#include <sysadm/Packet.h>
#include <sysadm/i18n.h>
#include <sysadm/LogFilter.h>
#include <sysadm/format.h>

USING_NAMESPACE(sysadm);

#define LOG_SERVICE "log"

//
// LogServiceListener sends log messages to the client.
//
class LogServiceListener : public LogListener {
  public:
    LogServiceListener(Connection& conn, const String& serviceName);
    virtual void handleMessage(const char* module, Log::Level level,
			       const char* format, va_list args);
  private:
    Connection& _conn;
    String _serviceName;
    bool _sending;
};

//
// LogServiceListener constructor.
//
LogServiceListener::LogServiceListener(Connection& conn,
				       const String& serviceName)
: _conn(conn), _serviceName(serviceName), _sending(false)
{
}

//
//  void LogServiceListener::handleMessage(const char* module,
//  				           Log::Level level,
//  				           const char* format, va_list args)
//
//  Description:
//      Log a message.  In our case this means creating a packet that
//      contains information about the message to be logged and
//      sending it to the client.
//
//  Parameters:
//      module    module that logged this message.
//      level     Log::TRACE .. Log::FATAL.
//      format    printf-style format.
//      args      args for "format".
//
void LogServiceListener::handleMessage(const char* module,
				       Log::Level level,
				       const char* format, va_list args)
{
    // Don't log if we're being called recursively.  If each packet
    // being sent is getting logged by the "conn" service
    // (lib/libsysadmComm/Connection.c++), we would go into an
    // infinite loop when we tried to send a log message to the
    // client.
    if (!_sending) {
	_sending = true;
	Packet packet(_serviceName, "messsage");
	packet.setAttr(Attribute("level", (long long)level));
	packet.setAttr(Attribute("stringLevel", Log::levelToString(level)));
	packet.setAttr(Attribute("module", module));
    
	char buf[Log::MAXLEN];
	SaStringFormatV(buf, sizeof buf, format, args);
	packet.setAttr(Attribute("message", buf));
	_conn.sendPacket(packet);
	_sending = false;
    }
}

//
// LogPacketListener receives packets from the client.  These packets
// are interpreted as specifications of LogFilters.
//
class LogPacketListener : public PacketListener {
  public:
    LogPacketListener(Connection& conn, const String& serviceName);
    virtual ~LogPacketListener();

    virtual void receivePacket(const Packet& packet);
    
    Connection& _conn;
    LogListener* _listener;
};

//
// LogPacketListener contructor.
//
LogPacketListener::LogPacketListener(Connection& conn,
				     const String& serviceName)
: _conn(conn)
{
    _listener = new LogServiceListener(conn, serviceName);
    Log::getLog().adoptLogListener(_listener);
}

//
// LogPacketListener destructor.
//
LogPacketListener::~LogPacketListener()
{
    delete Log::getLog().orphanLogListener(_listener);
}

//
//  void LogPacketListener::receivePacket(const Packet& packet)
//
//  Description:
//      Receive a packet from the client.  The client can clear our
//      list of LogFilters and add new filters to our list of
//      LogFilters.
//
//  Parameters:
//      packet  Packet from client.
//
void LogPacketListener::receivePacket(const Packet& packet)
{
    if (packet.getSelector() == "removeAllFilters") {
	Log::trace(LOG_SERVICE, "Request to remove all log filters.");
	_listener->removeAllFilters();
    } else if (packet.getSelector() == "addFilter") {
	Log::trace(LOG_SERVICE, "Request to add a log filter.");
	LogFilter* filter = new LogFilter;
	filter->levelOn(packet.getAttr("level").longValue());

	Attribute modulesAttr(packet.getAttr("modules"));
	if (modulesAttr != Attribute::NUL) {
	    filter->addModuleList(modulesAttr.stringValue());
	}

	_listener->adoptFilter(filter);
    } else {
	Log::error(LOG_SERVICE, i18n("Unrecognized request %s."),
		   (const char*)packet.getSelector());
    }
}

//
// Called by sysadmd to dynamically load the Log service.
//
extern "C" void* CreatePacketListener(Connection& conn,
				      const String& serviceName)
{
    return new LogPacketListener(conn, serviceName);
}
