//
// PrivListener.c++
//
//	Privilege Service Listener.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

#ident "$Revision: 1.13 $"

#include <sysadm/AppContext.h>
#include <sysadm/Command.h>
#include <sysadm/io.h>
#include <sysadm/Log.h>
#include <sysadm/i18n.h>
#include <sysadm/SaParam.h>
#include <sysadm/privdefs.h>
#include <sysadm/privilege.h>

#include <signal.h>
#include <unistd.h>
#include <pwd.h>
#include <string.h>

#include "PrivListener.h"

// Service name.
#define PRIVILEGE "privilege"

// Commands we accept.
#define RUN "run"
#define CHECK "check"
#define ADD "add"
#define INPUT "input"
#define DESTROY "destroy"
#define CHECK_PASSWORD "checkPassword"

// Asynchronous events we send back to the client.
#define FINISHED "finished"
#define OUTPUT "output"
#define ERROR "error"

// Attribute keys.
#define JOB_ID "jobId"
#define ARGC "argc"
#define ARG "arg"
#define ARGS "args"
#define COOKIE "cookie"
#define NUM_PRIVS "numPrivs"
#define PRIV "priv"
#define EXIT_VALUE "exitValue"
#define PASSWORD "password"
#define RESULT "result"
#define USER "user"
#define REASON "reason"

// Reasons why checkpriv can fail
#define NOT_PRIVILEGED (-1)
#define NO_SUCH_PRIV (-2)

#define SaPRIVENV CONFIG_SYSADM_VAR_DIR "/privenviron"

/*
 *  static char** CreateRunprivEnvironment()
 *
 *  Description:
 *      This code was stolen from isms/userenv/cmd/runpriv/runpriv.c.
 *
 *      Build an environment for the privileged program.  The
 *      environment is set up in the file SaPRIVENV, which consists
 *      of lines of the form
 *      
 *          VAR=value
 *      
 *      for environment variables that must be set to a specific
 *      value, or lines of the form
 *      
 *          VAR
 *      
 *      for environment variables that take their values from our
 *      environment.
 *      
 *      No environment variables that are not found in SaPRIVENV will
 *      be available in the privileged program.  This is to prevent
 *      hackers from spoofing us using environment variables.
 *
 *  Returns:
 *      The evironment that runpriv will use to execute a privileged
 *      command.
 */
static char** CreateRunprivEnvironment()
{
    FILE *fp;
    char line[2048];
    char *equal, *envStr, *newLine, **newEnviron;
    int nVars;

    fp = fopen(SaPRIVENV, "r");
    if (fp == NULL) {
	perror(SaPRIVENV);
	exit(SaPRIVNOENV);
    }

    /*
     * Build environment on the heap as we go.
     */
    newEnviron = (char**)malloc(sizeof *newEnviron);

    newEnviron[0] = NULL;

    nVars = 0;
    while (fgets(line, sizeof line - 1, fp) != NULL) {
	newLine = strrchr(line, '\n');
	if (newLine) {
	    *newLine = '\0';
	}
	equal = strchr(line, '=');
	if (equal) {
	    /*
	     * It's a variable with a value
	     */
	    envStr = line;
	} else {
	    /*
	     * It's a variable without a value, which means we get the
	     * value out of the current environment.
	     */
	    envStr = getenv(line);
	    if (envStr) {
		/*
		 * Don't put NULL variables in the environment
		 */
		if (strlen(envStr)) {
		    envStr -= strlen(line) + 1;
		} else {
		    envStr = NULL;
		}
	    }
	}

	if (envStr) {
	    /*
	     * Add the new variable to the privileged program's
	     * environment.
	     */
	    nVars++;
	    newEnviron = (char**)
		realloc(newEnviron, (nVars + 1) * sizeof *newEnviron);
	    newEnviron[nVars - 1] = strdup(envStr);
	    newEnviron[nVars] = NULL;
	}
    }

    return newEnviron;
}

//
//  static const char* const* GetEnv()
//
//  Description:
//      Get the environment that should be passed to
//      SaParamExecArgsCreate().  We will pass the bigger of our
//      current environment or the environment that runpriv
//      will use to run a priv command.
//      
//      The only thing SaParamExecArgsCreate() uses the environment
//      for is to see how much space within ARG_MAX is left for
//      command line arguments, so we need to pass in an environment
//      that's big enough to leave enough space both in our exec of
//      runpriv and runpriv's exec of the privileged command.
//
//  Returns:
//	Environment that should be passed to SaParamExecArgsCreate().
//
static const char* const* GetEnv()
{
    extern char const* const* _environ;

    static const char* const* runprivEnviron = NULL;

    if (runprivEnviron == NULL) {
	runprivEnviron = CreateRunprivEnvironment();
    }

    return SaParamGetEnvSize(_environ) > SaParamGetEnvSize(runprivEnviron)
	? _environ : runprivEnviron;
}

//
// Job is used to keep track of a running privileged command.
//
class Job {
  public:
    Job(long long jobId, PrivListener& listener)
    : _jobId(jobId), _command(SaRUNPRIV),
      _listener(listener), _outputId(-1), _errorId(-1), _numMonitors(0)
    {
    }

    long long _jobId;		// Job id specified by client.
    Command _command;		// Command that's running.
    PrivListener& _listener;	// Access to PrivListener instance.
    int _outputId;		// monitor id of _command's stdout.
    int _errorId;		// monitor id of _command's stderr.
    int _numMonitors;		// number of monitors currently active.
};

//
// PrivListener constructor.
//
PrivListener::PrivListener(Connection& conn, const String& serviceName)
: _conn(conn), _serviceName(serviceName)
{
}

//
// PrivListener destructor.
//
PrivListener::~PrivListener()
{
    RemoveAndDestroyContentsOf(&_jobs);
}

//
//  void PrivListener::receivePacket(const Packet& packet)
//
//  Description:
//      Called by Connection when a packet arrives for us.
//
//  Parameters:
//      packet
//
void PrivListener::receivePacket(const Packet& packet)
{
    if (packet.getSelector() == RUN) {
	run(packet);
    } else if (packet.getSelector() == INPUT) {
	input(packet);
    } else if (packet.getSelector() == DESTROY) {
	destroy(packet);
    } else if (packet.getSelector() == CHECK) {
	check(packet);
    } else if (packet.getSelector() == ADD) {
	add(packet);
    } else if (packet.getSelector() == CHECK_PASSWORD) {
	checkPassword(packet);
    } else {
	Log::error(PRIVILEGE, i18n("Unknown request: %s."),
		   (const char*)packet.getSelector());
    }
}

//
//  void PrivListener::run(const Packet& packet)
//
//  Description:
//      Run a privileged command.
//
//  Parameters:
//      packet  Specifies privcmd to run and parameters.
//
void PrivListener::run(const Packet& packet)
{
    // jobId is used to mark a running privileged command.  This is a
    // number specified by the client.  The client uses jobId to
    // specify which running command a request is to operate on, and
    // we use it to specify which privileged command an event comes
    // from.
    long long jobId = packet.getAttr(JOB_ID).longValue();

    Job* job = new Job(jobId, *this);
    _jobs.add(job);

    Command& command(job->_command);

    Attribute passAttr = packet.getAttr(PASSWORD);
    String privilege(packet.getAttr(PRIVILEGE).stringValue());

    // Arguments can come in two forms.  If the ARGC attribute is
    // present, ARGC attributes of the form "arg%d" should be present;
    // these are the arguments.  Otherwise, the arguments should be an
    // AttrBundle in the ARGS attribute.
    //
    // The ARGC form is used by the client to run privileged commands
    // with explicit control over the command line arguments.  The
    // ARGS form is used by clients that want to pass command line
    // arguments as AttrBundles.
    SaParamExecArgs* execArgs = NULL;
    Attribute argcAttr(packet.getAttr(ARGC));
    if (argcAttr != Attribute::NUL) {
	command.addArg("runpriv");
	if (passAttr != Attribute::NUL) {
	    command.addArg("-auth");
	    command.addArg("unix");
	}

	// Note that in ARGC parameter parsing, we don't check to see
	// if the arguments are bigger than ARG_MAX.  We will just let
	// exec fail.  The reason for this is that we don't have
	// any way of passing arguments other than on the command
	// line.  This is in contrast to the SaParam handling for the
	// ARGS parameter passing, where we can pass overflow
	// arguments via stdin.
	command.addArg(privilege);
	int argc = argcAttr.longValue();
	for (int i = 0; i < argc; i++) {
	    command.addArg(packet.getAttr(String(ARG)
					  + StringFromLong(i)).stringValue());
	}
    } else {
	int argc = 0;
	const char* argv[4];
	argv[argc++] = "runpriv";
	if (passAttr != Attribute::NUL) {
	    argv[argc++] = "-auth";
	    argv[argc++] = "unix";
	}
	argv[argc++] = privilege;

	// Make sure we didn't overrun the buffer.
	assert(argc <= sizeof argv/sizeof *argv);
	AttrBundle args(packet.getAttr(ARGS).bundleValue());
	CollectionOf<Attribute> attrs(args.copyAttrList());

	SaParam* param = SaParamCreate();
	IteratorOver<Attribute> iter(&attrs);
	Attribute* attr;
	while ((attr = iter()) != NULL) {
	    String key(attr->getKey());
	    (args.getAttrVisible(key) ? SaParamSet : SaParamSetHidden)
		(param, key, attr->getValueString());
	}

	execArgs = SaParamExecArgsCreateV(param, argc, argv, GetEnv());
	SaParamDestroy(param);

	const char* const* argList = SaParamExecArgsGet(execArgs);
	command.setArgs(argList);

	RemoveAndDestroyContentsOf(&attrs);
    }

    if (command.run() == -1) {
	jobFinished(job, -1);
	return;
    }

    struct sigaction act, oact;
    act.sa_handler = SIG_IGN;
    act.sa_flags = 0;
    sigemptyset(&act.sa_mask);
    (void)sigaction(SIGPIPE, &act, &oact);

    if (passAttr != Attribute::NUL) {
	sendPassword(command, passAttr.stringValue());
    }

    if (execArgs != NULL) {
	SaParamExecArgsSend(execArgs, command.getInput());
	SaParamExecArgsDestroy(execArgs);
    }

    (void)sigaction(SIGPIPE, &oact, NULL);

    AppContext& app(AppContext::getAppContext());
    job->_outputId = app.registerMonitor(command.getOutput(),
					 handleCommandOutput, job);
    job->_numMonitors++;
    job->_errorId = app.registerMonitor(command.getError(),
					handleCommandOutput, job);
    job->_numMonitors++;
}

//
//  void PrivListener::input(const Packet& packet)
//
//  Description:
//      Client is giving us some input that we're supposed to pass
//      along to a running privileged command.
//
//  Parameters:
//      packet  Packet specifying which job and the input.
//
void PrivListener::input(const Packet& packet)
{
    Job* job = getJob(packet.getAttr(JOB_ID).longValue());

    // If the job exited already, we can't send it more input.
    if (job == NULL) {
	return;
    }

    int length = packet.getDataLength();
    if (length == 0) {
	job->_command.closeInput();
	return;
    }

    struct sigaction act, oact;
    act.sa_handler = SIG_IGN;
    act.sa_flags = 0;
    sigemptyset(&act.sa_mask);
    (void)sigaction(SIGPIPE, &act, &oact);

    SaWriteFully(job->_command.getInput(), packet.getData(), length);

    (void)sigaction(SIGPIPE, &oact, NULL);
}

//
//  void PrivListener::destroy(const Packet& packet)
//
//  Description:
//      Client wants us to kill a running privileged command.
//
//  Parameters:
//      packet  specifies which privileged command to kill.
//
void PrivListener::destroy(const Packet& packet)
{
    Job* job = getJob(packet.getAttr(JOB_ID).longValue());

    // If the job exited already, we can't kill it.
    if (job == NULL) {
	return;
    }

    job->_command.kill();
}

//
//  void PrivListener::check(const Packet& packet)
//
//  Description:
//      Client wants to see if it has a set of privileges.
//
//  Parameters:
//      packet  Specifies the set of privileges to check.
//
void PrivListener::check(const Packet& packet)
{
    Packet result(packet.getType(), RESULT);
    result.setAttr(packet.getAttr(COOKIE));

    int numPrivs = (int)packet.getAttr(NUM_PRIVS).longValue();
    for (int i = 0; i < numPrivs; i++) {
	String priv(packet.getAttr(String(PRIV)
				   + StringFromLong(i)).stringValue());

	String privPath(SaPRIVCMDDIR);
	privPath += "/";
	privPath += priv;
	if (access(privPath, F_OK) == -1) {
	    Log::trace(PRIVILEGE, "access(%s) failed", (const char*)privPath);
	    result.setAttr(Attribute(RESULT, false));
	    result.setAttr(Attribute(REASON, (long long)NO_SUCH_PRIV));
	    result.setAttr(Attribute(PRIV, priv));
	    _conn.sendPacket(result);
	    return;
	}	    
    }

    Command command(SaCHECKPRIV);
    command.addArg("checkpriv");
    bool status = runAdminCommand(command, packet) == 0;
    result.setAttr(Attribute(RESULT, status));
    if (!status) {
	result.setAttr(Attribute(REASON, (long long)NOT_PRIVILEGED));
    }
    _conn.sendPacket(result);
}

//
//  void PrivListener::add(const Packet& packet)
//
//  Description:
//      Client wants to add a set of privileges.
//
//  Parameters:
//      packet  Specifies the set of privileges to add.
//
void PrivListener::add(const Packet& packet)
{
    Command command(SaADDPRIV);
    command.addArg("addpriv");
    Packet result(packet.getType(), RESULT);
    result.setAttr(packet.getAttr(COOKIE));
    result.setAttr(Attribute(
	RESULT, (bool)runAdminCommand(command, packet) == 0));
    _conn.sendPacket(result);
}
    
//
//  void PrivListener::checkPassword(const Packet& packet)
//
//  Description:
//      Check to see if the password passed in is the root password.
//
//  Parameters:
//      packet  Packet containing the password to check.
//
void PrivListener::checkPassword(const Packet& packet)
{
    bool result = false;
    if (geteuid() == 0) {
	// If we're running as root, password doesn't matter.
	result = true;
    } else {
	Command command(SaCHECKPRIV);
	command.addArg("checkpriv");
	command.addArg("-auth");
	command.addArg("unix");
	command.addArg("-authOnly");
	if (command.run() == 0) {
	    Attribute passAttr = packet.getAttr(PASSWORD);
	    if (passAttr != Attribute::NUL) {
		sendPassword(command, passAttr.stringValue());
	    }
	    command.wait();
	    if (command.getExitStatus() == 0) {
		result = true;
	    }
	}
    }
    Packet resultPacket(packet.getType(), packet.getSelector());
    resultPacket.setAttr(packet.getAttr(COOKIE));
    resultPacket.setAttr(Attribute(RESULT, result));
    _conn.sendPacket(resultPacket);
}

//
//  Job* PrivListener::getJob(long long jobId)
//
//  Description:
//      Get the Job object corresponding to "jobId".  The client
//      sends us "jobId"s which we map to Job objects.
//
//  Parameters:
//      jobId  identifier of Job to find.
//
//  Returns:
//	Job corresponding to "jobId".
//
Job* PrivListener::getJob(long long jobId)
{
    IteratorOver<Job> iter(&_jobs);
    Job* job;
    while ((job = iter()) != NULL) {
	if (job->_jobId == jobId) {
	    break;
	}
    }
    return job;
}

//
//  void PrivListener::handleCommandOutput(void* clientData, int id, int fd)
//
//  Description:
//      Called when a privileged command writes to its stdout or
//      stderr.  We send the data along to the client.
//
//  Parameters:
//      clientData  Job pointer (this is a static method).
//      id          monitor id.
//      fd          file descriptor.
//
void PrivListener::handleCommandOutput(void* clientData, int id, int fd)
{
    Job* job = (Job*)clientData;

    char buf[100];
    int n = SaRead(fd, buf, sizeof buf);
    if (n <= 0) {
	job->_command.wait();
	AppContext& app(AppContext::getAppContext());
	app.unregisterMonitor(id);
	job->_numMonitors--;
	if (job->_numMonitors == 0) {
	    jobFinished(job, job->_command.getExitStatus());
	}
	return;
    }

    Packet packet(job->_listener._serviceName,
		  id == job->_outputId ? OUTPUT : ERROR);
    packet.setAttr(Attribute(JOB_ID, (long long)job->_jobId));
    packet.setData(buf, n);
    job->_listener._conn.sendPacket(packet);
}

//
//  int PrivListener::runAdminCommand(Command& command, const Packet& packet)
//
//  Description:
//      Shared code between check() and add().  Run a command, and
//      send the exit status back to the client.
//
//  Parameters:
//      command  Command to run.
//      packet   Specifies parameters for command.
//
//  Returns:
//	0 if successful, -1 if error.
//
int PrivListener::runAdminCommand(Command& command, const Packet& packet)
{
    Attribute passAttr = packet.getAttr(PASSWORD);
    if (passAttr != Attribute::NUL) {
	command.addArg("-auth");
	command.addArg("unix");
    }
    
    Attribute userAttr = packet.getAttr(USER);
    if (userAttr != Attribute::NUL) {
	command.addArg(userAttr.stringValue());
    }

    int numPrivs = (int)packet.getAttr(NUM_PRIVS).longValue();
    for (int i = 0; i < numPrivs; i++) {
	command.addArg(packet.getAttr(String(PRIV)
				      + StringFromLong(i)).stringValue());
    }

    if (command.run() == -1) {
	return -1;
    }

    if (passAttr != Attribute::NUL) {
	sendPassword(command, passAttr.stringValue());
    }
    
    command.wait();
    
    return command.getExitStatus() == 0 ? 0 : -1;
}

//
//  void PrivListener::sendPassword(Command& command, const String& password)
//
//  Description:
//      Send a password to "command".
//
//  Parameters:
//      command   Command to send password to.
//      password  Password to send.
//
void PrivListener::sendPassword(Command& command, const String& password)
{
    int input = command.getInput();
    SaWriteFully(input, password, password.getLength());
    SaWriteFully(input, "\n", 1);
}

//
//  void PrivListener::jobFinished(Job* job, int exitCode)
//
//  Description:
//      Notify client that a job has finished and delete the job.
//
//  Parameters:
//      job       job that's finished.
//      exitCode  exit code to send to client.
//
void PrivListener::jobFinished(Job* job, int exitCode)
{
    assert(job != NULL);
    // It would be very bad to delete a job for which we still have
    // monitors outstanding.
    assert(job->_numMonitors == 0);

    Packet packet(job->_listener._serviceName, FINISHED);
    packet.setAttr(Attribute(EXIT_VALUE, (long long)exitCode));
    packet.setAttr(Attribute(JOB_ID, job->_jobId));
    job->_listener._conn.sendPacket(packet);
    job->_listener._jobs.removePointer(job);
    delete job;
}

//
// Called by sysadmd to dynamically load the privilege service.
//
extern "C" void* CreatePacketListener(Connection& conn,
				      const String& serviceName)
{
    return new PrivListener(conn, serviceName);
}
