#!/usr/bin/perl
#
# This script is a hack that works around the fact that the java 1.1
# javadoc command does not handle inner classes.  innerjavadoc extracts
# public and protected inner classes and puts them in their own files in
# such a way that the javadoc-generated web site will have documentation
# for the inner classes.  This is not perfect; each top-level class that
# defines inner classes gets its own package documentation, when such
# packages don't really exist.  This package documentation is important
# though because it's the only place that lists the inner classes for
# a particular top-level class.
#
# In addition to creating a bunch of .java files, innerjavadoc creates
# two files in the current working directory:
#
#   DIRLIST is the list of directories created by innerjavadoc
#   PACKLIST is the list of pseudo-packages created by innerjavadoc
#
# DIRLIST can be used to clean up all the junk that innerjavadoc creates,
# and PACKLIST should be passed on to javadoc so that javadoc will
# see all the .java files that innerjavadoc generated.
#

require "getopts.pl";
&Getopts('r:');
if (!$opt_r || $#ARGV == -1) {
    die("usage: innerjavadoc -r root package [ package ... ]\n");
}

#
# Report cleanup, then report an error and quit.
#
sub cleanupAndDie {
    local($msg) = @_;
    close(DIRLIST);
    die $msg;
}

#
# checkDir $dir creates $dir if it does not already exist.  Created
# directories are added to the DIRLIST for subsequent removal (by
# some other program).
#
sub checkDir {
    local($dir) = @_;

    if (! -w $dir) {
	mkdir($dir, 0755) || die "$dir: $!\n";
	print DIRLIST "$dir\n";
    }
}

#
# Add a package to the external list of packages.
#
sub addPackage {
    local($package) = @_;
    print PACKLIST "$package\n";
}

#
# Count the number of occurrences of a string in a line.
#
sub countStrings {
    local($line, $str) = @_;
    local($ii) = 0;
    local($numOpens) = 0;

    while (($ii = index($line, $str, $ii)) != -1) {
	$ii++;
	$numOpens++;
    }
    return $numOpens;
}

#
# Count the number of open braces in a line.
#
sub countOpens {
    local($line) = @_;
    return &countStrings($line, "{");
}

#
# Count the number of close braces in a line.
#
sub countCloses {
    local($line) = @_;
    return &countStrings($line, "}");
}

#
# expandFile <package> <file> extracts the inner classes from
# a .java file.
#
sub expandFile {
    local($package, $file) = @_;
    local($inner) = 0;
    local($className);
    local($comment) = "";
    local($imports) = "";

    $file =~ /.*\/([^\/]*).java/;
    local($className) = $1;
    $file =~ /^(.*).java$/;
    local($base) = $1;
    local($packageAdded) = 0;

    open(FILE, $file) || &cleanupAndDie("$file: $!\n");
    while ($line = <FILE>) {
	if ($line =~ /\/\*\*/) {
	    $inComment = 1;
	    do {
		$comment .= $line;
		if ($line =~ /\*\//) {
		    $inComment = 0;
		}
	    } while ($inComment && ($line = <FILE>));
	}
	if ($line =~ /^import.*/) {
	    $imports .= $line;
	}
	if ($line =~ /^\s*((final|static)\s+)?(protected|private|public|)(\s+(final|static))?\s+(class|interface)\s([a-zA-Z0-9_]+)/) {
	    $access = $3;
	    $decl = $6;
	    $class = $7;
	    if ($class eq $className
		&& $access ne "protected" && $access ne "public") {
		print "Skipping private or package class: $class\n";
		break;
	    }
	    if ($class ne $className
		&& ($access eq "protected" || $access eq "public")) {
		print "Creating $base/$class.java...\n";
		&checkDir($base);
		open(DEST, ">$base/$class.java") ||
		    &cleanupAndExit("$base/$class.java: $!\n");
		$started = 0;
		$inBraces = 0;
		print DEST "package $package.$className;\n";
		print DEST $imports;
		print DEST "import $package.*;\n";
		print DEST $comment;
		$comment = "";
		if (!$packageAdded) {
		    &addPackage("$package.$className");
		    $packageAdded = 1;
		}
		$line =~ s/static//g;
		do {
		    print DEST $line;
		    $numOpens = &countOpens($line);
		    $numCloses = &countCloses($line);
		    if ($numOpens > 0) {
			$started = 1;
		    }
		    $inBraces += $numOpens;
		    $inBraces -= $numCloses;
		} while (($inBraces || !$started) && ($line = <FILE>));
	    }
	    if (!$line) {
		break;
	    }
	}
	if ($line !~ /^\s*$/ && $line !~ /\*\//) {
	    # Don't remember comments across lines that look like they
	    # contain code.
	    $comment = "";
	}
    }
    close FILE;
}

#
# expandPackage <package> extracts the inner classes from all the
# .java files in a package.
#
sub expandPackage {
    local($package) = @_;
    local($packageDir) = $package;
    $packageDir =~ s/\./\//g;

    opendir(DIR, "$ROOT/$packageDir")
	|| &cleanupAndDie("$ROOT/$packageDir: $!\n");
    while (($ent = readdir DIR)) {
	if ($ent =~ /.*\.java$/) {
	    &expandFile($package, "$ROOT/$packageDir/$ent");
	}
    }
    closedir DIR;
}

$ROOT = $opt_r;

open(DIRLIST, ">>DIRLIST") || die "DIRLIST: $!\n";
open(PACKLIST, ">PACKLIST") || die "PACKLIST: $!\n";

for ($ii = 0; $ii <= $#ARGV; $ii++) {
    &expandPackage($ARGV[$ii]);
}

close DIRLIST;
close PACKLIST;
